/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RecordVariable.java,v 1.2 2005/02/25 22:17:10 hleung Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.trace.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.NotSerializableException;
import java.io.ObjectOutputStream;
import org.eclipse.hyades.logging.core.XmlUtility;


/**
 * This is the record for a parameter or a return value.
 * 
 * @author Richard Duggan, Qiyan Li
 */
public class RecordVariable {

    /**
     * the constant event type representing a parameter to and from a method
     */
    public static final String PARAMETER = "parameter";

    /**
     * the constant event type representing a return value from a method
     */
    public static final String RETURN_VALUE = "returnValue";

    /**
     * the constant event type representing an element inside an array
     */
    public static final String ARRAY_ELEMENT = "arrayElement";

    /**
     * the event type of this record
     */
    private final String _event;

    /**
     * the index position of an array element with the index of the first variable being 0 if the event
     * type is ARRAY_ELEMENT, or the parameter ID if the event type is PARAMETER.  This field is unused
     * by event RETURN_VALUE.
     */
    private final int _id;

    /**
     * the Java class of this variable
     */
    private final Class _clazz;

    /**
     * the reference to the value of this variable
     */
    private final Object _value;


    /**
     * Creates a record for the given <code>event</code>, <code>index</code>, <code>clazz</code>,
     * and <code>value</code>.  The <code>index</code> will only be valid for event type {@link #PARAMETER}.
     * 
     * @param event the event type of {@link #PARAMETER} or {@link #RETURN_VALUE}
     * @param index the position of this parameter in the parameter list
     * @param clazz the Java class of the parameter
     * @param value the referece to the actual object 
     */
    public RecordVariable(String event, int index, Class clazz, Object value) {

        this._event = event;
        this._id = index;
        this._clazz = clazz;
        this._value = value;
    }


    /**
     * Converts this record into a string.
     * 
     * @return  this record in string format, or <code>null</code> if the value of this variable
     *          is <code>null</code> 
     */
    public String toString() {

        /* If the value is null, then nothing is emitted. */
        if (_value == null) {
            return null;
        }

        /* Depending on whether the object is an array, get all array elements, or try to stringify
           the value of the variable. */
        String valueString = null;
        if (_clazz.isArray()) {
            valueString = new RecordArray(ARRAY_ELEMENT, _clazz.getComponentType(), (Object[]) _value).toString();
        } else {
            String stringified = Utilities.stringifyPrimitivesStrings(_clazz, _value);
            if (stringified != null) {
                valueString = XmlUtility.normalize(stringified);
            }
        }

        /* If the value cannot be represented in a straightforward way, use the serialized form
           to represent the value. */
        if (valueString == null) {

            try {
                /* Attempt to externalize the value. */
                ByteArrayOutputStream byteObjectStream = new ByteArrayOutputStream();
                ObjectOutputStream extrnObjectStream = new ObjectOutputStream(byteObjectStream);
                extrnObjectStream.writeObject(_value);
                extrnObjectStream.close();

                /* Convert the bytes into a hexdecimal string. */
                byte[] byteObject = byteObjectStream.toByteArray();
                StringBuffer valueBuffer = new StringBuffer(byteObject.length << 1);
                for (int i = 0; i < byteObject.length; i++) {
                    String hexByte = Integer.toHexString(byteObject[i] & 0xff);
                    valueBuffer.append(hexByte.length() < 2 ? "0" + hexByte : hexByte);
                }
                valueString = valueBuffer.toString();

            } catch (NotSerializableException e) {

                /* If the value turns out unserializable, simple return its string representation. */
                valueString = XmlUtility.normalize(_value.toString());

            } catch (IOException e) {

                e.printStackTrace();
                return null;
            }
        }
        return "<" + _event + (_event == ARRAY_ELEMENT ? " index=\"" + _id + "\"" : "") +
            (_event == PARAMETER ? " parameterIdRef=\"" + _id + "\"" : "") +
            (!_clazz.isArray() ? " value=\"" + valueString + "\"/>" : ">\n" + valueString + "</" + _event + ">"); 
    }
}
