/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Utilities.java,v 1.2 2005/02/25 22:17:10 hleung Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.execution.trace.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;


/**
 * This is a utility class to perform various common tasks.
 *
 * @author Richard Duggan, Qiyan Li
 */
public class Utilities {

    /**
     * Returns the current timestamp in seconds and milliseconds.
     * 
     * @return  the current timestamp in seconds and milliseconds.
     */
    public static String getCurrentTimeStamp() {
        String timeStamp = Long.toString(System.currentTimeMillis());
        int splitPoint = timeStamp.length() - 3;
        return timeStamp.substring(0, splitPoint) + "." + timeStamp.substring(splitPoint);
    }


    /**
     * Returns the JNI notation of a class including primitive types, arrays, and classes.
     *
     * @param clazz the class of a primitive, array, or class 
     * @return  the JNI notation for the class
     */
    public static String getJniNotation(Class clazz) {
    
        if (clazz == byte.class) {
            return "B";
        } else if (clazz == char.class) {
            return "C";
        } else if (clazz == double.class) {
            return "D";
        } else if (clazz == float.class) {
            return "F";
        } else if (clazz == int.class) {
            return "I";
        } else if (clazz == long.class) {
            return "J";
        } else if (clazz == short.class) {
            return "S";
        } else if (clazz == boolean.class) {
            return "Z";
        } else if (clazz == void.class) {
            return "V";
        } else if (clazz.isArray()) { 
            return clazz.getName();
        } else {
            return "L" + clazz.getName() + ";";
        }
    }


    /**
     * Returns an array of bytes containing the serialized object.
     *
     * @param object    the object to be serialized
     * @return  an array of bytes containing the serialized object
     */
    public static byte[] serializeObject(Object object) {

        ByteArrayOutputStream outBytes = new ByteArrayOutputStream();
        try {
            ObjectOutputStream outObject = new ObjectOutputStream(outBytes);
            outObject.writeObject(object);
            outObject.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return outBytes.toByteArray();
    }


    /**
     * Returns an object restored from the byte buffer given.
     *
     * @param buffer    the byte buffer containing the serialized object
     * @param offset    the first byte containing the serialized object
     * @param length    the length of the buffer from the first byte to the end of the buffer
     * @return  an object restored from the byte buffer given
     */
    public static Object deserializeObject(byte[] buffer, int offset, int length) {

        ByteArrayInputStream inBytes = new ByteArrayInputStream(buffer, offset, length);
        Object object = null;
        try {
            ObjectInputStream inObject = new ObjectInputStream(inBytes);
            object = inObject.readObject();
            inObject.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
        return object;
    }


    /**
     * Returns a string representation of all primitive types, their wrappers, and Strings.
     *
     * @param clazz     the class of the object to be stringified
     * @param object    the object to be stringified
     * @return  a string correponding to the object if possible, or a <code>null</code>
     */
    public static String stringifyPrimitivesStrings(Class clazz, Object object) {

        return clazz == byte.class || clazz == Byte.class || clazz == short.class || clazz == Short.class ||
            clazz == int.class || clazz == Integer.class || clazz == long.class || clazz == Long.class ||
            clazz == float.class || clazz == Float.class || clazz == double.class || clazz == Double.class ||
            clazz == char.class || clazz == Character.class || clazz == boolean.class || clazz == Boolean.class ||
            clazz == String.class ?
            object.toString() : null;
    }


    /**
     * Returns the JNI notation of a method signature, including the parameter and return types.
     * In the case of a constructor, there is no return type (null).
     * 
     * @param parmClasses   the parameter list
     * @param returnType    the return type in JNI notaion
     * @return  the JNI notation for the specified parameter and return types
     */
    public static String getInvokedMethodSignature(Class[] parmClasses, String returnType) {
    
        /* No signature can be generated without parameter types. */ 
        if (parmClasses == null) {
            return null;
        }
    
        /* Generate the method signature. */
        StringBuffer sigBuffer = new StringBuffer();
        for (int i = 0; i < parmClasses.length; i++) {
            sigBuffer.append(Utilities.getJniNotation(parmClasses[i]));
        }
        return "(" + sigBuffer.toString() + ")" + (returnType == null ? "" : returnType);
    }
}
