/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DataChannelImpl.java,v 1.2 2005/02/25 22:17:43 hleung Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.collection.framework.channel;

/**
 * DataChannel provides an implementation for a data channel between
 * two processes.  The channel is filled with data by one process
 * and can be flushed from the channel to a file ,a socket, or an
 * OutputStream by the other process.
 * Currently the flusher process creates the data channel and the
 * filler process attaches to it via its name.
 * This implementation uses native methods to implement the data
 * channel as a shared memory buffer.
 * <p>The native implementation of the data channel methods are located in the
 * dynamic linked library hcclsm.dll (shared object libhcclsm.so on unix)
 * This library is dynamically loaded by the JVM when the DataChannel class
 * is loaded and could result in a java.lang.LinkageError being thrown. Users
 * of this class should be prepared to catch this exception when declaring the
 * first reference to this type.</p>
 *
 * @version 0.0.1
 * Creation date: (3/31/03 1:55:57 PM)
 * @author: David N. Smith
 * @see java.lang.LinkageException
 */



import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.net.InetAddress;

import org.eclipse.hyades.execution.core.DataChannelConnectionException;
import org.eclipse.hyades.execution.core.DataChannelCreationException;
import org.eclipse.hyades.execution.core.DataChannelNotInitializedException;
import org.eclipse.hyades.execution.core.IDataChannel;
import org.eclipse.hyades.execution.core.InvalidDataChannelAccessException;

public class DataChannelImpl implements IDataChannel  {

    /* Whether the native impls are available. */
    private static boolean _nativesAvailable=false;;

    /* Whether the data channel is being flushed currently */
    protected boolean _isFlushing = false;

	/* Whether the data channel is being written to */
    protected boolean _isWriting = false;

    /* Whether the data channel has been created */
    private boolean _isDCBufferCreated = false;

    /* Static fields */
    private static String DC_BUFFER_NAME_ROOT = "hcbuffer";
    private static int _dc_buffer_num = 0;
    private static int DC_DEFAULT_BUFFER_SIZE = 1048576;
	private final static int INPUT_BUFFER_SIZE = 8096;
    protected final static int FLUSH_TYPE_PIPED_OUTPUT_STREAM  = 6;
    protected final static int FLUSH_TYPE_OUTPUT_STREAM  = 5;
    protected final static int FLUSH_TYPE_BINARY_FILE    = 4;
    protected final static int FLUSH_TYPE_ASCII_FILE     = 3;
    protected final static int FLUSH_TYPE_RAC_FILE       = 2;
    protected final static int FLUSH_TYPE_SOCKET         = 1;
    protected static int DC_TYPE_FLUSHABLE  = 1;
    protected static int DC_TYPE_FILLABLE = 2;

    /* Instance fields */
    private String _dc_buffer_name = null;
    private int  _dc_buffer_handle = -1;

    private int _dc_type = 0;			/* flushable or fillable */

    private int _dc_flushType = 0;	/* flush destination type */

    protected OutputStream _dc_outputStream;   /* OutputStream returned to the filler to write to the channel */
    private PipedOutputStream _dc_POutputStream;  /* used internally for filling the flushing InputStream with data from the channel */
    private InputStream _dc_inputStream;     /* InputStream returned to the flusher to read data from the channel */
    private PipedInputStream _dc_PInputStream;    /* used internally for reading data from the filling OutputStream to write to the channel */

    private int _dc_buffer_size = 0;

    /* Destination for flushing data channel */
    private String _file_name;
    private InetAddress _client_address;
    private int         _client_port;


    /* Increment the buffer number and return the old value */

    private synchronized int incrementDCBufferNum() {
       int oldnum = _dc_buffer_num;

       ++_dc_buffer_num;

       return oldnum;
    }

    /* Try and load the native shared memory library.  If this fails we will ignore the error */
    static {
        try {

            System.loadLibrary("hcclsm");
            _nativesAvailable=true;

        }
        catch (Throwable e) {
            System.out.println("Can't find hcclsm library.\n");
            _nativesAvailable=false;
        }
    }

    /**
     * Create a DataChannel object
     *
     */
    public DataChannelImpl() {
        super();
    }

	/**
	 * Creates the data channel.
	 * @throws DataChannelCreationException
	 */

    public void create() throws DataChannelCreationException {
    	if (getName() == null) {
    		setName(DC_BUFFER_NAME_ROOT + String.valueOf(incrementDCBufferNum()));
    	}
    	
    	if (getSize() == 0) {
    		setSize(DC_DEFAULT_BUFFER_SIZE);
    	}

		/* Call the native method that returns a handle to the new data channel buffer
		 * - negative handle indicates an error occurred
		 */    	
		int rc = hc_createDCBuffer(getName(), getSize());

		if (rc >= 0) {
        	/* handle can only be positive */
			setHandle(rc);
			_isDCBufferCreated = true;

		}
		else if (rc == -518) {
			throw new DataChannelCreationException("Data Channel with name " + getName() + " already exists");	
		}			
		else if (rc == -99) {
			throw new DataChannelCreationException("Could not allocate memory");	
		}
		else if (rc == -504) {
			throw new DataChannelCreationException("Data Channel size " + getSize() + " is invalid");	
		}
		else if (rc == -505) {
			throw new DataChannelCreationException("Data Channel name " + getName() + " is invalid");	
		}
		else if (rc < -500) {
			throw new DataChannelCreationException("Implementation failure code: " + rc);	
		}
        else {
        	throw new DataChannelCreationException();
        }
    }


	/**
	 * Destroys the data channel.
	 */
    public void destroy() {
       if (isDCBufferCreated()) {
       	/* call the native method to destroy the data channel buffer */
       	this.hc_destroyDCBuffer(getHandle());
       }
    }

	/**
	 * Connects to the data channel.
	 * @throws DataChannelConnectionException
	 */


    public void connect() throws DataChannelConnectionException {
		int rc;

   		if (getName() == null) {
   			throw new DataChannelConnectionException("No name has been set");
   		}
   		
		/* Call the native method to try to attach to the data channel buffer
		 * - a handle to the existing data channel buffer is returned
		 * - negative handle indicates an error occurred
		 */
		rc = hc_attachToDCBuffer(getName());

		/* If the data channel does not exist then create it */

		if (rc < 0) {
			if (rc == -3) {
				throw new DataChannelConnectionException("Data Channel with name " + getName() + " does not exist");	
			}			
			if (rc == -99) {
				throw new DataChannelConnectionException("Could not allocate memory");	
			}
			if (rc == -505) {
				throw new DataChannelConnectionException("Data Channel name " + getName() + " is invalid");	
			}
			else if (rc < -500) {
				throw new DataChannelConnectionException("Implementation failure code: " + rc);	
			}
			else {
				throw new DataChannelConnectionException();
			}
		}

		setHandle(rc);

		_isDCBufferCreated = true;    	
    }


	/**
	 * Disconnects from the data channel.
	 */

    public void disconnect() {
    	/* Stop flushing the data channel because we are not writing to it any more */
		stopFlushing();
		
		/* Call the native method to detach from the data channel buffer */
		hc_detachFromDCBuffer(getHandle());
    }


	/**
	 * Writes bytestream.length bytes from the specified byte array to this data channel.
	 * @param bytes byte []
	 * @throws IOException
	 * @throws InvalidDataChannelAccessException
	 */
	public void write(byte [] bytes)
					throws IOException, InvalidDataChannelAccessException {
						
		if (!isFillable()) {
			throw new InvalidDataChannelAccessException();
		}
		
		if (!isDCBufferCreated()) {
			throw new InvalidDataChannelAccessException();
		}

		/* Call the native method to write the array of bytes to the data channel buffer
		 * - negative return indicates an error occurred
		 */	
		
		int rc = hc_writeToDCBuffer(getHandle(), bytes, bytes.length);
		
		if (rc < 0) {
			throw new IOException();
		}
	}



	/**
 	 * Sets the output stream that acts as the destination for the data on the data channel.
 	 * This is set by the flusher of the data channel.
 	 * @param outstr java.io.OutputStream
	 * @throws InvalidDataChannelAccessException
	 * @throws DataChannelNotInitializedException
 	 */
	public void setOutputStream(OutputStream outstr)
					throws InvalidDataChannelAccessException, DataChannelNotInitializedException {
		
		if (!isFlushable()) {
			throw new InvalidDataChannelAccessException();
		}			
		
		if (isDCBufferCreated()) {

			/* Start flushing to the specified output stream
			 * - this call will block until flushing is stopped by the filler
			 */
			 		
			startFlushingToStream(outstr);
		}
		else {
			throw new DataChannelNotInitializedException();
		}
			
	}

	/* Returns the internal PipedOutputStream */
	
	protected PipedOutputStream getPOutputStream() {
		return _dc_POutputStream;
	}

	/**
	 * Returns an output stream that can be used to write to the data channel.
	 * This is called by the filler of the data channel.
	 * @return java.io.OutputStream
	 * @throws InvalidDataChannelAccessException
	 * @throws DataChannelNotInitializedException
	 */
	public OutputStream getOutputStream()
				throws InvalidDataChannelAccessException, DataChannelNotInitializedException {

		if (!isFillable()) {
			throw new InvalidDataChannelAccessException();
		}			
		
		if (isDCBufferCreated()) {

			/* Create the PipedInputStream used internally to read the data that the caller writes to the other end of the pipe */
			PipedInputStream pis = new PipedInputStream();

			setPInputStream(pis);
			
			try {				
				/* Create the PipedOutputStream the caller will use to write data to the pipe */
				_dc_outputStream = new PipedOutputStream(pis);			

				/* Create the writer thread that reads data from the pipe and writes it to the shared memory */			
				WriterThread writer=new WriterThread();

				writer.start();

				/* Wait for the writer thread to get set up */				
				while (!_isWriting) {
					Thread.sleep(500);
				}

			}
			catch (InterruptedException ie) {
				System.out.println("Thread was interrupted before writer thread stopped");				
			}
			catch (IOException ioe) {
				System.out.println("IOException trying to create PipedOutputStream");				
			}							
		}
		else {
			throw new DataChannelNotInitializedException();
		}		

		/* Return the Piped Output Stream to the caller */		
		return _dc_outputStream;
	}

	/**
	 * Sets the input stream that acts as the input to the data channel.
	 * This is set by the writer of the data channel.
	 * This method blocks until all the data is read from the InputStream.
	 * @param instr java.io.InputStream
	 * @throws InvalidDataChannelAccessException
	 * @throws DataChannelNotInitializedException
	 */
	public void setInputStream(InputStream instr)
					throws InvalidDataChannelAccessException, DataChannelNotInitializedException {

		if (!isFillable()) {
			throw new InvalidDataChannelAccessException();
		}			
		
		if (isDCBufferCreated()) {

			_dc_inputStream = instr;

			/* Create writer thread to write data from the InputStream to the data channel */			
			WriterThread writer=new WriterThread();

			try {
				writer.start();

				/* Wait until the writer thread is finished writing all the data in the InputStream */				
				
				writer.join();			
			}
			catch (InterruptedException ie) {
				System.out.println("Thread was interrupted before writer thread stopped");				
			}
							
		}
		else {
			throw new DataChannelNotInitializedException();
		}		
		
	}
	
	/* Returns the internal InputStream */
	
	protected InputStream getPInputStream() {
		if (_dc_PInputStream == null)
			return _dc_inputStream;
		else
			return _dc_PInputStream;
	}

	/**
	 * Returns an input stream that can be used to read from the data channel.
	 * This is called by the flusher of the data channel.
	 * @return java.io.InputStream
	 * @throws InvalidDataChannelAccessException
	 * @throws DataChannelNotInitializedException
	 */

	public InputStream getInputStream()
					throws InvalidDataChannelAccessException, DataChannelNotInitializedException {
		
		if (!isFlushable()) {
			throw new InvalidDataChannelAccessException();
		}			
		
		if (isDCBufferCreated()) {
			/* Create a PipedOutputStream to flush the data from the channel to */
			PipedOutputStream pos = new PipedOutputStream();

			setPOutputStream(pos);
			
			try {				
				/* Create the PipedInputStream that the caller will read from to get data from the channel */
				_dc_inputStream = new PipedInputStream(pos);
				
				/* Start a flushing the data to the PipedOutputStream just created */
				startFlushingToStream(pos);
			}
			catch (IOException ioe) {
				System.out.println("IOException trying to create PipedOutputStream");				
			}
		}
		else {
			throw new DataChannelNotInitializedException();
		}	
		
		/* Return the PipedInputStream to the call to read from */
		return _dc_inputStream;
	}
    /**
     * Start flushing the data channel to a file
     *
     * @param flushtofilename java.lang.String - the name of the file to flush the data in the channel to
     */
    public void startFlushingToFile(String flushtofilename) {

        setFlushType(FLUSH_TYPE_RAC_FILE);
        setFileName(flushtofilename);

        if (isDCBufferCreated() && !isFlushing() && isFlushable()) {
	        /* Create and start flusher thread */
			FlusherThread flusher=new FlusherThread();
			try {
				flusher.start();
				flusher.join();
			}
			catch (InterruptedException ie) {
				System.out.println("Thread was interrupted before flusher thread stopped");
			}			
        }
    }

    /**
     * Start flushing the data channel to a socket
     *
     * @param flushtosocketaddress java.net.InetAddress - the address of the client to flush the data in the channel to
     * @param flushtosocketport int - the port number of the client to flush the data in the channel to
     */
    public void startFlushingToSocket(InetAddress flushtosocketaddress, int flushtosocketport) {

        setFlushType(FLUSH_TYPE_SOCKET);
        setInetAddress(flushtosocketaddress);
        setPort(flushtosocketport);

        if (isDCBufferCreated() && !isFlushing() && isFlushable()) {

	        /* Create and start flusher thread */
			FlusherThread flusher=new FlusherThread();
			try {
				flusher.start();
				flusher.join();
			}
			catch (InterruptedException ie) {
				System.out.println("Thread was interrupted before flusher thread stopped");
			}			
        }
    }

    /**
     * Start flushing the data channel to a piped output stream
     *
     * @param outStr java.io.PipedOutputStream - the PipedOutputStream to flush the data in the channel to
     */
	 public void startFlushingToStream(PipedOutputStream outStr) {
        if (isDCBufferCreated() && !isFlushing() && isFlushable()) {
			setFlushType(FLUSH_TYPE_PIPED_OUTPUT_STREAM);

			setPOutputStream(outStr);        	
			
	        /* Create and start flusher thread */
			FlusherThread flusher=new FlusherThread();
			try {
				flusher.start();
				
				/* Wait for the flusher thread to get set up */
				while (!isFlushing()) {
					Thread.sleep(500);
				}
				
			}
			catch (InterruptedException ie) {
				System.out.println("Thread was interrupted before flusher thread stopped");
			}
        }

    }

    /**
     * Start flushing the data channel to a stream
     *
     * @param outStr java.io.OutputStream - the OutputStream to flush the data in the channel to
     */
	 public void startFlushingToStream(OutputStream outStr) {
        if (isDCBufferCreated() && !isFlushing() && isFlushable()) {
			setFlushType(FLUSH_TYPE_OUTPUT_STREAM);
			
			_dc_outputStream = outStr;
       	
	        /* Create and start flusher thread */
			FlusherThread flusher=new FlusherThread();
			try {
				flusher.start();
				
				/* Wait for the flushing thread to end */
				
				flusher.join();
			}
			catch (InterruptedException ie) {
				System.out.println("Thread was interrupted before flusher thread stopped");
			}
        }

	}
	
    /**
     * Stop flushing the data channel
     */	
	public void stopFlushing() {
        if (isDCBufferCreated()) {
            int rc;

            /* Call the native method to force flushing to stop */
			rc = hc_stopFlushingDCBuffer(getHandle());
			
   			if (rc != 0) {
            	/* error processing */
            }
         }
    }

	/**
 	* Sets this data channel as flushable.
 	* @throws InvalidDataChannelAccessException
 	*/

	public void setFlushable() throws InvalidDataChannelAccessException {
		if (isFillable()) {
			throw new InvalidDataChannelAccessException();
		}
		
		_dc_type = DC_TYPE_FLUSHABLE;
		
	}

	/**
	* Tests whether the application can flush this data channel.
	* @return boolean
	*/

	public boolean isFlushable() {
		if (_dc_type == DC_TYPE_FLUSHABLE) {
			return true;
		}
		
		return false;
	}

	/**
 	* Sets this data channel as fillable.
 	* @throws InvalidDataChannelAccessException
 	*/

	public void setFillable() throws InvalidDataChannelAccessException {
		if (isFlushable()) {
			throw new InvalidDataChannelAccessException();
		}
		
		_dc_type = DC_TYPE_FILLABLE;
		
	}

	/**
	* Tests whether the application can fill this data channel.
	* @return boolean
	*/

	public boolean isFillable() {
		if (_dc_type == DC_TYPE_FILLABLE) {
			return true;
		}
		
		return false;
	}
	
	
    /**
     * Native delegate of destroy()
     * @param bufferhandle int  - handle of data channel buffer
     */
    protected native void hc_destroyDCBuffer(int bufferhandle);

    /**
     * Native delegate of disconnect() or detach from data channel buffer
     * @param bufferhandle int  - handle of data channel buffer
     */
    protected native void hc_detachFromDCBuffer(int bufferhandle);

    /**
     * Native method to validate the data channel buffer size and returns
     * a valid sized based on the input size.
     * @param size int - size to validate
     * @return int - a valid size
     */
    private native int hc_validateDCBufferSize(int size);

    /**
     * Native method to create the data channel buffer
     * @param buffername  java.lang.String  - name of data channel buffer
     * @param size int  - size of data channel buffer
     * @return int  - handle of data channel buffer (negative if error occurs)
     * 		Error codes:
     * 			-99		memory allocation error
     * 			-504	size is invalid
     * 			-505	name is invalid
     *          -517	insufficient space to create a data channel buffer of the specified size
     * 			-518	data channel buffer with specified name already exists
     * 			< -500	internal implementation specific errors
     */
    private native int hc_createDCBuffer(String buffername, int size);

    /**
     * Native method to attach to the data channel buffer
     * @param buffername  java.lang.String  - name of data channel buffer
     * @return int  - handle of data channel buffer (negative if error occurs)
     * 		Error codes:
     * 			-99		memory allocation error
     * 			-3		data channel buffer with specified name does not exist
     * 			-505	name is invalid
     * 			< -500	internal implementation specific errors
     */
    private native int hc_attachToDCBuffer(String buffername);

    /**
     * Native method to stop flushing the data channel buffer
     * @param bufferhandle int  - handle of data channel buffer
     * @return int  - return code (negative if error occurs)
     */
    private native int hc_stopFlushingDCBuffer(int bufferhandle);

    /**
     * Native method to start flushing a Agent Controller data channel buffer to a file
     * @param bufferhandle int  - handle of data channel buffer
     * @param filename  java.lang.String  - name of file to flush to
     * @return int  - return code (negative if error occurs)
     */
    protected native int hc_flushRACDCBufferToFile(int bufferhandle, String filename);

    /**
     * Native method to start flushing an ASCII only data channel buffer to a file
     * @param bufferhandle int  - handle of data channel buffer
     * @param filename  java.lang.String  - name of file to flush to
     * @return int  - return code (negative if error occurs)
     */
    protected native int hc_flushASCIIDCBufferToFile(int bufferhandle, String filename);

    /**
     * Native method to start flushing a Binary only data channel buffer to a file
     * @param bufferhandle int  - handle of data channel buffer
     * @param filename  java.lang.String  - name of file to flush to
     * @return int  - return code (negative if error occurs)
     */
    protected native int hc_flushBinaryDCBufferToFile(int bufferhandle, String filename);

    /**
     * Native method to start flushing the data channel buffer to a socket
     * @param bufferhandle int  - handle of data channel buffer
     * @param inetaddress  byte[]  - IP address to flush to
     * @param port int  - port to flush to
     * @return int  - return code (negative if error occurs)
     */
    protected native int hc_flushDCBufferToSocket(int bufferhandle, byte[] inetaddess, int port);

    /**
     * Native method to start flushing the data channel buffer to an OutputStream
     *    - this should probably throw an IOException
     * @param bufferhandle int  - handle of data channel buffer
     * @param outstr  java.io.OutputStream  - OutputStream to flush to
     * @return int  - return code (negative if error occurs)
     */
    protected native int hc_flushDCBufferToStream(int bufferhandle, OutputStream outstr);


    /**
     * Native method to write an array of bytes to the data channel buffer
     * @param bufferhandle int  - handle of data channel buffer
     * @param databytes  byte[]  - byte array of data write
     * @param numBytes int  - number of bytes of data to write
     * @return int  - return code (negative if error occurs)
     * 		Error codes:
     *          -1		data channel buffer has not been created yet
     *          -2		invalid parameter
     * 			-99		memory allocation error
     */
    protected native int hc_writeToDCBuffer(int bufferhandle, byte[] databytes, int numBytes);



    /**
     * Retrieve the name of the data channel buffer
     * @return java.lang.String
     */
    public String getName() {
        return _dc_buffer_name;
    }

    /**
     * Retrieve the type of descriptor that describes the destination of the data.
     * @return int
     */
    public int getFlushType() {
        return _dc_flushType;
    }

    /**
     * Retrieve the size of the data channel buffer.
     * @return int
     */
    public int getSize() {
        return _dc_buffer_size;
    }

    /**
     * Retrieve the handle of the data channel buffer.
     * @return int
     */
    public int getHandle() {
        return _dc_buffer_handle;
    }

    /**
     * Retrieve the name of the file name to flush to.
     * @return java.lang.String
     */
    public String getFileName() {
        return _file_name;
    }

    /**
     * Retrieve the name of the file name to flush to.
     * @return java.lang.String
     */
    public InetAddress getInetAddress() {
        return _client_address;
    }

    /**
     * Retrieve the client port to flush to.
     * @return int
     */
    public int getPort() {
        return _client_port;
    }


    /**
     * Returns whether or not the data channel is being flushed.
     */
    public boolean isFlushing() {
        return _isFlushing;
    }

    /**
     * Returns whether or not the data channel has been created.
     */
    public boolean isDCBufferCreated() {
        return _isDCBufferCreated;
    }


    /**
     * Set the name of the data channel buffer
     * @param name java.lang.String
     */
    public void setName(String name) {
        _dc_buffer_name = new String(name);
    }

    /**
     * Set the internal piped input stream to read data from that will be written to the data channel
     * @param pis PipedInputStream
     */
    private void setPInputStream(PipedInputStream pis) {
        _dc_PInputStream = pis;
    }

    /**
     * Set the internal piped output stream to write data from the data channel to
     * @param pis PipedInputStream
     */
    private void setPOutputStream(PipedOutputStream pos) {
        _dc_POutputStream = pos;
    }

    /**
     * Set the type of destination for the data on the channel.
     * @param type int
     */
    private void setFlushType(int type) {
        _dc_flushType = type;
    }

    /**
     * Set the size of the data channel buffer.
     * @param type int
     */
    public void setSize(int size) {
        _dc_buffer_size = hc_validateDCBufferSize(size);
    }

    /**
     * Set the handle of the data channel buffer.
     * @param type int
     */
    private void setHandle(int handle) {
        _dc_buffer_handle = handle;
    }

    /**
     * Set the name of the file to flush the data channel buffer to
     * @param name java.lang.String
     */
    private void setFileName(String filename) {
        _file_name = new String(filename);
    }

    /**
     * Set the network address to flush the data channel buffer to
     * @param name java.net.InetAddress
     */
    private void setInetAddress(InetAddress address) {
        _client_address = address;
    }

    /**
     * Set the port number to flush the data channel buffer to.
     * @param type int
     */
    private void setPort(int portnum) {
        _client_port = portnum;
    }

/* Defines a thread that reads data from an input stream and writes it to the
 * data channel buffer.
 */

class WriterThread extends Thread {
	
	public void run() {

		byte[] inbytes = new byte[INPUT_BUFFER_SIZE];
		int bytesRead = 0;
		int rc;
		
		_isWriting = true;
		
		try {
			
			while (true) {
				
				/* Read the data from an input stream */
				bytesRead = getPInputStream().read(inbytes);

				/* If data was read from the input stream */			
				if (bytesRead > 0) {
					
					/* Call the native method to write the data to the channel */
					rc = hc_writeToDCBuffer(getHandle(), inbytes, bytesRead);
					
					if (rc != 0) {
						/* problem writing to data channel.  What shall we do? */
						System.out.println("Error " + rc + " writing to DataChannel " + bytesRead + " bytes");
						_isWriting = false;
						return;
					}
				}
				/* An error occured reading from the input stream or we have reached the end of the stream */
				else if (bytesRead < 0) {
					/* End of File reached - no more data to read so quit */
					_isWriting = false;								
					return;	
				}
			}
		}
		catch (IOException ioe) {
			_isWriting = false;	
		}
	}
	
}

/* Defines a thread that flushes the data on the channel to a destination.
 * The destination must be set before creating an object of this type.
 */

class FlusherThread extends Thread {
	
	public void run() {
		int rc = -1;

  		if (isDCBufferCreated() && !isFlushing()) {

  			_isFlushing = true;

		
			switch (getFlushType()) {
				case FLUSH_TYPE_RAC_FILE:
				
				   	rc = hc_flushRACDCBufferToFile(getHandle(), getFileName());
				   	break;
				   	
				case FLUSH_TYPE_ASCII_FILE:
					rc = hc_flushASCIIDCBufferToFile(getHandle(), getFileName());
					break;
					
				case FLUSH_TYPE_BINARY_FILE:
					rc = hc_flushBinaryDCBufferToFile(getHandle(), getFileName());
					break;
					
				case FLUSH_TYPE_SOCKET:
					/* Flush the data to the socket */
					rc = hc_flushDCBufferToSocket(getHandle(), getInetAddress().getAddress(), getPort());				
					break;
				case FLUSH_TYPE_OUTPUT_STREAM:
				
					/* Flush the data to an OutputStream */
					
					rc = hc_flushDCBufferToStream(getHandle(), _dc_outputStream);

					break;
				case FLUSH_TYPE_PIPED_OUTPUT_STREAM:
				
					/* Flush the data to an PipedOutputStream */
					
					rc = hc_flushDCBufferToStream(getHandle(), getPOutputStream());
					try {
						getPOutputStream().close();
					}
					catch (IOException ioe) {
						System.out.println("Could not close piped output stream");
					}
					break;
				default:
					/* No destination defined */				
			}

  			_isFlushing = false;

			if (rc != 0)
				return;

			/* clean up */

			hc_destroyDCBuffer(getHandle());
			
   		}
	}
}

}