/**********************************************************************
Copyright (c) 2005 IBM Corporation and others.
All rights reserved.  This program and the accompanying materials
are made available under the terms of the Eclipse Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/epl-v10.html
$Id: ThreadDumpAgentImpl.java,v 1.3 2005/03/02 21:30:26 paules Exp $

Contributors:
 IBM Rational - initial implementation
**********************************************************************/
package org.eclipse.hyades.logging.jvm.threadanalysis;

import java.io.File;
import java.util.Vector;

import org.eclipse.hyades.collection.threadanalyzer.DumpData;
import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.util.EventFormatter;

public class ThreadDumpAgentImpl implements ThreadDumpAgent {
	private static boolean debug = false;
	private final static String dllname = "hcthread";
	private String name = "Thread Analyzer Agent";
	private String type = "Logging";

	static {
		if (System.getProperty("debug") != null) {
			debug = true;
			debugTrace("Debug is ON");
		}
		System.loadLibrary(dllname);
		debugTrace("Finished loading native library: " + dllname);
	}

	public ThreadDumpAgentImpl() {
	}

	public static void main(String[] args) {
		ThreadDumpAgentImpl tdAgent = new ThreadDumpAgentImpl();
		tdAgent.init();

		while (true) {
			try {
				Thread.sleep(5000); // 5 sec
			} catch (InterruptedException e) {
				e.printStackTrace();
				tdAgent.dump();
				tdAgent.deregister();
			}
		}
	}

	/**
	 * Initialize the thread dump agent to the Agent Controller
	 */
	public void init() {
		String myname = name + ":" + this.hashCode();

		init0(myname, type);
		debugTrace("Initialized agent: " + myname);
	}

	/**
	 * Produce a thread dump.
	 */
	public void dump() {
		dumpThreads0();
		debugTrace("Dump completed");
	}

	/**
	 * Deregister the thread dump agent.
	 */
	public void deregister() {
		deregister0();
		debugTrace("Deregisterd agent");
	}

	/**
	 * Locate a javacore file based on the pid and the modified time.
	 * @param path The directory containing the javacore.
	 * @param pid The process ID of the running java process.
	 * @return The File object.
	 */
	private static File findJavaCore(String path, int pid) {
		File result = null;
		long newestTime = 0;

		File dir = new File(path); // open the directory
		if(dir != null) {
			debugTrace("Checking directory: " + path);
			File[] dirList = dir.listFiles();
			if(dirList != null) {
				for(int i = 0; i < dirList.length; i++) {
					String fileName = dirList[i].getName();
					debugTrace("Checking file: " + fileName);
					if(isJavaCore(fileName, pid)) {
						if(dirList[i].lastModified() > newestTime) {
							debugTrace("Modified time: " + dirList[i].lastModified());
							debugTrace("Base time: " + newestTime);
							result = dirList[i];
							newestTime = dirList[i].lastModified(); 
						}
					}
				}
			}
		}
		return result;
	}

	/**
	 * Check if a file is a valid javacore.
	 * @param name Name of the file.
	 * @param pid The process ID of the running java process.
	 * @return True if the name identifies a javacore file.
	 */
	private static boolean isJavaCore(String name, int pid) {
		if( (name.startsWith("javacore" + pid) && name.endsWith(".txt")) ||
				(name.startsWith("Javadump") && name.endsWith(".txt")) || //OS/390
				(name.equals("javacore.txt")) || // Linux
				(name.startsWith("javacore") && name.endsWith(pid + ".txt")) ) { // Windows
			return true;
		}
		/*
		 * Special case Linux: just look for latest javacore since the pid might not be the correct one.
		 * The dump process is mutex-locked so that the latest javacore should be correct.
		 */
		else if(System.getProperty("os.name").equals("Linux") && (name.startsWith("javacore") && name.endsWith(".txt"))) {
			return true;
		}
		else {
			return false;
		}
	}

	/**
	 * Called by the native code to process a thread dump.
	 * @param path Location of the javacore file.
	 */
	private static String processDumpData(String path) {
		Vector 			cbes;
		String          cbeString="";
		DumpData dd;
		File fJavaCore;
		String sJavaCore;
		ThreadDumpGeneratorImpl thdGenerator= new ThreadDumpGeneratorImpl();
		ThreadDumpProcessorImpl thdProcessor= new ThreadDumpProcessorImpl();

		fJavaCore = findJavaCore(path, getPid0());
		if(fJavaCore != null) {
			sJavaCore = fJavaCore.getAbsolutePath().replace('\\', '/');
			debugTrace("Javacore found: " + sJavaCore);

			dd = thdGenerator.dump(sJavaCore);
			debugTrace("DumpData obtained");

			if(dd != null) {
				cbes = thdProcessor.generateCBEs(dd);
				if(cbes != null) {
					debugTrace("CBE generated");
					for (int j=0; j<cbes.size(); j++) {
						cbeString = cbeString + EventFormatter.toCanonicalXMLString((CommonBaseEvent)cbes.elementAt(j),false);			
					}				
					return cbeString;		
				}
				else {
					return new String("<CommonBaseEvent msg=\"No common base event\"/>");
				}
			}
			else {
				return new String("<CommonBaseEvent msg=\"No dump data\"/>");		
			}
		}
		else {
			return new String("<CommonBaseEvent msg=\"No java core file\"/>");		
		}
	}

	/**
	 * This method can be called to log debug messages, which are enabled by the -Ddebug jvm option.
	 * The native code can also call this method to log debug message on the native side.
	 * @param msg
	 */
	private static void debugTrace(String msg) {
		if (debug) {
			System.out.println("DEBUG(JAVA): " + msg);
		}
	}

	/**
	 * Native code to initiate a thread dump.
	 * @return The location of the javacore file.
	 */
	private native void dumpThreads0();

	/**
	 * Native code to register the thread dump agent to the Agent Controller.
	 * @param name Name of the thread dump agent.
	 * @param type Type of the thread dump agent.
	 */
	private native void init0(String name, String type);

	/**
	 * Native code to deregister the thread dump agent to the Agent Controller.
	 */
	private native void deregister0();

	/**
	 * Native code to determine the process ID of the java process producing the javacore.
	 * @return
	 */
	private static native int getPid0();
}
