/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ReportWindowWizard.java,v 1.27 2010/05/05 12:32:03 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.common.internal.report;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.util.ExecutionUtil;
import org.eclipse.hyades.test.tools.ui.common.internal.report.jscrib.JscribWriter;
import org.eclipse.hyades.test.tools.ui.common.internal.report.jscrib.ReportDataUtils;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.navigator.ITestSuiteProxyNode;
import org.eclipse.hyades.test.ui.wizard.TestLocationPage;
import org.eclipse.hyades.ui.report.ReportGeneratorWizard;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.tptp.platform.report.core.internal.DDocument;
import org.eclipse.tptp.platform.report.core.internal.DLink;
import org.eclipse.tptp.platform.report.core.internal.DSection;
import org.eclipse.tptp.platform.report.drivers.html.DHtmlWriter;
import org.eclipse.tptp.platform.report.drivers.html.DHtmlWriterException;
import org.eclipse.tptp.platform.report.drivers.html.DOneDocumentPolicy;
import org.eclipse.tptp.platform.report.extension.internal.DExtensible;
import org.eclipse.tptp.platform.report.tools.internal.IDIImageProvider;
import org.eclipse.tptp.test.samples.internal.resources.SamplesPluginImageManager;
import org.eclipse.tptp.test.samples.internal.resources.SamplesPluginResourceBundle;

/**
 * <p>Wizard for soliciting the report window from the user.</p>
 *
 * <p>The report window consists of a start date and time and 
 * an end date and time (UTC).  The window is used to resolve 
 * execution results for a test suite within a finite period 
 * of time.</p>
 * 
 * 
 * @author     Patrick Nedelec
 * @author     Paul E. Slauenwhite
 * @version    May 4, 2010
 * @since      June 6, 2005
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 */
public class ReportWindowWizard extends ReportGeneratorWizard {

    private TestLocationPage testLocationPage = null;
    private ReportWindowWizardPage reportWindowInputWizardPage = null;
    private boolean onlyConsiderMostRecentResult = false;
    private Map testSuiteExecutionResults = null;
    private EList selectedTestSuites = null;
    
    private final static String FILE_EXTENSION = "html"; //$NON-NLS-1$

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#getEditorId()
     */
    public String getEditorId() {		
		return "org.eclipse.ui.browser.editorSupport"; //$NON-NLS-1$
	}

	/**
     * Constructor
     */
    public ReportWindowWizard() {
        super();
        setWindowTitle(SamplesPluginResourceBundle.ReportWindowWizardPage_Title); 
        setDefaultPageImageDescriptor(SamplesPluginImageManager.getInstance().getImageDescriptor(SamplesPluginImageManager.IMG_WIZBAN_REPORT_WINDOW));
    }

    /**
     * @see org.eclipse.hyades.ui.internal.wizard.HyadesWizard#initPages()
     */
    protected void initPages() {
        reportWindowInputWizardPage = new ReportWindowWizardPage();
        testLocationPage = new TestLocationPage("location", adjustLocation()); //$NON-NLS-1$
        testLocationPage.setTitle(UiPluginResourceBundle.WIZ_TST_REP_PG_TTL); 
        testLocationPage.setDescription(UiPluginResourceBundle.WIZ_TST_REP_PG_LOC_DSC); 
        testLocationPage.setFileExtension(FILE_EXTENSION);
    }

    protected IStructuredSelection adjustLocation() {
        IStructuredSelection structuredSelection = getSelection();
        Object selection = structuredSelection.getFirstElement();
        if (selection instanceof IResource) {
            IResource res = (IResource) selection;
            IContainer parent = res.getParent();
            if ((parent != null) && parent.exists()) {
                structuredSelection = new StructuredSelection(parent);
            }
        }
        return structuredSelection;
    }

    /**
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#addReportPages()
     */
    protected void addReportPages() throws Exception {
        addPage(testLocationPage);
        addPage(reportWindowInputWizardPage);
    }

    /**
     * Retrieves the start date and time in milliseconds (UTC) of the 
     * report window.
     * <p>
     * 
     * @return The start date and time in milliseconds (UTC).
     */
    public long getStartTime() {
        return reportWindowInputWizardPage.getStartTime();
    }

    /**
     * Retrieves the end date and time in milliseconds (UTC) of the 
     * report window.
     * <p>
     * 
     * @return The end date and time in milliseconds (UTC).
     */
    public long getEndTime() {
        return reportWindowInputWizardPage.getEndTime();
    }

    /**
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#generate(org.eclipse.core.runtime.IProgressMonitor)
     */
    public IFile generate(IProgressMonitor monitor) throws Exception {
       
    	try {
           
    		monitor.beginTask("", 4); //$NON-NLS-1$             

        	//Find the execution results for the selected test suite and all referenced test suite(s):
    		testSuiteExecutionResults = ExecutionUtil.findExecutionResults(ReportDataUtils.getReferencedTestSuites(selectedTestSuites));
    		            
            long startTime = getStartTime();
            long endTime = getEndTime();

            //- Retrieve the test suites to be reported
            IStructuredSelection structuredSelection = ((IStructuredSelection) (new StructuredSelection(getSelection()).getFirstElement()));
            Iterator structuredSelectionIterator = structuredSelection.iterator();

            while (structuredSelectionIterator.hasNext()) {
                ((ITestSuiteProxyNode) (structuredSelectionIterator.next())).getTestSuite();
            }

            IFile reportFile = getReportFile();
            
            if ( isOnlyConsiderMostRecentResult() )
            {
            	filterOldExecutionResults();
            }

            DSection[] docs = new JscribWriter().generate(/*reportFile,*/ testSuiteExecutionResults, startTime, endTime, new SubProgressMonitor(monitor, 2,
                    SubProgressMonitor.PREPEND_MAIN_LABEL_TO_SUBTASK));
            
            // create the main document
            DDocument doc = new DDocument("report_document", reportFile.getName()); //$NON-NLS-1$ 
            for (int i=0; i< docs.length; i++){
                doc.addChild(docs[i]);
            }
            
            monitor.worked(1);

            //- Writes the .html file:
            DDocument[] documents = new DDocument[1];
            documents[0] = doc;

            generateHtml(documents, new File(reportFile.getLocation().toOSString()));

            monitor.worked(1);

            return reportFile;
        } 
    	finally {
            monitor.done();
        }
    }

    public static class ReportGenerationPolicy extends DOneDocumentPolicy
    {

        public ReportGenerationPolicy(DHtmlWriter wh, String basePath, IDIImageProvider imgp) {
            super(wh, basePath, imgp);
        }
        
        public void doMethod(DLink item, DExtensible ext, Object arg) throws DHtmlWriterException {            
            getWriter().doChildrenItem(item, getWriter().getItem(item),ext,arg);
        }
    }
    
    /**
     * @param docs
     * @param f
     * @return
     * @throws DHtmlWriterException
     * @throws FileNotFoundException
     */
    private void generateHtml(DDocument[] docs, File f) {
        //- Compute the filename        
        final String HTML_EXTENSION = "html"; //$NON-NLS-1$  
        IPath basePath = new Path(f.getParent()).makeAbsolute().removeTrailingSeparator();
        String rxmlFileName = f.getName();
        String htmlFileName = rxmlFileName.substring(0, rxmlFileName.lastIndexOf(".")) + "." + HTML_EXTENSION; //$NON-NLS-1$ //$NON-NLS-2$
        String htmlFilePath = basePath.toOSString() + File.separatorChar + htmlFileName;
        try {
            DHtmlWriter writer = new DHtmlWriter();
            writer.setGraphicExtensionClass(DHtmlWriter.GRAPHIC_SVG);
            File file = new File(htmlFilePath);
            ReportGenerationPolicy rp = new ReportGenerationPolicy(writer,file.getParent(), null);
            DOneDocumentPolicy policy = new DOneDocumentPolicy(writer, f.getParent(), null);
            policy.setImageDirectoryName(f.getName()+"_files");
            writer.setOutputStream(new FileOutputStream(htmlFilePath));
            ((PrintStream)(writer.getOutputStream())).print("<html><head><META http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\"></head><body>"); //$NON-NLS-1$
            writer.write(rp, docs);
            ((PrintStream)(writer.getOutputStream())).print("</body></html>"); //$NON-NLS-1$
        } catch (DHtmlWriterException e) {
            ErrorDialog.openError(getShell(), e.getMessage(), null, null);
        }
        catch (FileNotFoundException e)
        {
            ErrorDialog.openError(getShell(), htmlFilePath+"\n\nError When Generating HTML file:\n"+e.getMessage(), null, null);
        } 
    }

    private IFile getReportFile() {
        IFile file = null;
        IPath path = testLocationPage.getContainerFullPath();
        IResource res = createContainer(path);
        path = new Path(testLocationPage.getFileName()); //$NON-NLS-1$
        file = ((IContainer) res).getFile(path);
        return file;
    }

    private IContainer createContainer(IPath path) {
        IContainer container = null;
        IWorkspace workbench = ResourcesPlugin.getWorkspace();
        int segCount = path.segmentCount();

        for (int idx = 0; idx < segCount; idx++) {
            if (idx == 0) {
                //project
                IProject project = workbench.getRoot().getProject(path.uptoSegment(idx + 1).toString());
                if (project == null || !project.exists()) {
                    //create the project
                    try {
                        project.create(null);
                    } catch (Exception e) {
                        UiPlugin.logError(e);
                        e.printStackTrace();
                    }
                }
                try {
                    project.open(null);
                } catch (Exception e) {
                }
                container = project;
            } else // (idx > 1)
            { //folder
                IFolder folder = workbench.getRoot().getFolder(path.uptoSegment(idx + 1));
                if (folder == null || !folder.exists()) {
                    //create the folder
                    try {
                        folder.create(false, true, null);
                    } catch (Exception e) {
                        UiPlugin.logError(e);
                        e.printStackTrace();
                    }
                }
                container = folder;
            }
        }
        try {
            container.getProject().refreshLocal(IResource.DEPTH_INFINITE, null);
        } catch (CoreException exc) {
            UiPlugin.logError(exc);
            exc.printStackTrace();
        }
        return container;
    }

    /**
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#isAvailable(org.eclipse.jface.viewers.ISelection)
     */
    public boolean isAvailable(ISelection selection) {
    	
    	//Step 1: Determine if the selection contains at least one element:
		if ((selection == null) || (selection.isEmpty())){ 
			return false;
		}
		
		//Step 2: Determine if the selection element contains test suite(s) and resolve the selected test suite(s):
        selectedTestSuites = new BasicEList();
		Iterator selectionIterator = ((IStructuredSelection) (new StructuredSelection(selection).getFirstElement())).iterator();
        
		while (selectionIterator.hasNext()) {
             
            Object testSuiteProxyNode = selectionIterator.next();

            if (testSuiteProxyNode instanceof ITestSuiteProxyNode){
            	selectedTestSuites.add(((ITestSuiteProxyNode) (testSuiteProxyNode)).getTestSuite());
            }
            else{

            	selectedTestSuites = null;
    			
    			return false;
            }
        }     
		
		//Step 3: Determine if at least one test suite has been selected:
		if(selectedTestSuites.size() == 0){
			
			selectedTestSuites = null;
			
			return false;
		}
		
		return true;
    }

    protected boolean isOnlyConsiderMostRecentResult() {
		return onlyConsiderMostRecentResult;
	}

	protected void setOnlyConsiderMostRecentResult(boolean onlyConsiderMostRecentResult) {
		this.onlyConsiderMostRecentResult = onlyConsiderMostRecentResult;
	}

	/**
	 * This method filters the map of TestSuites to ExecutionResults by 
	 * removing all but the last execution result that falls within the 
	 * stard and end ranges of the report window.
	 */
	private void filterOldExecutionResults() {
		List keysToRemove = new ArrayList();
		Iterator iter = testSuiteExecutionResults.entrySet().iterator();
		while (iter.hasNext())
		{
			Map.Entry entry = (Map.Entry)iter.next();
			List results = (List) entry.getValue();
			Iterator resultIter = results.iterator();
			TPFExecutionResult latestResult = null;
			long latestTimestamp = 0;
			while (resultIter.hasNext())
			{
				TPFExecutionResult result = (TPFExecutionResult) resultIter.next();
				TPFExecutionHistory history = result.getExecutionHistory();
				if ( history != null && !history.getExecutionEvents().isEmpty())
				{
					TPFExecutionEvent event = (TPFExecutionEvent) history.getExecutionEvents().get(0);
					long timestamp = event.getTimestamp();
					if (timestamp > latestTimestamp && timestamp > getStartTime() &&
						timestamp < getEndTime())
					{
						latestResult = result;
						latestTimestamp = timestamp;
					}
				}
			}
			if (latestResult != null)
			{
				results.clear();
				results.add(latestResult);
			}
			else
			{
				// Don't modify the map while we're iterating over it
				keysToRemove.add(entry.getKey());
			}
		}

		// Remove any testsuite / execution result pairs for which no results
		// appear within the report window.
		Iterator removeIter = keysToRemove.iterator();
		while ( removeIter.hasNext() )
		{
			Object key = removeIter.next();
			testSuiteExecutionResults.remove(key);
		}
		
	}


}
