/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JscribWriter.java,v 1.22 2010/04/09 01:58:33 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.tools.ui.common.internal.report.jscrib;

import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.testprofile.TPFVerdict;
import org.eclipse.hyades.models.common.testprofile.impl.TPFTestSuiteImpl;
import org.eclipse.hyades.test.tools.ui.common.internal.report.DescendingVisitor;
import org.eclipse.hyades.test.ui.internal.model.EventUtil;
import org.eclipse.osgi.util.NLS;
import org.eclipse.tptp.platform.report.core.internal.DAxis;
import org.eclipse.tptp.platform.report.core.internal.DCategory;
import org.eclipse.tptp.platform.report.core.internal.DCell;
import org.eclipse.tptp.platform.report.core.internal.DCellText;
import org.eclipse.tptp.platform.report.core.internal.DColor;
import org.eclipse.tptp.platform.report.core.internal.DCoord;
import org.eclipse.tptp.platform.report.core.internal.DCoordObject;
import org.eclipse.tptp.platform.report.core.internal.DCurve;
import org.eclipse.tptp.platform.report.core.internal.DDocument;
import org.eclipse.tptp.platform.report.core.internal.DGraphic;
import org.eclipse.tptp.platform.report.core.internal.DItem;
import org.eclipse.tptp.platform.report.core.internal.DParagraph;
import org.eclipse.tptp.platform.report.core.internal.DPoint;
import org.eclipse.tptp.platform.report.core.internal.DRow;
import org.eclipse.tptp.platform.report.core.internal.DSection;
import org.eclipse.tptp.platform.report.core.internal.DStyle;
import org.eclipse.tptp.platform.report.core.internal.DStyleRegistry;
import org.eclipse.tptp.platform.report.core.internal.DTable;
import org.eclipse.tptp.platform.report.core.internal.DText;
import org.eclipse.tptp.platform.report.core.internal.IDAlignment;
import org.eclipse.tptp.platform.report.core.internal.IDItem;
import org.eclipse.tptp.platform.report.core.internal.IDStyle;
import org.eclipse.tptp.platform.report.tools.internal.DAlignment;
import org.eclipse.tptp.test.samples.internal.resources.SamplesPluginResourceBundle;

import com.ibm.icu.text.NumberFormat;
import com.ibm.icu.util.Calendar;

/**
 * <p>Writes the JScrib document or {@link DDocument}.</p>
 * 
 * 
 * @author     Patrick Nedelec
 * @author     Paul E. Slauenwhite
 * @version    April 8, 2010
 * @since      June 6, 2005
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 */
public class JscribWriter extends DescendingVisitor {
    private DStyleRegistry styleRegistry;


    protected Map resultsByTestSuiteMap;
    protected Map resultsByExecutionMap;
    protected Map execsByTestSuiteMap;
    private int[] resultSet;

    /**
     * 
     */
    public JscribWriter() {
        this.styleRegistry = null;
        //this.outputDirectory = null;
        //this.outputFile = null;
        this.resultsByTestSuiteMap = new HashMap();
        this.resultsByExecutionMap = new HashMap();
        this.execsByTestSuiteMap = new HashMap();
        this.resultSet = new int[4];
    }

    /**
     * @param destFile The file to be generated
     * @param executionsByTestsuiteMap A map <TPFTestSuite, List(IFile)>.
     * @param startTime the start time for the execution results capture.
     * @param endTime the end time for the execution results capture.
     * @param monitor A progress monitor.
     * 
     * @return The generated DDocument.
     */
    public DSection[] generate(/*IFile destFile,*/ Map executionsByTestsuiteMap_in, long startTime, long endTime, IProgressMonitor monitor) throws Exception {
        try {
            
            TreeMap executionsByTestsuiteMap = new TreeMap(new Comparator() {

                public int compare(Object arg0, Object arg1) {
                    if ((arg0 instanceof TPFTestSuiteImpl) && (arg1 instanceof TPFTestSuiteImpl))
                    {
                        String n1 = ((TPFTestSuiteImpl)arg0).getName();
                        String n2 = ((TPFTestSuiteImpl)arg1).getName();
                        return n1.compareTo(n2);
                    }
                    return 0;
                }
                
            });
            executionsByTestsuiteMap.putAll(executionsByTestsuiteMap_in);


            styleRegistry = StylesInitializer.getDefaultReportStyles();
            //outputFile = destFile.getFullPath().makeAbsolute().toOSString();
            //outputDirectory = outputFile.substring(0, outputFile.lastIndexOf(File.separator));
            //if (outputDirectory.charAt(outputDirectory.length() - 1) != IPath.SEPARATOR && outputDirectory.charAt(outputDirectory.length() - 1) != '\\' && //$NON-NLS-1$
            //        outputDirectory.charAt(outputDirectory.length() - 1) != '/') //$NON-NLS-1$
            //    outputDirectory += IPath.SEPARATOR;

            monitor.beginTask(SamplesPluginResourceBundle.JScribWriter_Message, 5); 

            monitor.worked(1);

            //- Create details document
            DSection detailsDocument = createDetailsDocument(executionsByTestsuiteMap, startTime, endTime);
            monitor.worked(1);
            
            DSection graphicSection = new DSection("graphic_section"); //$NON-NLS-1$ 
                
            graphicSection.addChild(ReportGraphicUtils.createTitle(SamplesPluginResourceBundle.JScribWriter_Overview, 1, getStyle(SamplesPluginResourceBundle.StylesInitializer_TITLE1), null)); 
            
            // Graphic Table
            DTable graphTable = new DTable();
            //graphTable.setColumnWidth(new float[]{0.3f,0.7f});
            graphTable.setBorder(0);
            graphicSection.addChild(graphTable);
            graphTable.addChild(new DRow());
            DCell graphCell[] = new DCell[2]; 
             
            graphTable.getFirstChild().addChild(graphCell[0]=new DCell());
            graphTable.getFirstChild().addChild(graphCell[1]=new DCell());
           // graphTable.getFirstChild().addChild(graphCell[2]=new DCell()); 

            //- Create the summary document
            graphCell[0].addChild(createSummaryChart());
            monitor.worked(1);
            // 
            graphCell[1].addChild(createSummaryTable(executionsByTestsuiteMap, startTime, endTime));
            monitor.worked(1);
            //- Create the by Test Suite document
            /*graphCell[1].addChild(createByTestSuiteDocument());
            monitor.worked(1);*/
            //- Create the by Execution document
            /*graphCell[2].addChild(createByExecutionDocument());
            monitor.worked(1);*/

            return new DSection[] { graphicSection, detailsDocument};
        } finally {
            monitor.done();
        }
    }

    /**
     * @param executionsByTestsuiteMap
     * @param startTime
     * @param endTime
     * @param monitor
     * @throws Exception
     */
    private DSection createDetailsDocument(Map executionsByTestsuiteMap, long startTime, long endTime) throws Exception {
        DSection indexSection = new DSection("index"); //$NON-NLS-1$
        //- Start the visit and fill the details document
        DItem title = ReportGraphicUtils.createTitle(SamplesPluginResourceBundle.JScribWriter_ResultsByTestSuite, 1, getStyle(SamplesPluginResourceBundle.StylesInitializer_TITLE1), null); 
        indexSection.addChild(title);

        DParagraph paragraph = new DParagraph();
        indexSection.addChild(paragraph);
        
        Iterator testSuiteIterator = executionsByTestsuiteMap.keySet().iterator();
        //- Iterate all test suites:
        while (testSuiteIterator.hasNext()) {
            //- Resolve the current test suite and relevant information:
            TPFTestSuite testSuite = ((TPFTestSuite) (testSuiteIterator.next()));
            accept(new TestSuiteVisitor(executionsByTestsuiteMap, startTime, endTime), testSuite, paragraph);
        }
        
        indexSection.insertChild(createByTestSuiteChart(), title);
        //indexSection.insertChild(createByExecutionChart(), title);
        
        return indexSection;
    }
        
    private IDItem createSummaryTable(Map executionsByTestsuiteMap, long startTime, long endTime)
    {
        DParagraph p = new DParagraph();
        DTable table = new DTable();
        p.addChild(table);
        createStatsSummary(table, executionsByTestsuiteMap, startTime, endTime);

        return p;
    }

    private void createStatsSummary(DTable table, Map executionsByTestsuiteMap, long startTime, long endTime) {
        
    	//Tests:
    	createTestSuitesRow(table, executionsByTestsuiteMap.keySet());
        
        //Verdict:
    	int verdict = ReportDataUtils.getVerdict(resultSet);
    	createRow(table, SamplesPluginResourceBundle.JScribWriter_Verdict, getVerdictLabel(verdict), 1, getVerdictCellStyle(verdict)); 
        
        //Report file creation date:
        createRow(table, SamplesPluginResourceBundle.JScribWriter_CreationDate, EventUtil.getTime(Calendar.getInstance().getTimeInMillis()), 1); 

    	//Start time:
        createRow(table, SamplesPluginResourceBundle.JScribWriter_StartTime, EventUtil.getTime(startTime), 1); 
        
    	//End time:
        createRow(table, SamplesPluginResourceBundle.JScribWriter_EndTime, EventUtil.getTime(endTime), 1); 

        //Pass/Fail/Inconclusive/Error:
        createVerdictRows(table);
    }

    private void createRow(DTable table, String headerText, String valueText, int colspan) {
        createRow(table, headerText, valueText, colspan, null);
    }

    private void createRow(DTable table, String headerText, String valueText, int colspan, IDStyle style) {
        DRow row = new DRow();

        DCell c1 = ReportGraphicUtils.createStyledCell(getStyle(SamplesPluginResourceBundle.StylesInitializer_HEADER_CELL), IDAlignment.LEFT, headerText);
        DCell c2 = ReportGraphicUtils.createStyledCell(getStyle(SamplesPluginResourceBundle.StylesInitializer_NORMAL_CELL), IDAlignment.LEFT, valueText);
        c1.setColSpan(colspan);
        c2.setColSpan(colspan);
        if (style != null) {
            c2.setStyle(style);
        }
        row.addChild(c1);
        row.insertChild(c2, c1);
        table.addChild(row);
    }

    private void createTestSuitesRow(DTable table, Collection testSuites) {
        DRow row1 = new DRow();
        DRow row2 = new DRow();
        DCell c1 = ReportGraphicUtils.createStyledCell(getStyle(SamplesPluginResourceBundle.StylesInitializer_HEADER_CELL), IDAlignment.LEFT | IDAlignment.VCENTER, SamplesPluginResourceBundle.JScribWriter_TestSuites); //$NON-NLS-1$
        DCell c2 = ReportGraphicUtils.createReportLinksCell(testSuites, getStyle(SamplesPluginResourceBundle.StylesInitializer_HEADER_CELL), getStyle(SamplesPluginResourceBundle.StylesInitializer_LINK_CELL));
        c1.setColSpan(3);
        c2.setColSpan(3);
        row1.addChild(c1);
        row2.addChild(c2);
        //row.insertChild(c2, c1);
        table.addChild(row1);
        table.addChild(row2);
    }

    private void createVerdictRows(DTable table) {
        int[] verdicts = new int[] { TPFVerdict.PASS, TPFVerdict.INCONCLUSIVE, TPFVerdict.FAIL, TPFVerdict.ERROR};
        DCell previous = null;
        for (int i = 0, max = resultSet.length; i < max; i++) {
            DRow row = new DRow();
            DCell c1 = ReportGraphicUtils.createStyledCell(getStyle(SamplesPluginResourceBundle.StylesInitializer_HEADER_CELL), IDAlignment.LEFT, getVerdictLabel(verdicts[i]));
            DCell c2 = ReportGraphicUtils.createStyledCell(getVerdictCellStyle(verdicts[i]), getVerdictCellTextStyle(verdicts[i]), IDAlignment.LEFT, ReportDataUtils
                    .normalize(resultSet[i]));
            if (previous == null) {
                row.addChild(c1);
            } else {
                row.insertChild(c1, previous);
            }
            row.insertChild(c2, c1);
            table.addChild(row);
            previous = null;
            if (i % 2 == 0) {
                previous = c2;
            }
        }
    }

    /**
     * @return
     */
    private IDItem createSummaryChart() {
        //- Compute the four percentages
        int total = resultSet[0] + resultSet[1] + resultSet[2] + resultSet[3];
        double passPercentage = getPercent(resultSet[0], total);
        double inconclusivePercentage = getPercent(resultSet[1], total);
        double failPercentage = getPercent(resultSet[2], total);
        double errorPercentage = getPercent(resultSet[3], total);
        
        if ((passPercentage==0) && (inconclusivePercentage==0)
                && (failPercentage==0) && (errorPercentage==0))
        {
            DParagraph p = new DParagraph(DParagraph.CENTER);
            p.addChild(new DText(SamplesPluginResourceBundle.JScribWriter_NO_CHART_VALUE));
            p.setStyle(getStyle(SamplesPluginResourceBundle.StylesInitializer_GENERATION_ERROR));
            return p;
        }

        //- Graphic
        DGraphic pieChart = new DGraphic();
       // graphCell[0].addChild(pieChart);
        pieChart.setGraphicType(DGraphic.T_SECTORS3D);
        pieChart.setStyle(getStyle(SamplesPluginResourceBundle.StylesInitializer_CHART_TITLE));
        pieChart.setTitle(SamplesPluginResourceBundle.JScribWriter_Title_ExecutionResultsRepartition); 
        //pieChart.setHeightScale(0.6f);
        //pieChart.setWidthScale(0.6f);
        pieChart.getProperties().store(DGraphic.P_VALUE_FORMAT , "##.##"); //$NON-NLS-1$
        pieChart.getProperties().store(DGraphic.P_SHOW_PERCENTAGE, true);
        pieChart.getProperties().store(DGraphic.P_SHOW_VALUES, false);
       
        //- Axis
        DAxis axis = new DAxis("y", SamplesPluginResourceBundle.JScribWriter_Symbol_Percentage, SamplesPluginResourceBundle.JScribWriter_Values); 
        axis.getProperties().store(DAxis.P_LABEL_FORMATTYPE, "number");
        axis.getProperties().store(DAxis.P_LABEL_FORMAT, "##.##");
        
        pieChart.addChild(axis);
        DAxis catAxis = new DAxis("x"); //$NON-NLS-1$//$NON-NLS-2$ //$NON-NLS-3$
        DCategory category = new DCategory();
        category.setId("c1");
        category.setLabel(SamplesPluginResourceBundle.JScribWriter_Pass); 
        catAxis.addChild(category);
        category = new DCategory();
        category.setId("c2");
        category.setLabel(SamplesPluginResourceBundle.JScribWriter_Inconclusive); 
        catAxis.addChild(category);
        category = new DCategory();
        category.setId("c3");
        category.setLabel(SamplesPluginResourceBundle.JScribWriter_Fail); 
        catAxis.addChild(category);
        category = new DCategory();
        category.setId("c4");
        category.setLabel(SamplesPluginResourceBundle.JScribWriter_Error); 
        catAxis.addChild(category);
        pieChart.addChild(catAxis);
        //- Curve(s)
        DCurve curvePass = new DCurve();
        curvePass.setStyle(getStyle(SamplesPluginResourceBundle.StylesInitializer_PASS));
        curvePass.setType("SECTOR.DCurve.core.jscrib"); //$NON-NLS-1$
        curvePass.setName(SamplesPluginResourceBundle.JScribWriter_Pass); 
        DPoint pointPass = new DPoint();
        DCoord coordPass = new DCoord(axis, passPercentage);
        DCoordObject categoryCoord = new DCoordObject(catAxis, "c1"); //$NON-NLS-1$
        pointPass.addChild(coordPass); 
        pointPass.addChild(categoryCoord);
        curvePass.addChild(pointPass);
        pieChart.addChild(curvePass);
        DCurve curveInconclusive = new DCurve();
        curveInconclusive.setStyle(getStyle(SamplesPluginResourceBundle.StylesInitializer_INCONCLUSIVE));
        curveInconclusive.setType("SECTOR.DCurve.core.jscrib"); //$NON-NLS-1$
        curveInconclusive.setName(SamplesPluginResourceBundle.JScribWriter_Inconclusive); 
        DPoint pointInconclusive = new DPoint();
        DCoord coordInconclusive = new DCoord(axis, inconclusivePercentage);
        DCoordObject coordInconclusiveCategory = new DCoordObject(catAxis, "c2"); //$NON-NLS-1$
        pointInconclusive.addChild(coordInconclusive);
        pointInconclusive.addChild(coordInconclusiveCategory);
        curveInconclusive.addChild(pointInconclusive);  
        pieChart.addChild(curveInconclusive);
        DCurve curveFail = new DCurve();
        curveFail.setStyle(getStyle(SamplesPluginResourceBundle.StylesInitializer_FAIL));
        curveFail.setType("SECTOR.DCurve.core.jscrib"); //$NON-NLS-1$
        curveFail.setName(SamplesPluginResourceBundle.JScribWriter_Fail); 
        DPoint pointFail = new DPoint();
        DCoord coordFail = new DCoord(axis, failPercentage);
        DCoordObject coordFailCategory = new DCoordObject(catAxis, "c3"); //$NON-NLS-1$
        pointFail.addChild(coordFailCategory);
        pointFail.addChild(coordFail);
        curveFail.addChild(pointFail);
        pieChart.addChild(curveFail);
        DCurve curveError = new DCurve();
        curveError.setStyle(getStyle(SamplesPluginResourceBundle.StylesInitializer_ERROR));
        curveError.setType("SECTOR.DCurve.core.jscrib"); //$NON-NLS-1$
        curveError.setName(SamplesPluginResourceBundle.JScribWriter_Error); 
        DPoint pointError = new DPoint();
        DCoord coordError = new DCoord(axis, errorPercentage);
        DCoordObject coordErrorCategory = new DCoordObject(catAxis, "c4"); //$NON-NLS-1$
        pointError.addChild(coordErrorCategory);
        pointError.addChild(coordError);
        curveError.addChild(pointError);
        pieChart.addChild(curveError);
        return pieChart;
    }

    public double getPercent(int iPart, int iWhole) {
        return (iWhole == 0) ? 0 : (((double) iPart) / iWhole) * 100.0;
    }

    /**
     * @return
     */
    private IDItem createByTestSuiteChart() {
        
        //Four points by test suite
        HashMap validData = new HashMap();
        //int emptyTest = 0;
        Iterator it = resultsByTestSuiteMap.keySet().iterator();
        while (it.hasNext()) {
            TPFTestSuite ts = (TPFTestSuite) it.next();
            int[] res = (int[]) resultsByTestSuiteMap.get(ts);
            if ((res[0]!=0) || (res[1]!=0) || (res[2]!=0) || (res[3]!=0)) 
                validData.put(ts, res);
        }
        
        if (validData.size() == 0)
        {
            DParagraph p = new DParagraph(DParagraph.CENTER);
            p.addChild(new DText(SamplesPluginResourceBundle.JScribWriter_NO_CHART_VALUE));
            p.setStyle(getStyle(SamplesPluginResourceBundle.StylesInitializer_GENERATION_ERROR));
            return p;
        }
        
        //- Graphic
        DGraphic stackbar = new DGraphic();
        stackbar.setGraphicType("T_STACKBARS.DGraphic.core.jscrib"); //$NON-NLS-1$
        stackbar.setTitle(SamplesPluginResourceBundle.JScribWriter_Title_ExecutionResultsByTestSuite); 
        //stackbar.setHeightScale(0.7f);
        //stackbar.setWidthScale(0.5f);

        stackbar.getProperties().store(DGraphic.P_3D, true);
        stackbar.getProperties().store(DGraphic.P_THIN_BAR, true);
        int wHint = (validData.size()*20>500) ? validData.size()*20:500;
        stackbar.getProperties().store(DGraphic.P_MAX_WIDTH, wHint);
        stackbar.getProperties().store(DGraphic.P_MAX_HEIGHT, (int)(wHint));// * 0.7f));
        stackbar.getProperties().store(DGraphic.P_MIN_WIDTH, wHint);
        stackbar.getProperties().store(DGraphic.P_MIN_HEIGHT, (int)(wHint));// * 0.7f));
        
        
        int max=0;
        it = validData.keySet().iterator(); 
        while (it.hasNext()) {
            TPFTestSuite ts = (TPFTestSuite) it.next();
            int[] res = (int[]) validData.get(ts);
            int val = res[0]+res[1]+res[2]+res[3];
            max = Math.max(val, max);
        }
        
        //Round up to the nearest tens:
        if(max > 10){
        	max = ((int)(Math.ceil(((double)(max))/10.0))) * 10;
        }
        
        //- Axis
        DAxis axis_y = new DAxis("y", "", ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        axis_y.getProperties().store(DAxis.P_STEP_UNIT, max/2);
        axis_y.getProperties().store(DAxis.P_STEP_DOT, max/4);
        axis_y.setStyle(getStyle(SamplesPluginResourceBundle.StylesInitializer_NORMAL_CELL));
        axis_y.getProperties().store(DAxis.P_MAJUNIT_VALUE, max/2);
        axis_y.getProperties().store(DAxis.P_MINUNIT_VALUE, max/4);
        axis_y.getProperties().store(DAxis.P_MAX, max);
        stackbar.addChild(axis_y);
        DAxis axis_x = new DAxis("x", "", ""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        axis_x.getProperties().store(DAxis.P_LABEL_ALIGNMENT, new DAlignment(IDAlignment.TOP));
        axis_x.setStyle(getStyle(SamplesPluginResourceBundle.StylesInitializer_NORMAL_CELL));
        stackbar.addChild(axis_x);
        
        //- One curve by verdict
        DCurve curvePass = new DCurve();
        curvePass.setStyle(getStyle(SamplesPluginResourceBundle.StylesInitializer_PASS));
        curvePass.setType("SECTOR.DCurve.core.jscrib"); //$NON-NLS-1$
        curvePass.setName(SamplesPluginResourceBundle.JScribWriter_Pass); 
        DCurve curveInconclusive = new DCurve();
        curveInconclusive.setStyle(getStyle(SamplesPluginResourceBundle.StylesInitializer_INCONCLUSIVE));
        curveInconclusive.setType("SECTOR.DCurve.core.jscrib"); //$NON-NLS-1$
        curveInconclusive.setName(SamplesPluginResourceBundle.JScribWriter_Inconclusive); 
        DCurve curveError = new DCurve();
        DCurve curveFail = new DCurve();
        curveFail.setStyle(getStyle(SamplesPluginResourceBundle.StylesInitializer_FAIL));
        curveFail.setType("SECTOR.DCurve.core.jscrib"); //$NON-NLS-1$
        curveFail.setName(SamplesPluginResourceBundle.JScribWriter_Fail); 
        curveError.setStyle(getStyle(SamplesPluginResourceBundle.StylesInitializer_ERROR));
        curveError.setType("SECTOR.DCurve.core.jscrib"); //$NON-NLS-1$
        curveError.setName(SamplesPluginResourceBundle.JScribWriter_Error); 
        
        it = validData.keySet().iterator(); //resultsByTestSuiteMap.keySet().iterator();
        while (it.hasNext()) {
            TPFTestSuite ts = (TPFTestSuite) it.next();
            int[] res = (int[]) validData.get(ts); //resultsByTestSuiteMap.get(ts);
            
            DCategory cat = new DCategory();
            cat.setId(ts.getName());
            cat.setLabel(ts.getName());
            axis_x.addChild(cat);
            
            DPoint pointPass = new DPoint();
            DCoord coordPass = new DCoord(axis_y, res[0]);
            DText txt = new DText(ts.getName());
            //txt.setStyle(getStyle(StylesInitializer.AXIS));
            pointPass.addChild(coordPass);
            pointPass.addChild(new DCoordObject(axis_x, ts.getName()));
            pointPass.addChild(txt);
            curvePass.addChild(pointPass);

            DPoint pointInconclusive = new DPoint();
            DCoord coordInconclusive = new DCoord(axis_y, res[1]);
            pointInconclusive.addChild(coordInconclusive);
            pointInconclusive.addChild(new DCoordObject(axis_x, ts.getName()));
            curveInconclusive.addChild(pointInconclusive);

            DPoint pointFail = new DPoint();
            DCoord coordFail = new DCoord(axis_y, res[2]);
            pointFail.addChild(coordFail);
            pointFail.addChild(new DCoordObject(axis_x, ts.getName()));
            curveFail.addChild(pointFail);

            DPoint pointError = new DPoint();
            DCoord coordError = new DCoord(axis_y, res[3]);
            pointError.addChild(coordError);
            pointError.addChild(new DCoordObject(axis_x, ts.getName()));
            curveError.addChild(pointError);
        }

        stackbar.addChild(curvePass);
        stackbar.addChild(curveInconclusive);
        stackbar.addChild(curveFail);
        stackbar.addChild(curveError);
        return stackbar;
    }
   
    public IDStyle getStyle(String styleName) {
        IDStyle ret = styleRegistry.getFirstStyle(styleName);
        return ret;
    }

    /**
     * Utility API generate a coloured rate cell (e.g.
     * <td></td>) with a specified title within a HTML table.
     * <p>
     * Rates are calculated as a simple percentage, dividing the numerator by
     * the denominator and multiplying by 100. Both the numerator and
     * denominator may be negative or zero.
     * <p>
     * The cell is coloured using an increasing (0% [white] --> 100% [red]) or
     * decreasing (0% [red] --> 100% [white]) gradiant.
     * <p>
     * 
     * @param numerator
     *            The numerator of the rate calculation.
     * @param denominator
     *            The denominator of the rate calculation.
     * @param increasingGradiant
     *            Flag indicating if an increasing or decreasing gradiant.
     * @param titleName
     *            The title attribute of the of the rate cell (e.g.
     *            <td title=""></td>)
     * @return The coloured rate cell (e.g.
     *         <td></td>) with a specified title within a table.
     */
    public DCellText generateRateCell(int numerator, int denominator, boolean increasingGradiant, IDStyle style) {
        //- Compute rate and color
        double rate = 0;
        if ((numerator > 0) && (denominator > 0)) {
            rate = (((double) (numerator)) / denominator);
        }
        int green = 255;
        int blue = 255;
        if (((rate > 0) && (rate < 1)) || ((rate == 0) && (!increasingGradiant))) {
            green = blue = Math.abs((increasingGradiant ? 225 : 0) - ((int) (rate * 225)));
        }
        //- Create the cell
        NumberFormat numberFormatter = NumberFormat.getInstance();
        numberFormatter.setMaximumFractionDigits(2);
        String cellValue = numberFormatter.format(rate * 100.0) + "%"; //$NON-NLS-1$
        DCellText rateCell = new DCellText(cellValue);
        String styleName = NLS.bind(SamplesPluginResourceBundle.StylesInitializer_RATE_CELL, cellValue); 
        IDStyle styleCell = getStyle(styleName);
        if (styleCell == null) {
            //- Then create a new style and register it.
            styleCell = new DStyle(styleName, style);
            styleRegistry.putStyle(styleCell); //$NON-NLS-1$
        }
        styleCell.setBackColor(new DColor(255, green, blue));
        rateCell.setStyle(styleCell);
        return rateCell;
    }

    public String getVerdictLabel(int verdict) {

        switch (verdict) {
            case TPFVerdict.ERROR:
            	return SamplesPluginResourceBundle.JScribWriter_Error; 
            case TPFVerdict.FAIL:
            	return SamplesPluginResourceBundle.JScribWriter_Fail; 
            case TPFVerdict.INCONCLUSIVE:
            	return SamplesPluginResourceBundle.JScribWriter_Inconclusive; 
            case TPFVerdict.PASS:
            	return SamplesPluginResourceBundle.JScribWriter_Pass; 
        }
        
        return SamplesPluginResourceBundle.JScribWriter_NoVerdict;
    }

    public IDStyle getVerdictCellStyle(int verdict) {

        switch (verdict) {
            case TPFVerdict.ERROR:
                return getStyle(SamplesPluginResourceBundle.StylesInitializer_ERROR_CELL);              
            case TPFVerdict.FAIL:
            	return getStyle(SamplesPluginResourceBundle.StylesInitializer_FAIL_CELL);
            case TPFVerdict.INCONCLUSIVE:
            	return getStyle(SamplesPluginResourceBundle.StylesInitializer_INCONCLUSIVE_CELL);
            case TPFVerdict.PASS:
            	return getStyle(SamplesPluginResourceBundle.StylesInitializer_PASS_CELL);
        }
        
        return getStyle(SamplesPluginResourceBundle.StylesInitializer_NORMAL_CELL);
    }

    public IDStyle getVerdictCellTextStyle(int verdict) {

        switch (verdict) {
            case TPFVerdict.ERROR:
                return getStyle(SamplesPluginResourceBundle.StylesInitializer_ERROR_CELL_TEXT);
            case TPFVerdict.FAIL:
                return getStyle(SamplesPluginResourceBundle.StylesInitializer_FAIL_CELL_TEXT);
            case TPFVerdict.INCONCLUSIVE:
                return getStyle(SamplesPluginResourceBundle.StylesInitializer_INCONCLUSIVE_CELL_TEXT);
            case TPFVerdict.PASS:
                return getStyle(SamplesPluginResourceBundle.StylesInitializer_PASS_CELL_TEXT);
        }
        
        return getStyle(SamplesPluginResourceBundle.StylesInitializer_PASS_CELL_TEXT);
    }

    public void addToResults(int[] newResultSet) {
        for (int i = 0, max = resultSet.length; i < max; i++) {
            this.resultSet[i] += newResultSet[i];
        }
    }

}
