/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ReportDataUtils.java,v 1.10 2008/12/23 17:58:54 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.tools.ui.common.internal.report.jscrib;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.common.fragments.BVRCombinedFragment;
import org.eclipse.hyades.models.common.fragments.BVRInteractionOperand;
import org.eclipse.hyades.models.common.interactions.BVRExecutionOccurrence;
import org.eclipse.hyades.models.common.interactions.BVRInteractionFragment;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFInvocationEvent;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.testprofile.TPFVerdict;
import org.eclipse.hyades.models.common.testprofile.TPFVerdictEvent;
import org.eclipse.hyades.models.common.util.ExecutionUtil;

import com.ibm.icu.text.NumberFormat;

/**
 * This util class is useful to generate data for the JScrib report.
 * 
 * 
 * @author     Patrick Nedelec
 * @author     Paul E. Slauenwhite
 * @version    December 23, 2008
 * @since      June 6, 2005
 * @deprecated As of TPTP 4.5.0, use the TPTP Business Intelligence and Reporting Tools (BIRT) reporting infrastructure (<code>org.eclipse.tptp.platform.report.birt</code>).
 */
public class ReportDataUtils {

    public static String normalize(int i) {
        return NumberFormat.getInstance().format(i);
    }

    /**
     * Resolves the arbitrated verdict from the array of verdicts (pass, inconclusive, fail, error).
     * <p/>
     * 
     * @param results An array of verdicts (pass, inconclusive, fail, error).
     * @return The arbitrated verdict, or -1 if the array contains no values.
     * @see TPFVerdict
     */
    public static int getVerdict(int[] results) {
        
    	if (results[3] > 0) {
            return TPFVerdict.ERROR;
        }
        if (results[2] > 0) {
            return TPFVerdict.FAIL;
        }
        if (results[1] > 0) {
            return TPFVerdict.INCONCLUSIVE;
        }
        if (results[0] > 0) {
            return TPFVerdict.PASS;
        }
        
        return -1;
    }

    /**
     * Resolves all execution results within a window of time.
     * <p>
     * The window consists of a start date and time and an end date and time
     * (UTC).
     * <p>
     * For an execution result to be included in the window, its start date and
     * time as well as the completion date and time must fall within the window
     * of time.
     * <p>
     * 
     * @param executionResults
     *            The list of execution results to be evaluated.
     * @param startTime
     *            The start date and time in milliseconds (UTC).
     * @param endTime
     *            The end date and time in milliseconds (UTC).
     * @return The list of all execution results within a window of time.
     */
    public static List getExecutionResultsForWindow(List executionResults, long startTime, long endTime) {
        ArrayList executionResultsWithinWindow = new ArrayList();
        TPFExecutionResult currentExecutionResult = null;
        long currentTimeStamp = 0;
        List executionEvents = null;

        Iterator executionResultsIterator = executionResults.iterator();
        while (executionResultsIterator.hasNext()) {
            currentExecutionResult = ((TPFExecutionResult) (executionResultsIterator.next()));

            executionEvents = currentExecutionResult.getExecutionHistory().getExecutionEvents();
            if (executionEvents.size()>0) {
                currentTimeStamp = ((TPFExecutionEvent) executionEvents.get(0)).getTimestamp();
                if ((currentTimeStamp >= startTime) && (currentTimeStamp <= endTime)) {
                    currentTimeStamp = ((TPFExecutionEvent) (executionEvents.get(executionEvents.size() - 1))).getTimestamp();
                    if ((currentTimeStamp >= startTime) && (currentTimeStamp <= endTime)) {
                        executionResultsWithinWindow.add(currentExecutionResult);
                    }
                }
            }
        }

        return executionResultsWithinWindow;
    }

    /**
     * Recursively resolves the count of all test cases in the parameter root
     * test suite including any referenced test cases by the parameter root test
     * suite.
     * <p>
     * 
     * @param testSuite
     *            The root test suite.
     * @return Count of all test cases in or referenced by the parameter root
     *         test suite.
     */
    public static int resolveTestCaseCount(TPFTestSuite testSuite) {
        int testCaseCount = 0;
        Iterator interactionFragmentsIterator = null;

        Iterator referencedSuitesIterator = ReportDataUtils.getReferencedTestSuites(testSuite).iterator();
        while (referencedSuitesIterator.hasNext()) {
            interactionFragmentsIterator = ((TPFTestSuite) (referencedSuitesIterator.next())).getBehavior().getInteraction().getInteractionFragments().iterator();
            while (interactionFragmentsIterator.hasNext()) {
                testCaseCount += resolveExecutionOccurrences(((BVRInteractionFragment) (interactionFragmentsIterator.next())));
            }
        }

        return testCaseCount;
    }

    /**
     * Recursively resolves the count of all execution occurrences in the
     * parameter root interaction fragment including any referenced execution
     * occurrences by the parameter root interaction fragment.
     * <p>
     * Looping is handled by multiplying the number of iterations in the loop by
     * number of execution occurrences within the body of the loop.
     * <p>
     * 
     * @param interactionFragment
     *            The root interaction fragment.
     * @return Count of all execution occurrences in or referenced by the
     *         parameter root interaction fragment.
     */
    public static int resolveExecutionOccurrences(BVRInteractionFragment interactionFragment) {

        int executionOccurrences = 0;

        if (interactionFragment instanceof BVRExecutionOccurrence) {

            if (((BVRExecutionOccurrence) (interactionFragment)).getOtherBehavior().getResource() != null) {
                executionOccurrences++;
            }
        } else if (interactionFragment instanceof BVRCombinedFragment) {

            Iterator interactionOperandsIterator = ((BVRCombinedFragment) (interactionFragment)).getInteractionOperands().iterator();
            Iterator interactionFragmentsIterator = null;
            BVRInteractionOperand interactionOperand = null;

            while (interactionOperandsIterator.hasNext()) {

                interactionOperand = ((BVRInteractionOperand) (interactionOperandsIterator.next()));

                interactionFragmentsIterator = interactionOperand.getInteractionFragments().iterator();

                while (interactionFragmentsIterator.hasNext()) {
                    executionOccurrences += resolveExecutionOccurrences(((BVRInteractionFragment) (interactionFragmentsIterator.next())));
                }

                try {
                    executionOccurrences *= Integer.parseInt(interactionOperand.getInteractionConstraint().getConstraint());
                } catch (NumberFormatException n) {
                    //Ignore since the constraint is not a numeric.
                }
            }
        }

        return executionOccurrences;
    }

    /**
	 * Recursively adds all referenced test suites from and including the 
	 * parameter root test suite.
	 * <p>
	 * Referenced test suites are defined as invocations of external test suites.
	 * <p>
	 * The return list ({@link TPFTestSuite}) does not contain duplicate test suites.
	 * <p> 
	 * 
	 * @param rootTestSuite The root test suites.
	 * @param referencedTestSuites The list ({@link TPFTestSuite}) of referenced test suites.
	 */
	public static EList getReferencedTestSuites(TPFTestSuite rootTestSuite){
    	
		EList referencedTestSuites = new BasicEList();
		
        ExecutionUtil.addReferencedTestSuites(rootTestSuite, referencedTestSuites);
        
        return referencedTestSuites;
	}	

	/**
	 * Recursively adds all referenced test suites from and including the 
	 * parameter list of root test suites.
	 * <p>
	 * Referenced test suites are defined as invocations of external test suites.
	 * <p>
	 * The return list ({@link TPFTestSuite}) does not contain duplicate test suites.
	 * <p> 
	 * 
	 * @param rootTestSuites The list of root test suites.
	 * @param referencedTestSuites The list ({@link TPFTestSuite}) of referenced test suites.
	 */
	public static EList getReferencedTestSuites(EList rootTestSuites){
    	
		EList referencedTestSuites = new BasicEList();
		
        Iterator rootTestSuitesIterator = rootTestSuites.iterator();
        
        while (rootTestSuitesIterator.hasNext()) {
        	ExecutionUtil.addReferencedTestSuites(((TPFTestSuite)(rootTestSuitesIterator.next())), referencedTestSuites);
        }
        
        return referencedTestSuites;
	}	
	
    /**
     * <p>Recursively resolves all verdicts from the parameter execution history.
     * </p>
     * 
     * @param executionHistory
     *            The root execution history.
     * @return all <code>TPFVerdict</code> from every <code>TPFInvocationEvent</code>.
     */
    public static List findAllVerdictEvents(TPFExecutionHistory executionHistory) {

    	return findAllVerdicts(executionHistory.getExecutionEvents());
    }
    
    /**
     * Find the verdict of every <code>TPFInvocationEvent</code> from the specified list of <code>TPFExecutionEvent</code> and their children events recursively.
     * @param tpfExecutionEvents list of <code>TPFExecutionEvent</code> to search from.
     * @return all <code>TPFVerdict</code> from every <code>TPFInvocationEvent</code>.
     */
    public static List findAllVerdicts(List tpfExecutionEvents) {
    	
    	List verdicts = new ArrayList();
    	Iterator iterator = tpfExecutionEvents.iterator();
    	TPFExecutionEvent executionEvent;
    	while(iterator.hasNext())
    	{
    		try{
	    		executionEvent = (TPFExecutionEvent)iterator.next();
		    	if(executionEvent instanceof TPFInvocationEvent)
				{
		    		TPFExecutionResult invokedExecutionResult = ((TPFInvocationEvent)executionEvent).getInvokedExecutionResult();
		            // use the verdict directly from execution result of a invocation event.
		    		verdicts.add(invokedExecutionResult.getVerdict());			
				}
				
		    	// recursively find the verdicts of all invocation events from the children of each event.
		    	verdicts.addAll( findAllVerdicts(executionEvent.getChildren()));
	    	
    		} catch (ClassCastException c) {
                //Ignore since not an instance of TPFExecutionEvent.
            }    		
    	}
    	
    	return verdicts;
    }

    /**
     * Arbitrates a single verdict event from one or more verdict events.
     * <p>
     * Arbitration is done using the following inequality:
     * <p>
     * Error > Fail > Inconclusive > Pass
     * <p>
     * If no verdict events exist in the parameter list of execution events,
     * <code>null</code> is returned.
     * 
     * @param executionEvents
     *            A list of execution events that may contain one or more
     *            verdict events.
     * @return The single verdict event arbitrated from one or more verdict
     *         events, otherwise <code>null</code>.
     */
    public static TPFVerdictEvent arbitrateVerdictEvent(List executionEvents) {

        TPFVerdictEvent verdictEvent = null;
        TPFVerdictEvent currentVerdictEvent = null;
        Iterator executionEventsIterator = executionEvents.iterator();

        //Iterate all execution events:
        while (executionEventsIterator.hasNext()) {

            try {

                currentVerdictEvent = ((TPFVerdictEvent) (executionEventsIterator.next()));

                switch (currentVerdictEvent.getVerdict().getValue()) {

                    case TPFVerdict.ERROR:
                        return currentVerdictEvent;

                    case TPFVerdict.FAIL:

                        verdictEvent = currentVerdictEvent;

                        break;

                    case TPFVerdict.INCONCLUSIVE:

                        if ((verdictEvent == null) || ((verdictEvent != null) && (verdictEvent.getVerdict().getValue() == TPFVerdict.PASS))) {
                            verdictEvent = currentVerdictEvent;
                        }

                        break;

                    case TPFVerdict.PASS:

                        if (verdictEvent == null) {
                            verdictEvent = currentVerdictEvent;
                        }

                        break;
                }
            } catch (ClassCastException c) {
                //Ignore since not an instance of TPFVerdictEvent.
            }
        }

        return verdictEvent;
    }

}
