/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionEvent.java,v 1.12 2010/02/05 17:44:39 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.event;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.xml.type.XMLTypeFactory;
import org.eclipse.hyades.loaders.util.Guid;
import org.eclipse.hyades.test.common.util.XMLUtil;

/**
 * <p>Mirror of the {@link org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent}
 * for the execution environment with support for annotations.  An annotation can be 
 * a file or byte array.</p>
 * 
 * 
 * @author  Kris Kobylinski
 * @author  Marcelo Paternostro
 * @author  Ashish K. Mathur
 * @author  Paul E. Slauenwhite
 * @author  Bianca Xue Jiang
 * @version February 5, 2010
 * @since   October 7, 2005
 */
public class ExecutionEvent implements IEventConstants
{
	private String id;
	private String parentId;
	private String ownerId;
	private String ownerType;
	private long timestamp;
	private String text;
	private String name;
	private String eventType;
	private String sortBy;
	private int conflict = CONFLICT_NONE;
	private ArrayList properties = new ArrayList();
	private ArrayList annotations = new ArrayList();
	private ArrayList repositoryRecords = new ArrayList();

	/**
	 * Returns the id of this event.
	 * @return String
	 */
	public String getId()
	{
		return id;
	}

	/**
	 * Sets the id of this event.
	 * @param value
	 */
	public void setId(String value)
	{
		id = value;
	}

	/**
	 * Returns the ID of the parent event
	 * @return String
	 */
	public String getParentId() {
		return parentId;
	}

	/**
	 * Sets the ID of the parent event
	 * @param parentId
	 */
	public void setParentId(String parentId) {
		this.parentId = parentId;
	}

	/**
	 * Returns the owner id of this event.
	 * @return String
	 */
	public String getOwnerId()
	{
		return ownerId;
	}

	/**
	 * Sets the owner id of this event.
	 * @param value
	 */
	public void setOwnerId(String value)
	{
		ownerId = value;
	}

	/**
     * Returns the owner type of this event.
	 * @return the ownerType
	 */
	public String getOwnerType() {
		return ownerType;
	}

	/**
     * Sets the owner type of this event.
	 * @param ownerType the ownerType to set
	 */
	public void setOwnerType(String ownerType) {
		this.ownerType = ownerType;
	}
	
	/**
	 * Returns the timestamp of this event.
	 * @return String
	 */
	public long getTimestamp()
	{
		return timestamp;
	}

	/**
	 * Sets the timestamp of this event.  The timestamp
	 * is set by the {@link #toString()} method if this
	 * {@link #getTimestamp()} returns 0.
	 * @param value
	 */
	public void setTimestamp(long value)
	{
		timestamp = value;
	}

	/**
	 * Returns the text of this event.
	 * @return String
	 */
	public String getText()
	{
		return text;
	}

	/**
	 * Sets the text of this event.
	 * @return String
	 */
	public void setText(String value)
	{
		text = value;
	}

	/**
	 * @return Returns the eventType.
	 */
	public String getEventType() {
		return eventType;
	}
	/**
	 * This attribute is optional. If provided, the execution history viewer, will query the 
	 * extension point that will need to be provided, for the label of the tree element 
	 * for this event, in the execution history viewer.
	 * @param eventType The eventType to set.
	 */
	public void setEventType(String eventType) {
		this.eventType = eventType;
	}
	/**
	 * @return Returns the name.
	 */
	public String getName() {
		return name;
	}
	/**
	 * If this attribute is speicified, the execution history viewer will use it
	 * as the label of the tree element for this event. However, if the eventType is
	 * specified, the it will query the corresponding extension point for the label 
	 * text. 
	 * @param name The name to set.
	 */
	public void setName(String name) {
		this.name = name;
	}
	
	/**
	 * @return Returns the sortBy.
	 */
	public String getSortBy() {
		return sortBy;
	}
	
	/**
	 * Set this attribute if this event needs to be sorted within the scope of its
	 * immediate parent in the execution history. This attribute does not make it 
	 * all the way to the execution history model but is used by the loaders to 
	 * figure out sorting.
	 * @param sortBy The sortBy to set.
	 */
	public void setSortBy(String sortBy) {
		this.sortBy = sortBy;
	}

	/**
	 * @return Returns the conflict.
	 */
	public int getConflict() {
		return conflict;
	}
	
	/**
	 * This attribute indicates what should be done by the execution history loader
	 * if it finds an existing event with the same credentials (hierarchical id).
	 * Default behavior isto not worry about conflicts and just insert this event
	 * into the execution history
	 * 
	 * @param conflict The conflict to set.
	 */
	public void setConflict(int conflict) {
		this.conflict = conflict;
	}

	/**
	 * Returns the non-null list of properties ({@link EventProperty}).
	 * <p>
	 * 
	 * @return List of properties ({@link EventProperty}).
	 */
	public ArrayList getProperties() {
		return properties;
	}

	/**
	 * Adds the parameter property ({@link EventProperty}) to add to the list 
	 * of properties.
	 * <p>
	 * 
	 * @param eventProperty The property to be added to the list of properties.
	 */
	public void addProperty(EventProperty eventProperty){
		properties.add(eventProperty);
	}

	/**
	 * Clears the list of properties ({@link EventProperty}).
	 * <p>
	 */
	public void clearProperties(){
		properties.clear();
	}
	
	/**
	 * Returns the non-null list of annotations ({@link EventAnnotation}).
	 * <p>
	 * 
	 * @return List of annotations ({@link EventAnnotation}).
	 */
	public List getAnnotations(){
		return annotations;
	}

	/**
	 * Adds the parameter annotation ({@link EventAnnotation}) to add to the list 
	 * of annotations.
	 * <p>
	 * 
	 * @param eventAnnotation The annotation to be added to the list of annotations.
	 */
	public void addAnnotation(EventAnnotation eventAnnotation){
		annotations.add(eventAnnotation);
	}

	/**
	 * Clears the list of annotations ({@link EventAnnotation}).
	 * <p>
	 */
	public void clearAnnotations(){
		annotations.clear();
	}
	
	/**
	 * Returns the non-null list of repository records ({@link RepositoryRecord}).
	 * 
	 * @return List of repository records ({@link RepositoryRecord}).
	 * @provisional
	 */
	public ArrayList getRepositoryRecords() {
		return repositoryRecords;
	}

	/**
	 * Adds the specified record ({@link RepositoryRecord}) to add to the list 
	 * of repository records.
	 * 
	 * @param record The repository records to be added to the list of repository records.
	 * @provisional
	 */
	public void addRepositoryRecords(RepositoryRecord record){
		repositoryRecords.add(record);
	}

	/**
	 * Clears the list of repository records ({@link RepositoryRecord}).
	 * @provisional
	 */
	public void clearRepositoryRecords(){
		repositoryRecords.clear();
	}
	
	/**
	 * @see java.lang.Object#toString()
	 */
	public String toString()
	{
		StringBuffer sb = new StringBuffer();
		sb.append("<").append(getXMLRoot());
		addXMLAttributes(sb);
		sb.append(">");
		addXMLProperties(sb);
		addXMLAnnotations(sb);
		addXMLRepositoryRecords(sb);
		sb.append("</").append(getXMLRoot());
		sb.append(">");
		
		return sb.toString();
	}
	
	/**
	 * Returns the string that is used as the root of the
	 * XML fragment generated by the {@link #toString()} method.
	 * @return a not <code>null</code> String
	 */
	protected String getXMLRoot()
	{
		return "executionEvent";
	}
	
	/**
	 * Returns the string that is used as the tag for property
	 * XML fragment generated by the {@link #toString()} method.
	 * @return a not <code>null</code> String
	 */
	protected String getXMLPropertyRoot()
	{
		return "property";
	}
	
	/**
	 * Returns the string that is used as the tag for annotation
	 * XML fragment generated by the {@link #toString()} method.
	 * @return a not <code>null</code> String
	 */
	protected String getXMLAnnotationRoot()
	{
		return "annotation";
	}
	
	/**
	 * Returns the string that is used as the tag for repository record
	 * XML fragment generated by the {@link #toString()} method.
	 * @return a not <code>null</code> String
	 * @provisional
	 */
	protected String getXMLRepositoryRecordRoot()
	{
		return "repositoryRecord";
	}
	
	/**
	 * <p>Serializes the execution event's attributes to an XML fragment containing attributes
	 * and adds the XML fragment to the parameter <code>StringBuffer</code>.</p>
	 * 
	 * <p>This method is called by the {@link #toString()} method.</p>
	 * 
	 * @param xmlBuffer The XML <code>StringBuffer</code>.
	 */
	protected void addXMLAttributes(StringBuffer xmlBuffer){
		
		String id = getId();
		
		if ((id == null) || (id.length() == 0)){
			id = new Guid().toString();
		}
		
		xmlBuffer.append(XMLUtil.createXMLAttribute("id", id, false));
		xmlBuffer.append(XMLUtil.createXMLAttribute("ownerId", getOwnerId(), false));
		xmlBuffer.append(XMLUtil.createXMLAttribute("ownerType", getOwnerType(), false));
		xmlBuffer.append(XMLUtil.createXMLAttribute("text", getText(), false));
		
		if ((parentId != null) && (parentId.length() > 0)){
			xmlBuffer.append(XMLUtil.createXMLAttribute("parentId", getParentId(), false));
		}
		
		long timeStamp = getTimestamp();
		
		if(timeStamp == 0){
			timeStamp = System.currentTimeMillis();
		}
		
		xmlBuffer.append(XMLUtil.createXMLAttribute("timestamp", Long.toString(timeStamp), false));
		xmlBuffer.append(XMLUtil.createXMLAttribute("name", getName(), false));
		xmlBuffer.append(XMLUtil.createXMLAttribute("eventType", getEventType(), false));
		xmlBuffer.append(XMLUtil.createXMLAttribute("sortBy", getSortBy(), false));
		
		if (getConflict() != IEventConstants.CONFLICT_NONE){
			xmlBuffer.append(XMLUtil.createXMLAttribute("conflict", Integer.toString(getConflict()), false));			
		}
	}	
	
	/**
	 * <p>Serializes the execution event's properties to an XML fragment containing elements with attributes
	 * and adds the XML fragment to the parameter <code>StringBuffer</code>.</p>
	 * 
	 * <p>This method is called by the {@link #toString()} method.</p>
	 * 
	 * @param xmlBuffer The XML <code>StringBuffer</code>.
	 */
	protected void addXMLProperties(StringBuffer xmlBuffer) {
		
		EventProperty[] propertiesArray = ((EventProperty[])(properties.toArray(new EventProperty[properties.size()])));
		
		for (int counter = 0; counter < propertiesArray.length; counter++) {

			if(propertiesArray[counter] != null){

				xmlBuffer.append("<");
				xmlBuffer.append(getXMLPropertyRoot());
				xmlBuffer.append(XMLUtil.createXMLAttribute("pname", propertiesArray[counter].getName(), false));
				xmlBuffer.append(XMLUtil.createXMLAttribute("ptype", propertiesArray[counter].getType(), false));
				xmlBuffer.append(XMLUtil.createXMLAttribute("pvalue", propertiesArray[counter].getValue(), false));
				xmlBuffer.append("/>");

			}
		}
	}
	
	/**
	 * <p>Serializes the execution event's annotations to an XML fragment containing elements with attributes 
	 * and a child CDATA section and adds the XML fragment to the parameter <code>StringBuffer</code>.</p>
	 * 
	 * <p>This method is called by the {@link #toString()} method.</p>
	 * 
	 * @param xmlBuffer The XML <code>StringBuffer</code>.
	 */
	protected void addXMLAnnotations(StringBuffer xmlBuffer) {
		
		EventAnnotation[] annotationsArray = ((EventAnnotation[])(annotations.toArray(new EventAnnotation[annotations.size()])));
		
		for (int counter = 0; counter < annotationsArray.length; counter++) {

			if(annotationsArray[counter] != null){
				
				xmlBuffer.append("<");
				xmlBuffer.append(getXMLAnnotationRoot());
				xmlBuffer.append(XMLUtil.createXMLAttribute("annotationFilename", annotationsArray[counter].getFileName(), false));
				xmlBuffer.append(XMLUtil.createXMLAttribute("annotationType", annotationsArray[counter].getType(), false));
				xmlBuffer.append(">");			
	
				try {
					
					byte[] contents = annotationsArray[counter].getContents();
					
					if((contents != null) && (contents.length > 0)){
						
						xmlBuffer.append("<![CDATA[");
						xmlBuffer.append(XMLTypeFactory.eINSTANCE.convertBase64Binary(contents));
						xmlBuffer.append("]]>");			
					}				
				} 
				catch (IOException i) {				
					//Ignore since the annotation file does not exist, cannot be read, or is too large.
				}
	
				xmlBuffer.append("</");
				xmlBuffer.append(getXMLAnnotationRoot());
				xmlBuffer.append(">");
			}
		}
	}
	
	/**
	 * <p>Serializes the execution event's repository records to an XML fragment containing elements with attributes
	 * and adds the XML fragment to the parameter <code>StringBuffer</code>.</p>
	 * 
	 * <p>This method is called by the {@link #toString()} method.</p>
	 * 
	 * @param xmlBuffer The XML <code>StringBuffer</code>.
	 */
	protected void addXMLRepositoryRecords(StringBuffer xmlBuffer){

		RepositoryRecord[] repositoryRecordsArray = ((RepositoryRecord[])(repositoryRecords.toArray(new RepositoryRecord[repositoryRecords.size()])));
		
		for (int counter = 0; counter < repositoryRecordsArray.length; counter++) {

			if(repositoryRecordsArray[counter] != null){
				
				xmlBuffer.append("<");
				xmlBuffer.append(getXMLRepositoryRecordRoot());
				xmlBuffer.append(XMLUtil.createXMLAttribute("repositoryRecordId", repositoryRecordsArray[counter].getId(), false));
				xmlBuffer.append(XMLUtil.createXMLAttribute("repositoryRecordUri", repositoryRecordsArray[counter].getURI(), false));
				xmlBuffer.append(XMLUtil.createXMLAttribute("repositoryRecordType", repositoryRecordsArray[counter].getType(), false));
				xmlBuffer.append(XMLUtil.createXMLAttribute("repositoryRecordLabel", repositoryRecordsArray[counter].getLabel(), false));
				xmlBuffer.append("/>");
			}
		}
	}
}
