/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HyadesTestRunner.java,v 1.16 2007/12/05 20:02:55 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.junit;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EmptyStackException;
import java.util.Hashtable;
import java.util.Stack;
import java.util.Vector;

import junit.extensions.TestDecorator;
import junit.framework.AssertionFailedError;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestListener;
import junit.framework.TestResult;
import junit.framework.TestSuite;

import org.eclipse.hyades.internal.execution.remote.RemoteComponentSkeleton;
import org.eclipse.hyades.test.common.event.ExecutionEvent;
import org.eclipse.hyades.test.common.event.InvocationEvent;
import org.eclipse.hyades.test.common.event.JUnitInvocationEvent;
import org.eclipse.hyades.test.common.event.LoopEvent;
import org.eclipse.hyades.test.common.event.TypedEvent;
import org.eclipse.hyades.test.common.event.VerdictEvent;
import org.eclipse.hyades.test.common.runner.HyadesRunner;
import org.eclipse.hyades.test.common.util.BaseString;

/**
 * @author marcelop
 * @since 1.0.1
 */
public class HyadesTestRunner 
extends HyadesRunner implements TestListener
{
	protected final static boolean SYSOUT = false;
	
	private Test rootTest;
	/* bugzilla_140715: even if setTest() has not been invoked yet, we must have a temporary
	 * root object that ensures the condition rootObject!=null. This allow very early invokation
	 * of _seedParentStack to return non-null object in any case.
	 */
	private RootObject rootObject = new RootObject();
	
	private StringBuffer failureCauses;
	private StringBuffer errorCauses;
	
	private Collection succeedTests = new ArrayList();
	private Collection failureTests = new ArrayList();
	private Collection errorTests = new ArrayList();
	
	private static Hashtable threadIDtoParentStackTable = new Hashtable();
	private static Hashtable threadIDtoParentThreadTable = new Hashtable();
	
	/**
	 * Describes an element in the parent stack.
	 * @since 4.2
	 */
	protected static interface ParentObject {
		String getId();
	}
	
	/**
	 * Describes an invocation element in the parent stack.
	 * @since 4.2
	 */
	protected static class Invocation implements ParentObject {
		private String invocationId;
		private Collection succeedTests = Collections.synchronizedCollection(new ArrayList());
		private Collection failureTests = Collections.synchronizedCollection(new ArrayList());
		private Collection errorTests = Collections.synchronizedCollection(new ArrayList());
		public Invocation(String invocationId) {
			this.invocationId = invocationId; 
		}
		public void addSucceedTest(Test test) {
			succeedTests.add(test);
		}
		public void addFailureTest(Test test) {
			failureTests.add(test);
		}
		public void addErrorTest(Test test) {
			errorTests.add(test);
		}
		public Collection getErrorTests() {
			return errorTests;
		}
		public Collection getFailureTests() {
			return failureTests;
		}
		public Collection getSucceedTests() {
			return succeedTests;
		}
		public String getId() {
			return invocationId;
		}
	}
	
	/**
	 * Describes a loop in the parent stack.
	 * @since 4.2
	 */
	protected static class LoopObject implements ParentObject {
		private String loopId;
		public LoopObject(String loopId) {
			this.loopId = loopId;
		}
		public String getId() {
			return loopId;
		}
	}
	
	/**
	 * Describes the root of the parent stack.
	 * @since 4.2
	 */
	protected static class RootObject extends Invocation {
		public RootObject() {
			super(ExecutionEvent.ROOT_PARENT);
		}
	}
	
	/**
	 * Constructor for HyadesTestRunner
	 */
	public HyadesTestRunner()
	{
		super();
		initialize();
	}
	
	public HyadesTestRunner(RemoteComponentSkeleton agent) {
		super(agent);
		initialize();
	}
	
	public HyadesTestRunner(String args[])
	{
		this(args, null);
	}
	
	
	public HyadesTestRunner(String args[],  Vector commandHandler)
	{
		super(args, commandHandler);
		initialize();
	}
	
	private void initialize() {
		succeedTests = Collections.synchronizedCollection(new ArrayList());
		failureTests = Collections.synchronizedCollection(new ArrayList());
		errorTests = Collections.synchronizedCollection(new ArrayList());
		
		failureCauses = new StringBuffer();
		errorCauses = new StringBuffer();
		
	}
	public static void setParentThread(Thread child, Thread parent)
	{
		threadIDtoParentThreadTable.put(new Integer(child.hashCode()), 
			new Integer(parent.hashCode()));
	}
	
	public static Integer getParentThreadID(Thread t)
	{
		return (Integer) threadIDtoParentThreadTable.get(new Integer(t.hashCode()));
	}
	
	private static Invocation peekParentInvocation(Stack stack) {
		for (int i = stack.size(); i > 0; i--) {
			Object o = stack.elementAt(i-1);
			if (o instanceof Invocation) {
				return (Invocation)o;
			}
		}
		return null;
	}
	
	public static Integer getParentThreadID(Integer i)
	{
		return (Integer) threadIDtoParentThreadTable.get(i);
	}
	
	public static Stack getParentStack()
	{
		int myThreadID = Thread.currentThread().hashCode();
		return (Stack) threadIDtoParentStackTable.get(new Integer(myThreadID));
	}
	
	public static void setParentStack(Stack parentStack)
	{
		int myThreadID = Thread.currentThread().hashCode();
		threadIDtoParentStackTable.put(new Integer(myThreadID), parentStack);
	}
	
	/**
	 * @deprecated Use the non-static method {@link #_peekParentEventID()} instead.
	 */
	public static String peekParentEventID()
	{
		return peekParentEventID(new Integer(Thread.currentThread().hashCode()));
	}
	
	/**
	 * @since 4.2
	 */
	protected String _peekParentEventID() {
		return _peekParentEventID(new Integer(Thread.currentThread().hashCode()));
	}
	
	/**
	 * @since 4.2
	 */
	protected ParentObject peekParentObject()
	{
		return peekParentObject(new Integer(Thread.currentThread().hashCode()));
	}
	
	/**
	 * @since 4.2
	 */
	protected Invocation peekParentInvocation() {
		return peekParentInvocation(new Integer(Thread.currentThread().hashCode()));
	}
	
	/**
	 * @deprecated Use the non-static method {@link #_peekParentEventID(Integer)} instead.
	 */
	public static String peekParentEventID(Integer threadID)
	{
		return peekParentObject(threadID, null).getId();
	}
	
	/**
	 * @since 4.2
	 */
	protected String _peekParentEventID(Integer threadID) {
		return peekParentObject(threadID).getId();
	}
	
	/**
	 * @since 4.2
	 */
	protected ParentObject peekParentObject(Integer threadID) {
		return peekParentObject(threadID, this);
	}
	
	private static ParentObject peekParentObject(Integer threadID, HyadesTestRunner runner)
	{
		Stack parentIDStack = (Stack) threadIDtoParentStackTable.get(threadID);

		try {		
			if ( parentIDStack != null && !parentIDStack.isEmpty())
			{
				ParentObject parent =(ParentObject) parentIDStack.peek();
				if (parent != null )
				{
					return parent;
				}
			}
		}
		catch ( EmptyStackException e )
		{
		}
		
		// If we have no parent stack, or if it's empty, see if this thread
		// is a child of another thread that has a non-empty parent stack
		Integer parentThreadID = getParentThreadID(threadID);
		if ( parentThreadID == null )
		{
			return runner != null ? runner.rootObject : new RootObject();
		}
		
		// Tail recursion -- don't add code after this line without a 
		// really good reason!
		return peekParentObject(parentThreadID, runner);
	}
	
	/**
	 * @since 4.2
	 */
	protected Invocation peekParentInvocation(Integer threadID) {
		Stack parentIDStack = (Stack) threadIDtoParentStackTable.get(threadID);

		try {		
			if ( parentIDStack != null)
			{
				Invocation parent = peekParentInvocation(parentIDStack);
				if (parent != null )
				{
					return parent;
				}
			}
		}
		catch ( EmptyStackException e )
		{
		}
		
		// If we have no parent stack, or if it's empty, see if this thread
		// is a child of another thread that has a non-empty parent stack
		Integer parentThreadID = getParentThreadID(threadID);
		if ( parentThreadID == null )
		{
			return rootObject;
		}
		
		// Tail recursion -- don't add code after this line without a 
		// really good reason!
		return peekParentInvocation(parentThreadID);
	}
	
	/**
	 * @deprecated Use the non-static method {@link #_pushParentObject(org.eclipse.hyades.test.common.junit.HyadesTestRunner.ParentObject)} instead.
	 */
	public static void pushParentObject(ParentObject object) {
		__pushParentObject(object);
	}
	
	/**
	 * @since 4.2
	 */
	protected void _pushParentObject(ParentObject object) {
		__pushParentObject(object);
	}
	
	private static synchronized void __pushParentObject(ParentObject object)
	{
		Stack myParentIDStack = getParentStack();
		
		if ( myParentIDStack != null )
		{
			myParentIDStack.push(object);
		}
		else
		{
			myParentIDStack = new Stack();
			myParentIDStack.push(object);
			setParentStack(myParentIDStack);
		}
	}
	
	/**
	 * @deprecated Use the non-static method {@link #_popParentObject()} instead.
	 */
	public static ParentObject popParentObject() {
		return popParentObject(null);
	}
	
	/**
	 * @since 4.2
	 * @return
	 */
	protected ParentObject _popParentObject() {
		return popParentObject(this);
	}
	
	private static synchronized ParentObject popParentObject(HyadesTestRunner runner)
	{
		Stack myParentIDStack = getParentStack();
		
		try {		
			if ( myParentIDStack != null && !myParentIDStack.isEmpty())
			{
				ParentObject parent =(ParentObject) myParentIDStack.pop();
				if (parent == null)
				{
					return runner != null ? runner.rootObject : new RootObject(); 
				}
				else {
					return parent;
				}
			}
		}
		catch ( EmptyStackException e )
		{
		}

		return runner != null ? runner.rootObject : new RootObject();
	}

	/**
	 * @deprecated Use the non-static method {@link #_seedParentStack(Thread)} instead.
	 */
	public static synchronized void seedParentStack(Thread t)
	{
		// This method seeds an asynchronous thread's parentID stack with the
		// parent ID from the top of the invoking thread's stack before the 
		// invocation completes (and thus, before the invoking thread can 
		// pop off the correct parent.)
		Stack newParentIDStack = new Stack();
		newParentIDStack.push(peekParentObject(new Integer(Thread.currentThread().hashCode()), null));
		int newThreadID = t.hashCode();
		threadIDtoParentStackTable.put(new Integer(newThreadID), newParentIDStack);
	}
	
	/**
	 * @since 4.2
	 */
	public void _seedParentStack(Thread t)
	{
		// This method seeds an asynchronous thread's parentID stack with the
		// parent ID from the top of the invoking thread's stack before the 
		// invocation completes (and thus, before the invoking thread can 
		// pop off the correct parent.)
		Stack newParentIDStack = new Stack();
		newParentIDStack.push(peekParentObject());
		int newThreadID = t.hashCode();
		threadIDtoParentStackTable.put(new Integer(newThreadID), newParentIDStack);
	}
	
	/**
	 * Returns the root test, ie. the test that is passed to this
	 * runner.
	 * @return Test
	 */
	public Test getRoot()
	{
		return rootTest;
	}

	/**
	 * Sets the root test, ie. the test that is passed to this
	 * runner.
	 * @param root
	 */
	protected void setRoot(Test root)
	{
		rootTest = root;
		rootObject = new RootObject();
	}
	
	/**
	 * Returns a collection with the hirarchy ids of the test cases that have 
	 * been succeed.
	 * @return Collection
	 */
	public Collection getSucceedTests()
	{
		return succeedTests;
	}

	/**
	 * Returns a collection with the hirarchy ids of the test cases that have 
	 * ended because of a failure.
	 * @return Collection
	 */
	public Collection getFailureTests()
	{
		return failureTests;
	}

	/**
	 * Returns a collection with the hirarchy ids of the test cases that have
	 * ended because of an error.
	 * @return Collection
	 */
	public Collection getErrorTests()
	{
		return errorTests;
	}

	/**
	 * Runs a suite extracted from a TestCase subclass.
	 * @param testClass
	 */
	public void run(Class testClass)
	{
		run(new HyadesTestSuite(testClass));
	}
	
	/**
	 * Runs a single test and collects its results. This method can be used to 
	 * start a test run from your program.
	 * @param suite
	 */
	public void run(Test test)
	{
		setRoot(test);

		TestResult result= createTestResult();
		result.addListener(this);

		runnerStarted();
		
		long startTime= getCurrentTime();
		try
		{
			if (!(test instanceof IHyadesTest)) {
				// HyadesTestSuites have the special feature of generating a start and end
				// events for their listeners, while regular Test Suites do not. Since the test
				// won't inform us, tell ourselves that the root test was started.
				startTest(test);
			}
			test.run(result);
		}
		finally
		{
			if (!(test instanceof IHyadesTest)) {
				// See comment above.
				endTest(test);
			}
			long endTime= getCurrentTime();
			runnerExit((result == null || result.shouldStop()), (endTime-startTime));
		}		
	}

	/**
	 * Creates the TestResult to be used for the test run.
	 * @return TestResult
	 */
	protected TestResult createTestResult()
	{
		return new HyadesTestResult();
	}

	/**
	 * Returns the current timestamp.
	 * @return long
	 */
	protected long getCurrentTime()
	{
		return System.currentTimeMillis();
	}
	
	/**
	 * The runner has started. 
	 */
	protected void runnerStarted()
	{
		int count = HyadesTestUtil.countTests(new Test[]{getRoot()});
		if(SYSOUT)System.out.println("Run Started\n\tNumber of tests: " + count);
	}

	/**
	 * The runner is up to finish its activities. 
	 * @param runnerStopped
	 * @param the elapsed time
	 */
	protected void runnerExit(boolean runnerStopped, long elapsedTime)
	{
		if(SYSOUT)
		{
			if(runnerStopped)
				System.out.println("\nRun End");
			else
				System.out.println("\nRun Stop");
				
			System.out.println("\t" + HyadesTestUtil.getHierarchy(getRoot()) + "\n\t" + HyadesTestUtil.getHierarchyIds(getRoot()));  //$NON-NLS-1$//$NON-NLS-2$
			System.out.println("\n\nResults:" 
				+ "\n\tsucceedCount: " + getSucceedTests().size()
				+ "\n\tfailureCount: " + getFailureTests().size()
				+ "\n\terrorCount: " + getErrorTests().size());
		}

		writeEvent(getDefaultVerdictEvent(getRoot(), rootObject));

		TypedEvent typedEvent = new TypedEvent();
		typedEvent.setOwnerId(HyadesTestUtil.getHierarchyIds(getRoot()));
		typedEvent.setType(TypedEvent.STOP);
		typedEvent.setText(getLastEventText(runnerStopped, elapsedTime));
		typedEvent.setParentId(ExecutionEvent.ROOT_PARENT);
		writeEvent(typedEvent);
	}
	
	protected String getLastEventText(boolean runnerStopped, long elapsedTime)
	{
		return null;
	}
	
	/**
	 * @see junit.framework.TestListener#startTest(junit.framework.Test)
	 */
	public void startTest(Test test)
	{
		if(SYSOUT) System.out.println("\nStart\n\t" + HyadesTestUtil.getHierarchy(test) + "\n\t" + HyadesTestUtil.getHierarchyIds(test));
		
		TypedEvent typedEvent = new TypedEvent();
		typedEvent.setType(TypedEvent.START);

		if(test instanceof IHyadesTest)
		{
			IHyadesTest hyadesTest = (IHyadesTest)test;
			String invocationID = hyadesTest.getTestInvocationId();
			
			if(invocationID != null)
			{
				String hierarchyId = HyadesTestUtil.getHierarchyIds(hyadesTest.getParent());
				hierarchyId = HyadesTestUtil.appendToHierarchyId(hierarchyId, invocationID, hyadesTest.getIteration());
				
				InvocationEvent invocationEvent = new InvocationEvent();
				
				// Add the loop iterations to the ID to make each iteration a unique event ID
				invocationID = appendLoopIterationToID(invocationID, (IHyadesTest)hyadesTest.getParent());
				invocationEvent.setId(invocationID);
				invocationEvent.setParentId(_peekParentEventID());
				
				_pushParentObject(new Invocation(invocationID));
				
				invocationEvent.setOwnerId(hyadesTest.getTestInvocationId());
				invocationEvent.setStatus(InvocationEvent.STATUS_SUCCESSFUL);
				invocationEvent.setReason(InvocationEvent.REASON_NONE);
				invocationEvent.setInvokedId(HyadesTestUtil.getHierarchyIds(hyadesTest));
				writeEvent(invocationEvent);
				
				// The start event should also reference the test invocation
				typedEvent.setOwnerId(hyadesTest.getTestInvocationId());

			}
			else if ( hyadesTest.getParent() != null ){
				
				// If we don't have an invocationID, and we do have a parent test
				// (meaning we're not the outermost test), then we're a loop.

				String loopId = hyadesTest.getId();
				loopId = appendLoopIterationToID(loopId, (IHyadesTest)hyadesTest.getParent());
				
				if ( hyadesTest.getIteration() == 1 )
				{
					String parentID = _peekParentEventID();

					// Create a loop event
					LoopEvent loopEvent = new LoopEvent();
					loopEvent.setId(loopId);
					loopEvent.setAsychronous(!(hyadesTest.isSynchronous()));
					// loop id from test is also the id of the loop in the test suite.
					loopEvent.setOwnerId(hyadesTest.getId());
					loopEvent.setParentId(parentID);
					if (hyadesTest instanceof HyadesTestSuite)
						loopEvent.setIterations(((HyadesTestSuite) hyadesTest).getLoopCount());
					writeEvent(loopEvent);
					
					// Push the ID onto the parentstack
					// and make sure we remove it in in the endTest call.
					_pushParentObject(new LoopObject(loopId));
					
					// The start event should also reference the loop element in the test model.
					typedEvent.setOwnerId(hyadesTest.getId());
				}
				else if ( hyadesTest.getIteration() > 1 )
				{
					// indicate that any child events that come along before 
					// the corresponding stop event are children of this.
					_pushParentObject(new LoopObject(loopId));

					// The start event should also reference the loop element in the test model.
					typedEvent.setOwnerId(hyadesTest.getId());
				}
			}
		} else {
			// Not a Hyades Test
			if (test != getRoot()) {
				
				String invocationID = HyadesTestUtil.getTestId(test);
				JUnitInvocationEvent invocationEvent = new JUnitInvocationEvent();
				invocationEvent.setId(invocationID);
				invocationEvent.setParentId(_peekParentEventID());			
				_pushParentObject(new Invocation(invocationID));
				
				// No known ownerId here: the loaders will have to figure out themselves.
				// invocationEvent.setOwnerId(hyadesTest.getTestInvocationId());
				invocationEvent.setStatus(InvocationEvent.STATUS_SUCCESSFUL);
				invocationEvent.setReason(InvocationEvent.REASON_NONE);
				invocationEvent.setName(getTestName(test));
				invocationEvent.setClassName(test.getClass().getName());
				// No behavior model = no invocation id
				// invocationEvent.setInvokedId(HyadesTestUtil.getHierarchyIds(hyadesTest));
				writeEvent(invocationEvent);
				
				// Dito for typedEvent's ownerId
				// typedEvent.setOwnerId(hyadesTest.getTestInvocationId());
			}
		}

		typedEvent.setParentId(_peekParentEventID());	
		writeEvent(typedEvent);
	}
	
	protected static String getTestName(Test test) {
		if (test instanceof TestSuite) {
			return ((TestSuite)test).getName();
		} else if (test instanceof TestCase) {
			return ((TestCase)test).getName();
		} else if (test instanceof TestDecorator) {
			return getTestName(((TestDecorator)test).getTest());
		} else {
			return test.toString();
		}
	}

	/**
	 * @see junit.framework.TestListener#endTest(junit.framework.Test)
	 */
	public void endTest(Test test)
	{
		if(test == getRoot())
			return;

		ParentObject parent = _popParentObject();
		Invocation parentInvocation = (parent instanceof Invocation) ? (Invocation)parent : null;
		
		if(SYSOUT)
		{
			System.out.println("\nEnd\n\t" + HyadesTestUtil.getHierarchy(test) + "\n\t" + HyadesTestUtil.getHierarchyIds(test)); //$NON-NLS-2$
			if(parentInvocation != null && (test instanceof HyadesTestSuite) && (((HyadesTestSuite)test).getArbiter() != null))
			{
				System.out.println("\tResults:" 
					+ "\n\t\tsucceedCount: " + parentInvocation.getSucceedTests().size()
					+ "\n\t\tfailureCount: " + parentInvocation.getFailureTests().size()
					+ "\n\t\terrorCount: " + parentInvocation.getErrorTests().size());
			}
		}

		if (parentInvocation != null) {
			if((test instanceof TestCase) && (!parentInvocation.getFailureTests().contains(test)) && (!parentInvocation.getErrorTests().contains(test)))
				parentInvocation.addSucceedTest(test);

			VerdictEvent verdictEvent = getDefaultVerdictEvent(test, parentInvocation);
			if (!isRedundantVerdict(test, parentInvocation)) {
				writeEvent(verdictEvent);
			}
			
			getFailureTests().addAll(parentInvocation.getFailureTests());
			getErrorTests().addAll(parentInvocation.getErrorTests());
			getSucceedTests().addAll(parentInvocation.getSucceedTests());
			if (verdictEvent != null) {
				propagateVerdict(test, verdictEvent);
			}
		}

		TypedEvent typedEvent = new TypedEvent();
		if(test instanceof IHyadesTest)
		{
			IHyadesTest hyadesTest = (IHyadesTest)test;
			String invocationID = hyadesTest.getTestInvocationId();
			
			if(invocationID != null)
			{
				String hierarchyId = HyadesTestUtil.getHierarchyIds(hyadesTest.getParent());
				hierarchyId = HyadesTestUtil.appendToHierarchyId(hierarchyId, invocationID, hyadesTest.getIteration());
				
				typedEvent.setParentId(parent.getId());
				
				typedEvent.setOwnerId(invocationID);

			}
			else if ( hyadesTest.getParent() != null && hyadesTest.getIteration() > 0 )
			{
				typedEvent.setParentId(parent.getId());
				typedEvent.setOwnerId(hyadesTest.getId());
			}
		} else {
			// Not a Hyades Test
			typedEvent.setParentId(parent.getId());
		}
		typedEvent.setType(TypedEvent.STOP);
		writeEvent(typedEvent);
		
	}
	
	protected void propagateVerdict(Test test, VerdictEvent event) {
		Invocation parentInvocation = peekParentInvocation();
		if (event.getVerdict() == VerdictEvent.VERDICT_ERROR) {
			parentInvocation.addErrorTest(test);
		} else if (event.getVerdict() == VerdictEvent.VERDICT_FAIL) {
			parentInvocation.addFailureTest(test);
		} else if (event.getVerdict() == VerdictEvent.VERDICT_PASS) {
			parentInvocation.addSucceedTest(test);
		}
	}
	
	/**
	 * Returns the default verdict event, which is pass for test
	 * cases and the arbiter evaluation for test suites.
	 * @param test
	 * @return VerdictEvent
	 */
	protected VerdictEvent getDefaultVerdictEvent(Test test, Invocation parent)
	{
		VerdictEvent verdictEvent = null;
		if(test instanceof TestSuite)
		{
			if(test instanceof HyadesTestSuite)
			{
				HyadesTestSuite hyadesTestSuite = (HyadesTestSuite)test;
				if(hyadesTestSuite.getArbiter() != null)
				{
					verdictEvent = hyadesTestSuite.getArbiter().analyse(hyadesTestSuite, 
						parent.getSucceedTests(), parent.getErrorTests(), 
						parent.getFailureTests(), 
						errorCauses.toString(), failureCauses.toString());
						
					if(verdictEvent != null)
					{
						if(verdictEvent.getOwnerId() == null)
							verdictEvent.setOwnerId(hyadesTestSuite.getId());
					}
				}
			} else {
				// Make a synthetic verdict event
				verdictEvent = new VerdictEvent();
				if (!parent.getErrorTests().isEmpty()) {
					verdictEvent.setVerdict(VerdictEvent.VERDICT_ERROR);
				} else if (!parent.getFailureTests().isEmpty()) {
					verdictEvent.setVerdict(VerdictEvent.VERDICT_FAIL);
				} else if (!parent.getSucceedTests().isEmpty()) {
					verdictEvent.setVerdict(VerdictEvent.VERDICT_PASS);
				} else {
					verdictEvent.setVerdict(VerdictEvent.VERDICT_INCONCLUSIVE);
				}
			}
		}
		else if(test instanceof TestCase)
		{
			if (parent.getFailureTests().contains(test)) {
				// The individual failure has already been reported. Generating an
				// event here would just be redundant.
				verdictEvent = new VerdictEvent();
				verdictEvent.setVerdict(VerdictEvent.VERDICT_FAIL);
			} else if (parent.getErrorTests().contains(test)) {
				// The individual error has already been reported. Generating an
				// event here would just be redundant.
				verdictEvent = new VerdictEvent();
				verdictEvent.setVerdict(VerdictEvent.VERDICT_ERROR);
			} else {
				verdictEvent = new VerdictEvent();
				verdictEvent.setVerdict(VerdictEvent.VERDICT_PASS);
			}
			if (verdictEvent != null) {
				verdictEvent.setId(computeVerdictId(parent.getId()));
				if (test instanceof HyadesTestCase) {
					HyadesTestCase hyadesTestCase = (HyadesTestCase)test;
					verdictEvent.setOwnerId(hyadesTestCase.getId());
				}
			}
		}
		
		if (verdictEvent != null)
			verdictEvent.setParentId(parent.getId());		
		return verdictEvent;
	}

	protected boolean isRedundantVerdict(Test test, Invocation parent) {
		if (test instanceof TestSuite) {
			return false;
		}
		if (parent.getFailureTests().size() + parent.getErrorTests().size() == 1) {
			if (parent.getSucceedTests().size() == 0) {
				// No synthetic error or failure is needed as exactly one
				// test has been executed, and the individual failure/error
				// has already been reported.
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Returns the default verdict event, which is pass for test
	 * cases and the arbiter evaluation for test suites.
	 * @param test
	 * @return VerdictEvent
	 * @deprecated Use {@link #getDefaultVerdictEvent(Test, org.eclipse.hyades.test.common.junit.HyadesTestRunner.Invocation)} instead.
	 */
	protected VerdictEvent getDefaultVerdictEvent(Test test)
	{
		VerdictEvent verdictEvent = null;
		if(test instanceof TestSuite)
		{
			if(test instanceof HyadesTestSuite)
			{
				HyadesTestSuite hyadesTestSuite = (HyadesTestSuite)test;
				if(hyadesTestSuite.getArbiter() != null)
				{
					verdictEvent = hyadesTestSuite.getArbiter().analyse(hyadesTestSuite, 
						filterTests(test, getSucceedTests()), filterTests(test, getErrorTests()), 
						filterTests(test, getFailureTests()), 
						errorCauses.toString(), failureCauses.toString());
						
					if(verdictEvent != null)
					{
						if(verdictEvent.getOwnerId() == null)
							verdictEvent.setOwnerId(hyadesTestSuite.getId());
					}
				}
			} else if (test == getRoot()) {
				// Make a synthetic verdict event
				verdictEvent = new VerdictEvent();
				if (errorTests.size() > 0) {
					verdictEvent.setVerdict(VerdictEvent.VERDICT_ERROR);
				} else if (failureTests.size() > 0) {
					verdictEvent.setVerdict(VerdictEvent.VERDICT_FAIL);
				} else if (succeedTests.size() > 0) {
					verdictEvent.setVerdict(VerdictEvent.VERDICT_PASS);
				} else {
					verdictEvent.setVerdict(VerdictEvent.VERDICT_INCONCLUSIVE);
				}
			}
		}
		else if(test instanceof HyadesTestCase)
		{
			HyadesTestCase hyadesTestCase = (HyadesTestCase)test;
			if((!getFailureTests().contains(test)) && (!getErrorTests().contains(test)))
			{
				verdictEvent = new VerdictEvent();
				verdictEvent.setOwnerId(hyadesTestCase.getId());
				verdictEvent.setVerdict(VerdictEvent.VERDICT_PASS);
				verdictEvent.setId(computeVerdictId(peekParentEventID()));
			}
		} else {
			if (failureTests.contains(test)) {
				// The individual failure has already been reported. Generating an
				// event here would just be redundant.
				//verdictEvent = new VerdictEvent();
				//verdictEvent.setVerdict(VerdictEvent.VERDICT_FAIL);
			} else if (errorTests.contains(test)) {
				// The individual error has already been reported. Generating an
				// event here would just be redundant.
				//verdictEvent = new VerdictEvent();
				//verdictEvent.setVerdict(VerdictEvent.VERDICT_ERROR);
			} else {
				verdictEvent = new VerdictEvent();
				verdictEvent.setVerdict(VerdictEvent.VERDICT_PASS);
				verdictEvent.setId(computeVerdictId(peekParentEventID()));
			}
		}
		
		if (verdictEvent != null)
			verdictEvent.setParentId(peekParentEventID());		
		return verdictEvent;
	}

	/**
	 * @see junit.framework.TestListener#addError(junit.framework.Test, java.lang.Throwable)
	 */
	public void addError(Test test, Throwable throwable)
	{
		if(SYSOUT) System.out.println("\nError\n\t" + HyadesTestUtil.getHierarchy(test) + "\n\t" + HyadesTestUtil.getHierarchyIds(test)); //$NON-NLS-2$
		Invocation parentInvocation = peekParentInvocation();
		ParentObject parent = peekParentObject();

		VerdictEvent verdictEvent = new VerdictEvent();

		if ( test instanceof IHyadesTest )
		{
			String id = computeVerdictId(parent.getId());
			verdictEvent.setId(id);
			synchronized (errorCauses)
			{
				if (errorCauses.length() > 0)
					errorCauses.append(',');
				errorCauses.append(id);
			}						
		}
		
		parentInvocation.addErrorTest(test);

		verdictEvent.setParentId(parent.getId());
		verdictEvent.setVerdict(VerdictEvent.VERDICT_ERROR);
		verdictEvent.setReason(VerdictEvent.REASON_SEE_DESCRIPTION);
		verdictEvent.setText(BaseString.getStackTrace(throwable));
		writeEvent(verdictEvent);
	}

	/**
	 * @see junit.framework.TestListener#addFailure(junit.framework.Test, junit.framework.AssertionFailedError)
	 */
	public void addFailure(Test test, AssertionFailedError throwable)
	{
		if(SYSOUT) System.out.println("\nFailure\n\t" + HyadesTestUtil.getHierarchy(test) + "\n\t" + HyadesTestUtil.getHierarchyIds(test)); //$NON-NLS-2$
		Invocation parentInvocation = peekParentInvocation();
		ParentObject parent = peekParentObject();
		
		VerdictEvent verdictEvent = new VerdictEvent();

		if ( test instanceof IHyadesTest )
		{
			String id = computeVerdictId(parent.getId());
			verdictEvent.setId(id);
			synchronized (failureCauses)
			{
				if (failureCauses.length() > 0)
					failureCauses.append(',');
				failureCauses.append(id);
			}
		}

		parentInvocation.addFailureTest(test);

		verdictEvent.setParentId(parent.getId());
		verdictEvent.setVerdict(VerdictEvent.VERDICT_FAIL);
		verdictEvent.setReason(VerdictEvent.REASON_SEE_DESCRIPTION);
		verdictEvent.setText(BaseString.getStackTrace(throwable));
		writeEvent(verdictEvent);
	}

	/**
	 * Writes an event.
	 * @param executionEvent
	 */	
	protected void writeEvent(ExecutionEvent executionEvent)
	{
		if(executionEvent != null)
			System.out.println(executionEvent);
	}
	
	/**
	 * Filters and retains from a collection of tests, the tests are direct children
	 * of a parent test.
	 * @param parent
	 * @param tests
	 * @return
	 * @deprecated This method does not work properly when tests contains non-TPTP JUnit tests 
	 * (i.e. tests that do not implement {@link IHyadesTest}).
	 * Use {@link Invocation} instances from the stack {@link #getParentStack()} to determine
	 * which tests are direct children of a given parent.
	 */
	protected Collection filterTests(Test parent, Collection tests)
	{
		Test[] testArray = (Test[])tests.toArray(new Test[tests.size()]);
		Collection filteredTests = new ArrayList(testArray.length);
		for (int i = 0, maxi = testArray.length; i < maxi; i++)
		{
			if((testArray[i] instanceof IHyadesTest) && HyadesTestUtil.isParentOf(parent, testArray[i]))
				filteredTests.add(testArray[i]);
		}
		
		return Collections.unmodifiableCollection(filteredTests);
	}
	
	protected String computeVerdictId(String parentId)
	{
		return parentId + "_verdict"; //$NON-NLS-1$
	}
	
	protected String appendLoopIterationToID( String ID, IHyadesTest parentTest )
	{
		String iterations = HyadesTestSuite.getIterationsString(_peekParentEventID(), parentTest);
		return ID + iterations;
/*		
		// Walk the parent tests and compute the loop iteration for this ID.
		for ( IHyadesTest test = parentTest; test != null; test = (IHyadesTest) test.getParent() )
		{
			if ( test.isLoop() )
			{
				int iteration = HyadesTestSuite.getParentIteration(test);
				
				if ( iteration > 0 )
				{
					iterationString.append('(' + Integer.toString(iteration) + ')');
				}
				else
				{
					HyadesTestSuite.debug_out("[Hey!] Parent iteration == 0 for " + test + ".  CurrentThread = " + Thread.currentThread() + ", (Async[" + (Thread.currentThread() instanceof HyadesTestSuite.AsynchronousThread) + "]");
				}
			}
		}
		if ( iterationString.length() > 0 )
		{
			ID = ID + iterationString.toString();
		}
		
		return ID;
*/

	}
}
