/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HyadesTestUtil.java,v 1.4 2010/02/03 13:49:20 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.junit;

import java.lang.reflect.Field;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import junit.extensions.TestDecorator;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestListener;
import junit.framework.TestResult;
import junit.framework.TestSuite;

/**
 * <p>Utility methods for TPTP JUnit testsHyades JUnit extensions.
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul Slauenwhite
 * @version February 3, 2010
 * @since   January 27, 2005
 */
public class HyadesTestUtil
{
	/**
	 * @see #getHierarchy(Test, boolean)
	 */
	public static String getHierarchy(Test test)
	{
		return getHierarchy(test, true);
	}

	/**
	 * Returns a String with the hierarchy of the specified test.
	 * 
	 * <p>This method uses the extended API provided by IHyadesTest to 
	 * retrieve this test case's parent and the iteration information.
	 * 
	 * <p>The hierarchy has the following format: 
	 * parent<b>_</b>iteration<b>&gt;</b>child<b>_</b>iteration.
	 * 
	 * @param test The test to be analyzed
	 * @param iteration whether the iteration should be printed or not
	 * @return a not <code>null</code> String
	 */
	public static String getHierarchy(Test test, boolean iteration)
	{
		StringBuffer hierarchy = doGetHierarchy(test, iteration);
		if(hierarchy.length() == 0)
			return "";
		
		if(hierarchy.charAt(0) == '-')
			hierarchy.deleteCharAt(0);
		return hierarchy.toString();
	}

	/**
	 * Private implementation of {@link #getHierarchy(Test)}. 
	 * @param Test The test to be analyzed
	 * @param iteration whether the iteration should be printed or not
	 * @return a not <code>null</code> StringBuffer
	 */
	private static StringBuffer doGetHierarchy(Test test, boolean iteration)
	{
		if(test != null)
		{
			if(test instanceof IHyadesTest)
			{
				IHyadesTest hyadesTest = (IHyadesTest)test;
				StringBuffer hierarchy = doGetHierarchy(hyadesTest.getParent(), iteration);
				
				hierarchy.append("-").append(getNotNullString(hyadesTest.getName()));
				if(iteration && (hyadesTest.getIteration() > 0))
					hierarchy.append("[").append(hyadesTest.getIteration()).append("]");
				return hierarchy;
			}
			
			return new StringBuffer("-?").append(test.toString());
		}

		return new StringBuffer(0);
	}
		
	/**
	 * Returns a String with the id's of the hierarchy of the specified test.  The
	 * test invocation ids are considered during the String creation.
	 * 
	 * <p>This method uses the extended API provided by IHyadesTest to 
	 * retrieve this test case's parent and the iteration information.
	 * 
	 * @param Test The test to be analyzed
	 * @return a not <code>null</code> String
	 */
	public static String getHierarchyIds(Test test)
	{
		StringBuffer hierarchy = doGetHierarchyIds(test);
		if(hierarchy.length() == 0)
			return "";
			
		if(hierarchy.charAt(0) == '.')
			hierarchy.deleteCharAt(0);
		return hierarchy.toString();
	}

	/**
	 * Private implementation of {@link #getHierarchyIds(Test)}. 
	 * @param Test The test to be analyzed
	 * @return a not <code>null</code> StringBuffer
	 */
	private static StringBuffer doGetHierarchyIds(Test test)
	{
		if(test != null)
		{
			if(test instanceof IHyadesTest)
			{
				IHyadesTest hyadesTest = (IHyadesTest)test;
				StringBuffer hierarchy = doGetHierarchyIds(hyadesTest.getParent());
				
				int iteration = hyadesTest.getIteration();
				if(hyadesTest.getTestInvocationId() != null)
					appendToHierarchyId(hierarchy, hyadesTest.getTestInvocationId(), iteration);

				return appendToHierarchyId(hierarchy, hyadesTest.getId(), iteration);
			}
			
			return new StringBuffer(".?").append(test.toString());
		}

		return new StringBuffer(0);
	}
	
	/**
	 * Appends a new segment to an existing hierarchy of Id.  The iteration argument
	 * is not considered if it is lower than 1.
	 * @param hierarchy
	 * @param segment
	 * @param iteration
	 * @return String
	 */
	public static String appendToHierarchyId(String hierarchy, String segment, int iteration)
	{
		StringBuffer sb = null;
		if(hierarchy != null)
			sb = new StringBuffer(hierarchy);
			
		return appendToHierarchyId(sb, segment, iteration).toString();
	}

	/**
	 * StringBuffer version of {@link #appendToHierarchyId(String, String, int)}.  The
	 * hierarchy argument is modified inside the method.
	 * @param hierarchy
	 * @param segment
	 * @param iteration
	 * @return StringBuffer
	 */
	public static StringBuffer appendToHierarchyId(StringBuffer hierarchy, String segment, int iteration)
	{
		if(hierarchy == null)
		{
			if(segment == null)
				return null;
			hierarchy = new StringBuffer();
		}
		
		if(segment != null)
		{
			if(hierarchy.length() > 0)
				hierarchy.append(".");
			hierarchy.append(segment);
		}
		
		if(iteration >= 1)
			hierarchy.append("(").append(iteration).append(")");
		
		return hierarchy;
	}

	/**
	 * Counts the number of tests and child tests provided in
	 * the specified array.
	 * @param tests
	 * @return int
	 */
	public static int countTests(Test[] tests)
	{
		int count = 0;
		for(int i = 0; i < tests.length; i++)
			count = count + tests[i].countTestCases();

		return count;
	}
	
	/**
	 * Returns a not <code>null</code> String.
	 * @param object
	 * @return the object <code>toString()</code> method value if the object 
	 * is not <code>null</code> or an empty string otherwise.
	 */
	public static String getNotNullString(Object object)
	{
		if(object == null)
			return "";
		return object.toString();
	}
	
	/**
	 * Returns the Object value of a given object's field - or, using OO words, 
	 * attribute.  If the field is not available in the object's class, this
	 * method will look for it in the superclass hierarchy.
	 * 
	 * @param object
	 * @param fileName
	 * @return Object
	 * @throws Exception
	 * 
	 * @see Class#getDeclaredField(java.lang.String)
	 * @see Field#get(java.lang.Object)
	 * @deprecated As of TPTP 4.7.0, this method is unsupported.
	 */
	public static Object getObjectFieldValue(Object object, String fieldName)
	throws Exception
	{
		if((object == null) || (fieldName == null))
			return null;
		
		Field field = null;
		Exception exception = null;
		Class currentClass = object.getClass();
		while((field == null) && (currentClass != null))
		{
			try
			{
				field = currentClass.getDeclaredField(fieldName);
			}
			catch(Exception e)
			{
				if(exception == null)
					exception = e;

				currentClass = currentClass.getSuperclass();
			}			
		}
		if(field == null)
		{
			if(exception != null)
				throw exception;
				
			return null;
		}

		field.setAccessible(true);
		return field.get(object);
	}

	/**
	 * <p>Resolve an iterator over the list of registered test listeners in the test result.</p>
	 * 
	 * <p>If the list of registered test listeners cannot be resolved from the test result, an 
	 * iterator over an empty list is returned.</p>
	 * 
	 * @param testResult The test result with the list of registered test listeners.
	 * @return An iterator over the list of registered test listeners in the test result.
	 */
	public static Iterator<TestListener> getTestListeners(TestResult testResult){
		
		if(testResult instanceof HyadesTestResult){
			return (((HyadesTestResult)(testResult)).getTestListeners().iterator());
		}
		else{

			try{
				return ((List)(getObjectFieldValue(testResult, "fListeners"))).iterator(); //$NON-NLS-1$
			}
			catch(Exception e){
				//Ignore and return an empty list iterator.
			}
		}
		
		return (Collections.EMPTY_LIST.iterator());
	}
	
	/**
	 * Returns whether the parent candidate test is a parent of the child test. 
	 * @param parentCandidate
	 * @param child
	 * @return boolean
	 */
	public static boolean isParentOf(Test parentCandidate, Test child)
	{
		if((parentCandidate != null) && (child != null))
		{
			if(child instanceof IHyadesTest)
			{
				Test parent = ((IHyadesTest)child).getParent();
				if(parentCandidate.equals(parent))
					return true;
					
				return isParentOf(parentCandidate, parent);
			}
		}
		
		return false;
	}
	
	/**
	 * Returns the number of test cases in the given suite
	 * @return int
	 */
	public static int getTestCaseCount(TestSuite testSuite)
	{
		int count = 0;
		for(Enumeration e = testSuite.tests(); e.hasMoreElements();)
		{
			Test test = (Test)e.nextElement();
			if(test instanceof TestDecorator)
				test = getDecoratorTest((TestDecorator)test);
				
			if(test instanceof TestCase)
				count++;
		}
		return count;
	}	
	
	/**
	 * Recursivelly returns the test of a given decorator
	 * @param testDecorator
	 * @return a non decorator test
	 */
	public static Test getDecoratorTest(TestDecorator testDecorator)
	{ 
		if(testDecorator == null)
			return null;
			
		Test test = testDecorator.getTest();
		if(test instanceof TestDecorator)
			return getDecoratorTest((TestDecorator)test);
			
		return test;
	}
	
	private static Map testToTestIdMap;
	private static int testIdsCounter = 0;
	
	/**
	 * Returns a unique Id for a non-Hyades Test
	 * @param test
	 * @return
	 */
	public static String getTestId(Test test) {
		if (testToTestIdMap == null) {
			testToTestIdMap = new HashMap();
		}
		String id = (String)testToTestIdMap.get(test);
		if (id == null) {
			id = Integer.toString(++testIdsCounter);
			testToTestIdMap.put(test, id);
		}
		return id;
	}
	
}