/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HyadesRunner.java,v 1.15 2008/04/09 19:58:02 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.common.runner;

import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.StringTokenizer;
import java.util.Vector;

import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.execution.core.IControlMessage;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.remote.AgentControllerUnavailableException;
import org.eclipse.hyades.internal.execution.remote.CustomCommandHandler;
import org.eclipse.hyades.internal.execution.remote.RemoteComponentSkeleton;
import org.eclipse.hyades.models.common.datapool.util.DPLPasswordCollection;
import org.eclipse.hyades.models.common.facades.behavioral.impl.FacadeResourceFactoryImpl;
import org.eclipse.hyades.models.common.testprofile.impl.Common_TestprofilePackageImpl;
import org.eclipse.hyades.test.common.agent.ComptestAgent;
import org.eclipse.hyades.test.common.agent.PrimaryTestAgent;
import org.eclipse.hyades.test.common.agent.ServiceCommandHandler;
import org.eclipse.hyades.test.common.agent.UnknownTestServiceException;
import org.eclipse.hyades.test.common.event.ExecutionEvent;
import org.eclipse.hyades.test.common.testservices.resources.DatapoolPasswordProvider;

/**
 * Hyades runner.
 * 
 * 
 * @author  Judy Sutton
 * @author  Paul E. Slauenwhite
 * @version April 3, 2008
 * @since   January 27, 2005
 */
public class HyadesRunner {

	private boolean isOKToStart = false;
	protected ComptestAgent comptestAgent;
	protected RemoteComponentSkeleton agent;
	
	// bugzilla 84790: runner system properties
	protected final String TEST_CLASS = "hyades.runner.testClass";
	protected final String TEST_ID = "hyades.runner.testID";
	protected final String AGENT_NAME = "hyades.runner.agentName";
	protected String testClass = null;
	protected String testID = null;
	protected String agentName = null;

	// bugzilla 84790: original method of positional arguments (to maintain backward compatibility)
	// args array index for testClass and testID
	protected final int TEST_CLASS_INDEX = 0;
	protected final int TEST_ID_INDEX = 1;
	
	public HashMap agentCollection = new HashMap();

	public class Commandhandler implements CustomCommandHandler 
	{
		public void handleCommand(CustomCommand command) 
		{
			String cmd = command.getData();
			
			if (cmd.equals(IControlMessage.START)) 
			{
				setOKToStart(true);
			} 
			else if(cmd.startsWith("AgentIDs:"))
			{
				/* Bug 83955 begins */
				String dpString;
				try {
					/*
					 * Try to reconstruct the string from bytes encoded in UTF-8 since all messages
					 * coming from the RAC are now encoded in UTF-8 to support DBCS systems.
					 */
					dpString = new String(command.getDataBinary(), "UTF-8");
				} catch(UnsupportedEncodingException e) {
					/*
					 * If UTF-8 is not supported, try to use default encoding. 
					 */
					dpString = new String(command.getDataBinary());
				}
				/* Bug 83955 ends */
				dpString = dpString.substring(dpString.indexOf(':')+1,dpString.length());
				StringTokenizer dpTokenizer = new StringTokenizer(dpString,",");
				while(dpTokenizer.hasMoreTokens())
				{
					String id = dpTokenizer.nextToken();
					RemoteComponentSkeleton agent = new RemoteComponentSkeleton("Executor$" + id, "tester");
					agent.addCommandListener
					(
						new CustomCommandHandler()
						{
							public void handleCommand(CustomCommand command) 
							{
								command.getData();
							}
						}
					);
					try
					{
						agent.initialize();
						agentCollection.put(id,agent);
					}
					catch (AgentControllerUnavailableException e)
					{
						e.printStackTrace();
					}
				
				}
			}
			else 
			{
					handleCustomCommand(command);
			}
		}
	}

	/**
	 * Is it okay to continue starting the test?
	 * @return <code>true</code> if it is okay to start the test,
	 * <code>false</code> otherwise
	 */
	public synchronized boolean isOKToStart()
	{
		return isOKToStart;
	}

	// Set the "okay to start" flag and notify any threads that are waiting
	// on this object's monitor.
	protected synchronized void setOKToStart(boolean isOK)
	{
		isOKToStart = isOK;
		notifyAll();
	}


	public HyadesRunner() {
		super();
	}
	
	public HyadesRunner(String args[]) {
		this (args, null);
	}
	
	public HyadesRunner(RemoteComponentSkeleton agent) {
		this.agent = agent;
		comptestAgent = new ComptestAgent(agent);
	}
	
	/**
	 * Use this constructor to add command handlers to the agent that is to be created
	 * <br/>
	 * Pre-condition:
	 * <ul>
	 * 	<li> The items of commandHandler must be of type CustomCommandHandler </li>
	 * </ul>
	 * 
	 * @param commandListener - A list of command handlers
	 */
	public HyadesRunner (String args[], Vector commandHandler) 
	{
		super();
		
		// waitForDebugger(true);
		
		// bugzilla 84790: set runner properties
		setRunnerProperties(args);
		
		//Initialize EMF models for datapools
		Resource.Factory.Registry.INSTANCE.getExtensionToFactoryMap().put("datapool", new FacadeResourceFactoryImpl()); 
	    Common_TestprofilePackageImpl.init();		
					
		/* Create our agent */
		agent = new RemoteComponentSkeleton("Executor$" + this.agentName, "tester");
		
		comptestAgent = new ComptestAgent(agent);
		PrimaryTestAgent.setComptestAgent(comptestAgent);
		
		// Add a command listener that can okay the starting of the test.
		agent.addCommandListener(new Commandhandler());
		
		// Add a command listener to handle return messages from test service invocations.
		agent.addCommandListener(new ServiceCommandHandler());
		
		// Add the custom command listeners that the client may want to add
		if (commandHandler != null)
		{
			final int size = commandHandler.size();
			for (int i = 0; i < size; i++)
			{
				Object currentHandler = commandHandler.get(i);
				if (currentHandler instanceof CustomCommandHandler)
				{
					agent.addCommandListener((CustomCommandHandler) currentHandler);
				}
			}
		}

		
		/* Handshake with the Agent controller */
		try
		{
			agent.initialize();
			Runtime.getRuntime().addShutdownHook(disposeOnExitHook);
		}
		catch (Throwable e)
		{
		}
		
		/* Wait for our start.  We will never wait more that 3 minutes */
		while (!isOKToStart())
		{
			synchronized (this)
			{
				try
				{
					wait(3 * 60000);
					if (!isOKToStart())
					{
						/* Try and tell the client we are timing out */
						agent.sendMessageToAttachedClient("Test was not started after 3 minutes.  Exiting testcase.", 0);

						/* We have timed out.  exit */
						System.exit(-1);

					}
				}
				catch (InterruptedException e)
				{

				}
			}
		}
		//set password to Models
		try {
			DPLPasswordCollection.getInstance().
			setDatapoolPassword(DatapoolPasswordProvider.getDatapoolPassword(testID));
		} catch (UnknownTestServiceException e) {			
		}

	}
	
	private Thread disposeOnExitHook = new Thread() {
		public void run() {
			if (agent != null && agent.isRegistered()) {
				agent.deregister();
			}
		}
	};
	
	public void dispose() {
		//clear the passwords in Models
		DPLPasswordCollection.getInstance().clear();
		if (agent != null && agent.isRegistered()) {
			Runtime.getRuntime().removeShutdownHook(disposeOnExitHook);
			agent.deregister();
		}
	}
	
	public void handleCustomCommand(CustomCommand command) {}
	
	public void writeExecEvent(ExecutionEvent executionEvent)
	{
		if((comptestAgent != null) && (executionEvent != null))
			comptestAgent.write(executionEvent.toString());
						
	}

	public void writeExecEvent(String executionEventStr)
	{
		if((comptestAgent != null) && (executionEventStr != null))
			comptestAgent.write(executionEventStr);
	}
	/**
	 * @param key
	 * @return
	 */
	public RemoteComponentSkeleton getAgent(Object key)
	{
		if(key==null)
			return agent;
		return (RemoteComponentSkeleton)agentCollection.get(key);
	}
	
	// bugzilla 84790: sets runner properties
	protected void setRunnerProperties(String args[])
	{ 
		testClass = null;
		testID = null;
		agentName = null;

		// assume if system property TEST_CLASS is found, named paramters are used
		testClass = System.getProperty(TEST_CLASS);
		if (testClass != null)
		{
			testID = System.getProperty(TEST_ID);
			agentName = System.getProperty(AGENT_NAME);
		}
		else
		{
			// assume positional arguments are used
			if (TEST_CLASS_INDEX < args.length)
				testClass = args[TEST_CLASS_INDEX];
			if (TEST_ID_INDEX < args.length)
				testID = args[TEST_ID_INDEX];
			
			// in this case, agentName is always assumed to be the last parameter
			agentName = args[args.length-1];
		}
	}

}
