/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestCase.java,v 1.1 2007/12/05 20:02:58 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.runner.model;

import java.util.Arrays;

/**
 * Test Case class.
 * <p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @author  Marcelo Paternostro
 * @version April 12, 2007
 * @since   October 7, 2005
 * @since   1.0.2
 */
public class TestCase extends Test {
	
	private TestSuite testSuite;
	private String descriptionAsHTML = null;

	//HTML V4 elements excluding 'html', 'head', 'title' and 'body' [source: http://www.w3.org/TR/html4/index/elements.html]:
	private static final String[] SUPPORTED_HTML_ELEMENTS = new String[]{"!--", "a", "abbr", "acronym", "address", "applet", "area", "b", "base", "basefont", "bdo", "big", "blockquote", "br", "button", "caption", "center", "cite", "code", "col", "colgroup", "dd", "del", "dfn", "dir", "div", "dl", "dt", "em", "fieldset", "font", "form", "frame", "frameset", "h1", "h2", "h3", "h4", "h5", "h6", "hr", "i", "iframe", "img", "input", "ins", "isindex", "kbd", "label", "legend", "li", "link", "map", "menu", "meta", "noframes", "noscript", "object", "ol", "optgroup", "option", "p", "param", "pre", "q", "s", "samp", "script", "select", "small", "span", "strike", "strong", "style", "sub", "sup", "table", "tbody", "td", "textarea", "tfoot", "th", "thead", "tr", "tt", "u", "ul", "var"};

	public void dispose()
	{
		testSuite = null;
		super.dispose();
	}
	
	public TestSuite getTestSuite()
	{
		return testSuite;
	}

	public void setTestSuite(TestSuite newTestSuite)
	{
		if(testSuite == newTestSuite)
			return;
		
		if(testSuite != null)
			testSuite.getTestCases().remove(this);
			
		this.testSuite = newTestSuite;
		
		if(!testSuite.getTestCases().contains(this))
			testSuite.getTestCases().add(this);
	}
	
	public String getDescriptionAsHTML(){		
		return descriptionAsHTML;
	}
	
	public void setDescription(String description){		
		
		super.setDescription(description);
		
		if(description != null){
			
			if(description.trim().toLowerCase().matches("<\\s*html[\\s\\S]*<\\s*/\\s*html\\s*>")){
				descriptionAsHTML = description;
			}
			else{
				descriptionAsHTML = ("<html><body><pre>".concat(normalize(description))).concat("</pre></body></html>");
			}
		}
		else{
			descriptionAsHTML = null;
		}
	}
	
	/**
     * Normalizes the following characters in parameter non-null string:
     * <p>
     * <ul>
     * 	<li>&lt;</li>
     * 	<li>&gt;</li>
     * 	<li>&quot;</li>
     * </ul>
     * <p>
     * The above characters are normalized with the following entity reference 
     * values represented in hexadecimal:
     * <p>
     * <ul>
     * 	<li>&amp;lt;</li>
     * 	<li>&amp;gt;</li>
     * 	<li>&amp;quot;</li>
     * </ul>
     * <p>
     * <a href="http://www.w3.org/TR/html4/index/elements.html">HTML V4 elements</a> 
     * (excluding 'html', 'head', 'title' and 'body') are not normalized.  
     * <p>
     * 
     * @param string The non-null string to be normalized.
     * @return The normalized string, "null" or an empty string.
     */
    private String normalize(String string) {

        int stringLength = string.length();

        //Return an empty string if the string is empty:
        if (stringLength == 0) { 
            return (""); 
        }

        int marker = 0;
        int counter = 0;
        char[] characters = string.toCharArray();
        StringBuffer normalizedString = new StringBuffer(stringLength);
    	boolean normalize = true;

        //Check if any characters require normalization or replacement of invalid characters:
        while (counter < stringLength) {

        	//0x003C:
			if (characters[counter] == '<') {

				int offset = (counter + 1);
				int count = 0;
				
				for (int index = (counter + 1); index < characters.length; index++) {
					
					if((characters[index] == '/') || (characters[index] == ' ') || (characters[index] == '\t') || (characters[index] == '\n') || (characters[index] == '\r') || (characters[index] == '\f')) {
						
						if(count == 0){
							offset = (index + 1);
						}
						else{
							break;
						}
					}
					else if (characters[index] == '>') {
						break;
					}
					else{
						count++;
					}
				}
				
				if((count > 0) && (Arrays.binarySearch(SUPPORTED_HTML_ELEMENTS, new String(characters, offset, count).toLowerCase().trim()) >= 0)){
					normalize = false;
				} 
				else {

					normalizedString.append(characters, marker, (counter - marker));
					marker = (counter + 1);

					normalizedString.append("&lt;");

					normalize = true;
				}
			}

			//0x003E:
			else if (characters[counter] == '>') {

				if (normalize) {

					normalizedString.append(characters, marker, (counter - marker));
					marker = (counter + 1);

					normalizedString.append("&gt;");
				} 
				else {
					normalize = true;
				}
			}

			//0x0022:
			else if (characters[counter] == '"') {

				if (normalize) {

					normalizedString.append(characters, marker, (counter - marker));
					marker = (counter + 1);

					normalizedString.append("&quot;");
				}
			}
        	
	        counter++;
        }
        		 
        if(marker == 0){
        	return string;
        }
        
        if(marker < counter){
        	normalizedString.append(characters, marker, (counter - marker));
        }
        		
        return (normalizedString.toString());
    }
}