/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EventLogger.java,v 1.1 2007/12/05 20:02:56 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.runner.model.util;

import java.util.List;

import org.eclipse.hyades.test.common.event.ExecutionEvent;
import org.eclipse.hyades.test.common.event.IEventConstants;
import org.eclipse.hyades.test.common.event.InvocationEvent;
import org.eclipse.hyades.test.common.event.LoopEvent;
import org.eclipse.hyades.test.common.event.MessageEvent;
import org.eclipse.hyades.test.common.event.TypedEvent;
import org.eclipse.hyades.test.common.event.VerdictEvent;
import org.eclipse.hyades.test.common.runner.model.Loop;
import org.eclipse.hyades.test.common.runner.model.NamedElement;
import org.eclipse.hyades.test.common.runner.model.TestInvocation;

/**
 * Test event logger.
 * <p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @version May 2, 2007
 * @since   September 29, 2005
 */
public abstract class EventLogger{

	abstract public void log(ExecutionEvent executionEvent);

	public MessageEvent logMessage(NamedElement namedElement, int severity, String text){
		return (logMessage(namedElement, severity, text, null));
	}

	public MessageEvent logMessage(NamedElement namedElement, int severity, String text, List eventAnnotations){

		if(namedElement != null){
			
			MessageEvent messageEvent = new MessageEvent();
			messageEvent.setOwnerId(namedElement.getId());
			messageEvent.setParentId(ModelUtil.getParentId(namedElement));
			messageEvent.setSeverity(severity);
			messageEvent.setText(text);
			
			if((eventAnnotations != null) && (eventAnnotations.size() > 0)){
				messageEvent.getAnnotations().addAll(eventAnnotations);
			}
			
			log(messageEvent);
			
			return messageEvent;
		}
		
		return null;
	}

	public TypedEvent logTyped(NamedElement namedElement, int type){
		return (logTyped(namedElement, type, null));
	}
	
	public TypedEvent logTyped(NamedElement namedElement, int type, String text){
		return (logTyped(namedElement, type, text, null));
	}

	public TypedEvent logTyped(NamedElement namedElement, int type, String text, List eventAnnotations){
		
		if(namedElement != null){
				
			TypedEvent typedEvent = new TypedEvent();
			typedEvent.setOwnerId(namedElement.getId());
			typedEvent.setParentId(ModelUtil.getParentId(namedElement));
			typedEvent.setType(type);

			if((text != null) && (text.trim().length() > 0)){
				typedEvent.setText(text);
			}
			
			if((eventAnnotations != null) && (eventAnnotations.size() > 0)){
				typedEvent.getAnnotations().addAll(eventAnnotations);
			}
			
			log(typedEvent);
			
			return typedEvent;
		}
		
		return null;
	}
	
	public InvocationEvent logInvocation(TestInvocation testInvocation, int status){
		return (logInvocation(testInvocation, status, null));
	}
	
	public InvocationEvent logInvocation(TestInvocation testInvocation, int status, String text){
		return (logInvocation(testInvocation, status, text, null));
	}
	
	public InvocationEvent logInvocation(TestInvocation testInvocation, int status, String text, List eventAnnotations){
		
		if(testInvocation != null){
			
			InvocationEvent invocationEvent = new InvocationEvent();
			invocationEvent.setId(ModelUtil.getId(testInvocation));
			invocationEvent.setOwnerId(testInvocation.getId());
			invocationEvent.setParentId(ModelUtil.getParentIdOfParent(testInvocation));
			
			if(testInvocation.getTest() != null){
				
				invocationEvent.setInvokedId(ModelUtil.getInvokedId(testInvocation));
				invocationEvent.setStatus(status);
				invocationEvent.setReason(InvocationEvent.REASON_NONE);
			}
			else{
				
				invocationEvent.setStatus(InvocationEvent.STATUS_UNSUCCESSFUL);
				invocationEvent.setReason(InvocationEvent.REASON_NO_BEHAVIOR);				
			}
			
			if((text != null) && (text.trim().length() > 0)){
				invocationEvent.setText(text);
			}

			if((eventAnnotations != null) && (eventAnnotations.size() > 0)){
				invocationEvent.getAnnotations().addAll(eventAnnotations);
			}
			
			log(invocationEvent);
			
			return invocationEvent;
		}
		
		return null;
	}

	public LoopEvent logLoop(Loop loop){
		return (logLoop(loop, null));
	}

	public LoopEvent logLoop(Loop loop, String text){
		return (logLoop(loop, text, null));
	}
	
	public LoopEvent logLoop(Loop loop, String text, List eventAnnotations){
	
		if(loop != null){
			
			LoopEvent loopEvent = new LoopEvent();
			loopEvent.setId(ModelUtil.getId(loop));
			loopEvent.setOwnerId(loop.getId());
			loopEvent.setParentId(ModelUtil.getParentIdOfParent(loop));
			loopEvent.setIterations(loop.getIterations());
			loopEvent.setAsychronous(loop.isAsychronous());
			
			if((text != null) && (text.trim().length() > 0)){
				loopEvent.setText(text);
			}
			
			if((eventAnnotations != null) && (eventAnnotations.size() > 0)){
				loopEvent.getAnnotations().addAll(eventAnnotations);
			}
			
			log(loopEvent);
			
			return loopEvent;
		}
		
		return null;
	}

	public VerdictEvent logVerdict(TestInvocation testInvocation, int verdict){
		return (logVerdict(testInvocation, verdict, IEventConstants.NOT_INITIALIZED_VALUE));
	}

	public VerdictEvent logVerdict(TestInvocation testInvocation, int verdict, int reason){
		return (logVerdict(testInvocation, verdict, reason, null));
	}

	public VerdictEvent logVerdict(TestInvocation testInvocation, int verdict, int reason, String text){
		return (logVerdict(testInvocation, verdict, reason, null, null));
	}
	
	public VerdictEvent logVerdict(TestInvocation testInvocation, int verdict, int reason, String text, List eventAnnotations){
	
		if(testInvocation != null){
			
			VerdictEvent verdictEvent = ModelUtil.createVerdictEvent(testInvocation, verdict, reason, text);
						
			if((eventAnnotations != null) && (eventAnnotations.size() > 0)){
				verdictEvent.getAnnotations().addAll(eventAnnotations);
			}
			
			log(verdictEvent);
			
			return verdictEvent;
		}
		
		return null;
	}
}