/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionManager.java,v 1.1 2007/12/05 20:02:56 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.runner.model.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.test.common.event.TypedEvent;
import org.eclipse.hyades.test.common.runner.model.Action;
import org.eclipse.hyades.test.common.runner.model.IActionOwner;
import org.eclipse.hyades.test.common.runner.model.IVerdictListener;
import org.eclipse.hyades.test.common.runner.model.Loop;
import org.eclipse.hyades.test.common.runner.model.NamedElement;
import org.eclipse.hyades.test.common.runner.model.Test;
import org.eclipse.hyades.test.common.runner.model.TestCase;
import org.eclipse.hyades.test.common.runner.model.TestInvocation;
import org.eclipse.hyades.test.common.runner.model.TestSuite;

/**
 * Execution manager.
 * <p>
 * 
 * 
 * @author  Marcelo Paternostro 
 * @author  Paul E. Slauenwhite
 * @version November 27, 2007
 * @since   October 7, 2005
 */
public class ExecutionManager implements IVerdictListener {
	
	private TestSuite rootTestSuite = null;
	private TestInvocation lastExecuted = null;
	private List invalidTestInvocations = null;
	private List executed = null;
	private List startedLoops = null;
	private int testInvocationCount = 0;
	private int testCaseInvocationCount = 0;
	
	public void dispose()
	{		
		ModelUtil.dispose(rootTestSuite);
		
		testInvocationCount = 0;
		testCaseInvocationCount = 0;

		lastExecuted = null;
		rootTestSuite = null;		

		if(invalidTestInvocations != null){
			invalidTestInvocations.clear();
		}
		
		if(executed != null){
			executed.clear();
		}
		
		if(startedLoops != null){
			startedLoops.clear();
		}
	}
	
	/**
	 * @see org.eclipse.hyades.test.common.runner.model.IVerdictListener#handleVerdict(org.eclipse.hyades.test.common.runner.model.TestInvocation)
	 */
	public void handleVerdict(TestInvocation testInvocation)
	{
		lastExecuted = testInvocation;
		if(testInvocation.getVerdictEvent() == null)
		{
			if(executed != null)
				executed.remove(testInvocation);
		}
		else
		{
			if(executed == null){
				executed = new ArrayList();
			}
			
			executed.add(testInvocation);
		}
	}

	/**
	 * Returns the last element that had its verdict set.
	 * @return NamedElement
	 */
	public TestInvocation getLast()
	{
		return lastExecuted;
	}	
	
	public void setRoot(TestSuite testSuite)
	{
		rootTestSuite = testSuite;
	}
	
	public TestSuite getRoot()
	{
		return rootTestSuite;
	}
	
	public void registerTestInvocations(Collection testInvocations){
		
		if(testInvocations != null){
		
			invalidTestInvocations = new ArrayList();
			testInvocationCount = 0;
			testCaseInvocationCount = 0;
			
			//Required for filtering test invocations that reference other test suites:
			Iterator testInvocationsIterator = testInvocations.iterator();
			
			while(testInvocationsIterator.hasNext()){

				TestInvocation testInvocation = ((TestInvocation)(testInvocationsIterator.next()));
				Test test = testInvocation.getTest();
				
				if(test != null){

					testInvocationCount++;
					
					if(test instanceof TestCase){
						testCaseInvocationCount++;
					}
				}
				else{
					invalidTestInvocations.add(testInvocation);
				}
			}
		}
	}

	public List getInvalidTestInvocations(){
		
		if(invalidTestInvocations == null){
			invalidTestInvocations = new ArrayList();
		}
		
		return invalidTestInvocations;
	}
	
	public int getTestInvocationCount(){		
		return testInvocationCount;
	}

	public int getTestCaseInvocationCount(){		
		return testCaseInvocationCount;
	}

	public List getExecutedTestInvocations()
	{
		if(executed == null){
			executed = new ArrayList();
		}
		
		return executed;
	}
	
	public int getExecutionCount(){
		
		if(executed != null){
			return (executed.size());
		}
		
		return 0;	
	}
	
	public void startLoop(Loop loop)
	{
		if(loop == null){
			return;
		}
		
		if(startedLoops == null){
			startedLoops = new ArrayList();
		}
		
		if(!startedLoops.contains(loop)){
		
			startedLoops.add(loop);
			
			ModelUtil.getEventLogger().logLoop(loop, null);
	
			ModelUtil.getEventLogger().logTyped(loop, TypedEvent.START);
		}
	}
	
	public void closeStatedLoops()
	{
		if(startedLoops != null){
			
			Iterator i = startedLoops.iterator(); 
			Loop loop = null;
			
			while(i.hasNext()){
				
				loop = ((Loop)(i.next()));
				
				ModelUtil.getEventLogger().logTyped(loop, TypedEvent.STOP);
			}
			
			startedLoops.clear();
		}
	}
	
	/**
	 * Returns the next named element to be executed or {@link #getRoot()}
	 * if all elements were already executed.
	 * @return NamedElement
	 */
	public NamedElement next()
	{
		if(hasNext())
		{
			NamedElement next = getNext(getRoot());
			
			if(next != null)
			{
				return next;
			}
		}
		
		return getRoot();
	}
	
	private TestInvocation getNext(IActionOwner actionOwner)
	{
		if(actionOwner != null)
		{
			for (Iterator i = actionOwner.getActions().iterator(); i.hasNext();)
			{
				Action action = (Action)i.next();
				if(action instanceof Loop)
				{
					TestInvocation testInvocation = getNext((Loop)action);
					if(testInvocation != null)
						return testInvocation;
				}
				else if(action instanceof TestInvocation)
				{
					TestInvocation testInvocation =(TestInvocation)action;
					if(testInvocation.getTest() != null)
					{
						if(testInvocation.getTest() instanceof TestSuite)
						{
							TestInvocation ti = getNext((TestSuite)testInvocation.getTest());
							if(ti != null)
								return ti;
						}
						else if(testInvocation.getVerdictEvent() == null)
							return testInvocation;
					}
				}
			}
		}
		
		return null;
	}
	
	public boolean hasNext()
	{
		if(getTestInvocationCount() == getExecutionCount())
		{
			return false;
		}
		return true;
	}
	
	/**
	 * Returns whether the specified named element is "atomically" executabled, ie,
	 * if it is executable by it self.
	 * @param namedElement
	 * @return
	 */
	public boolean isExecutable(NamedElement namedElement)
	{
		if((namedElement != null) && (namedElement instanceof TestInvocation))
		{
			Test test = ((TestInvocation)namedElement).getTest();
			if((test != null) && (test instanceof TestCase))
				return true;
		}
		
		return false;
	}
}
