/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ModelUtil.java,v 1.1 2007/12/05 20:02:56 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.runner.model.util;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.test.common.event.ExecutionEvent;
import org.eclipse.hyades.test.common.event.IEventConstants;
import org.eclipse.hyades.test.common.event.TypedEvent;
import org.eclipse.hyades.test.common.event.VerdictEvent;
import org.eclipse.hyades.test.common.runner.model.Action;
import org.eclipse.hyades.test.common.runner.model.IActionOwner;
import org.eclipse.hyades.test.common.runner.model.NamedElement;
import org.eclipse.hyades.test.common.runner.model.TestCase;
import org.eclipse.hyades.test.common.runner.model.TestInvocation;
import org.eclipse.hyades.test.common.runner.model.TestSuite;

/**
 * Model utilities.
 * <p>
 * 
 * 
 * @author  marcelop
 * @author  Paul E. Slauenwhite
 * @version May 2, 2007
 * @since   1.0.2
 */
public class ModelUtil {

	public static final Arbiter DEFAULT_ARBITER = new Arbiter();
	public static final ExecutionManager DEFAULT_EXECUTION_MANAGER = new ExecutionManager();
	
	private static List verdicts = Collections.EMPTY_LIST; 
	private static List verdictReasons = Collections.EMPTY_LIST; 
	private static EventLogger eventLogger = null;
	
	public static void setEventLogger(EventLogger el)
	{
		eventLogger = el;
	}
	
	public static EventLogger getEventLogger()
	{
		return eventLogger;
	}
	
	public static List getVerdicts()
	{
		return verdicts;
	}
	
	public static void setVerdicts(List verdictsList)
	{
		verdicts = verdictsList;
		
		Collections.sort(verdicts, new Comparator()
		{
			public int compare(Object o1, Object o2)
			{
				return o1.toString().compareTo(o2.toString());
			}

			public boolean equals(Object obj)
			{
				return (hashCode() != obj.hashCode());
			}
		});
	}
	
	public static Verdict getDefaultVerdict()
	{
		for (Iterator i = getVerdicts().iterator(); i.hasNext();)
		{
			Verdict verdict = (Verdict)i.next();
			if(verdict.isDefault())
				return verdict;
		}
		return null;
	}
	
	public static Verdict getVerdict(int value)
	{
		for (Iterator i = getVerdicts().iterator(); i.hasNext();)
		{
			Verdict verdict = (Verdict)i.next();
			if(verdict.getValue() == value)
				return verdict;
		}
		return null;
	}

	public static int getVerdictIndex(int verdictValue){
		
		for (int counter = 0; counter < getVerdicts().size(); counter++) {
			
			if(((Verdict)(getVerdicts().get(counter))).getValue() == verdictValue){
				return counter;
			}
		}

		return -1;
	}

	public static int getDefaultVerdictIndex(){

		for (int counter = 0; counter < getVerdicts().size(); counter++) {
			
			if(((Verdict)(getVerdicts().get(counter))).isDefault()){
				return counter;
			}
		}

		return -1;
	}

	public static Verdict getVerdict(String verdictName){
		
		for (Iterator i = getVerdicts().iterator(); i.hasNext();)
		{
			Verdict verdict = (Verdict)i.next();
			if(verdict.toString().equalsIgnoreCase(verdictName))
				return verdict;
		}
		return null;
	}

	public static void clearVerdicts()
	{
		if(verdicts != null)
			verdicts.clear();
	}
	
	public static List getVerdictReasons()
	{
		return verdictReasons;
	}
	
	public static void setVerdictReasons(List verdictReasonsList)
	{
		verdictReasons = verdictReasonsList;
		
		Collections.sort(verdictReasons, new Comparator()
		{
			public int compare(Object o1, Object o2)
			{
				return o1.toString().compareTo(o2.toString());
			}

			public boolean equals(Object obj)
			{
				return (hashCode() != obj.hashCode());
			}
		});
	}
	
	public static VerdictReason getDefaultVerdictReason()
	{
		for (Iterator i = getVerdictReasons().iterator(); i.hasNext();)
		{
			VerdictReason verdictReason = (VerdictReason)i.next();
			if(verdictReason.isDefault())
				return verdictReason;
		}
		return null;
	}
	
	public static VerdictReason getVerdictReason(int value)
	{
		for (Iterator i = getVerdictReasons().iterator(); i.hasNext();)
		{
			VerdictReason verdictReason = (VerdictReason)i.next();
			if(verdictReason.getValue() == value)
				return verdictReason;
		}
		return null;
	}

	public static int getVerdictReasonIndex(int verdictReasonValue){
		
		for (int counter = 0; counter < getVerdictReasons().size(); counter++) {
			
			if(((VerdictReason)(getVerdictReasons().get(counter))).getValue() == verdictReasonValue){
				return counter;
			}
		}

		return -1;
	}
	
	public static int getDefaultVerdictReasonIndex(){

		for (int counter = 0; counter < getVerdictReasons().size(); counter++) {
			
			if(((VerdictReason)(getVerdictReasons().get(counter))).isDefault()){
				return counter;
			}
		}

		return -1;
	}

	public static VerdictReason getVerdictReason(String verdictReasonName){
		
		for (Iterator i = getVerdictReasons().iterator(); i.hasNext();)
		{
			VerdictReason verdictReason = (VerdictReason)i.next();
			if(verdictReason.toString().equalsIgnoreCase(verdictReasonName))
				return verdictReason;
		}
		return null;
	}

	public static void clearVerdictReasons()
	{
		if(verdictReasons != null)
			verdictReasons.clear();
	}
	
	public static boolean isOwnerTestSuite(TestSuite testSuite, NamedElement namedElement)
	{
		if(testSuite == DEFAULT_EXECUTION_MANAGER.getRoot())
			return true;
			
		if((testSuite != null) && (namedElement != null))
		{
			if(namedElement instanceof Action)
				return isOwnerTestSuite(testSuite, (NamedElement)((Action)namedElement).getOwner());
				
			if(namedElement instanceof TestCase)
				return (testSuite == ((TestCase)namedElement).getTestSuite());
				
			if(namedElement instanceof TestSuite)
			{
				if(namedElement == testSuite)
					return true;
					
				namedElement = ((TestSuite)namedElement).getTestInvocation();
				return isOwnerTestSuite(testSuite, namedElement);
			}
		}
			
		return false;
	}
	
	public static String getHierarchyId(NamedElement namedElement)
	{
		StringBuffer sb = new StringBuffer();
		getHierarchyId(namedElement, sb);
		if(sb.length() == 0)
			return "";
		
		return sb.deleteCharAt(0).toString();
	}
	
	private static void getHierarchyId(NamedElement namedElement, StringBuffer id)
	{
		if(namedElement == null)
			return;
					
		if(namedElement instanceof Action)
		{
			Action action = (Action)namedElement;
			getHierarchyId((NamedElement)action.getOwner(), id);
		}
		else if(namedElement instanceof TestSuite)
		{
			getHierarchyId(((TestSuite)namedElement).getTestInvocation(), id);
		}
		
		appendHierarchyId(namedElement, id);
	}
	
	public static String appendHierarchyId(NamedElement namedElement, String id)
	{
		StringBuffer sb = new StringBuffer(id);
		appendHierarchyId(namedElement, sb);
		return sb.toString();
	}
	
	private static void appendHierarchyId(NamedElement namedElement, StringBuffer id){
		if(namedElement != null){
			
			id.append(".");
			id.append(getId(namedElement));
		}
	}

	public static String getInvokedId(TestInvocation testInvocation){

		if(testInvocation != null){
			return (getHierarchyId(testInvocation).concat(".").concat(getId(testInvocation.getTest())));
		}
			
		return null;
	}

	public static String getParentIdOfParent(NamedElement namedElement){

		if(namedElement != null){

			if(namedElement instanceof Action){
				return (getParentId(((NamedElement)(((Action)(namedElement)).getOwner()))));
			}
			
			return (getParentId(namedElement));
		}
			
		return null;
	}

	public static String getParentId(NamedElement namedElement){

		if(namedElement != null){

			if(namedElement instanceof TestSuite){
				
				if(namedElement.equals(DEFAULT_EXECUTION_MANAGER.getRoot())){
					return (ExecutionEvent.ROOT_PARENT);
				}
				else{
					return (getId(((TestSuite)(namedElement)).getTestInvocation()));
				}
			}
			
			return (getId(namedElement));
		}
			
		return null;
	}

	public static String getId(NamedElement namedElement){
		
		if(namedElement != null){

			StringBuffer buffer = new StringBuffer();
			buffer.append(namedElement.getId());
			
			if(namedElement.getIteration() > 0){
				
				buffer.append("(");
				buffer.append(namedElement.getIteration());
				buffer.append(")");	
			}
			
			return (buffer.toString());
		}
			
		return null;
	}
	
	public static VerdictEvent createVerdictEvent(TestInvocation testInvocation, int verdict){
		return (createVerdictEvent(testInvocation, verdict, IEventConstants.NOT_INITIALIZED_VALUE));	
	}

	public static VerdictEvent createVerdictEvent(TestInvocation testInvocation, int verdict, int reason){
		return (createVerdictEvent(testInvocation, verdict, reason, null));	

	}
	
	public static VerdictEvent createVerdictEvent(TestInvocation testInvocation, int verdict, int reason, String text){
		
		if((testInvocation != null) && (testInvocation.getTest() != null)){

			VerdictEvent verdictEvent = new VerdictEvent();
			verdictEvent.setOwnerId(testInvocation.getTest().getId());
			verdictEvent.setParentId(getParentId(testInvocation));
			verdictEvent.setVerdict(verdict);
			verdictEvent.setReason(reason);

			if((text != null) && (text.trim().length() > 0)){
				verdictEvent.setText(text);
			}
			
			return verdictEvent; 
		}
			
		return null;
	}
		
	public static VerdictEvent createVerdictEvent(TestSuite testSuite, int verdict){
		return (createVerdictEvent(testSuite, verdict, IEventConstants.NOT_INITIALIZED_VALUE));	
	}

	public static VerdictEvent createVerdictEvent(TestSuite testSuite, int verdict, int reason){
		return (createVerdictEvent(testSuite, verdict, reason, null));	

	}
	
	public static VerdictEvent createVerdictEvent(TestSuite testSuite, int verdict, int reason, String text){
		
		if(testSuite != null){

			VerdictEvent verdictEvent = new VerdictEvent();
			verdictEvent.setOwnerId(testSuite.getId());
			verdictEvent.setParentId(getParentId(testSuite));
			verdictEvent.setVerdict(verdict);
			verdictEvent.setReason(reason);

			if((text != null) && (text.trim().length() > 0)){
				verdictEvent.setText(text);
			}
			
			return verdictEvent; 
		}
			
		return null;
	}
	
	/**
	 * @deprecated As of TPTP 4.4, use {@link #createVerdictEvent(NamedElement)}.
	 */
	public static VerdictEvent createTestVerdictEvent(TestInvocation testInvocation)
	{
		if((testInvocation == null) || (testInvocation.getTest() == null))
			return null;
			
		StringBuffer sb = new StringBuffer(getHierarchyId(testInvocation));
		appendHierarchyId(testInvocation.getTest(), sb);

		VerdictEvent event = new VerdictEvent();
		event.setOwnerId(sb.toString());
		return event;
	}
	
	public static void dispose(NamedElement namedElement)
	{
		if(namedElement != null)
		{
			if(namedElement instanceof IActionOwner)
				dispose(((IActionOwner)namedElement).getActions());
			namedElement.dispose();
		}
	}
	
	public static void dispose(Collection collection)
	{
		if(collection == null)
			return;
		
		for (Iterator i = collection.iterator(); i.hasNext();)
		{
			Object object = i.next();
			if(object instanceof NamedElement)
				dispose((NamedElement)object);
		}	
		
		collection.clear();
	}
	
	public static void closeExecutedTests()
	{

		for(Iterator i = DEFAULT_EXECUTION_MANAGER.getExecutedTestInvocations().iterator(); i.hasNext();)
		{
			TestInvocation testInvocation = (TestInvocation)i.next();
			if((testInvocation.getTest() instanceof TestSuite) && (testInvocation.getVerdictEvent() == null)){
				getEventLogger().log(DEFAULT_ARBITER.analyse(testInvocation));
			}
			
			getEventLogger().logTyped(testInvocation, TypedEvent.STOP);								
		}
	}
}
