/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ScriptParser.java,v 1.1 2007/12/05 20:02:56 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.runner.model.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.hyades.test.common.runner.model.Action;
import org.eclipse.hyades.test.common.runner.model.IActionOwner;
import org.eclipse.hyades.test.common.runner.model.Loop;
import org.eclipse.hyades.test.common.runner.model.NamedElement;
import org.eclipse.hyades.test.common.runner.model.Test;
import org.eclipse.hyades.test.common.runner.model.TestCase;
import org.eclipse.hyades.test.common.runner.model.TestInvocation;
import org.eclipse.hyades.test.common.runner.model.TestSuite;
import org.eclipse.hyades.test.common.util.XMLUtil;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Script parser.
 * <p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @version May 2, 2007
 * @since   October 10, 2005
 */
public class ScriptParser {
	
	private List verdicts = null;
	private List verdictReasons = null;
	private List testInvocations = null;
	
	private Map namedElementById = null;
	private Map elementByTestSuite = null;
	
	private TestSuite root = null;
	
	private int executionId = -1;
	
	public void dispose()
	{
		if(namedElementById != null)
		{
			namedElementById.clear();
			namedElementById = null;
		}
		
		if(verdicts != null)
		{
			verdicts.clear();
			verdicts = null;
		}
		
		if(verdictReasons != null)
		{
			verdictReasons.clear();
			verdictReasons = null;
		}

		if(elementByTestSuite != null)
		{
			elementByTestSuite.clear();
			elementByTestSuite = null;
		}
		
		if(testInvocations != null)
		{
			testInvocations.clear();
			testInvocations = null;
		}
		
		root = null;
	}
	
	public TestSuite parse(byte[] xmlBytes) throws Exception {
		
		Element testScriptElement = XMLUtil.loadDom(new String(xmlBytes, "UTF-8"), "testScript");
		if(testScriptElement != null)
		{
			NodeList nodeList = XMLUtil.getChildrenByName(testScriptElement, "verdicts");
			if(nodeList != null)
			{
				for(int i=0; i<nodeList.getLength(); i++)
				{
					if(nodeList.item(i) instanceof Element)
					{
						Element verdictsElement = (Element)nodeList.item(i);
						if(verdictsElement != null)
						{
							verdicts = new ArrayList();
							parseVerdicts(verdictsElement);
							break;
						}
					}
				}
			}

			nodeList = XMLUtil.getChildrenByName(testScriptElement, "verdictReasons");
			if(nodeList != null)
			{
				for(int i=0; i<nodeList.getLength(); i++)
				{
					if(nodeList.item(i) instanceof Element)
					{
						Element verdictReasonsElement = (Element)nodeList.item(i);
						if(verdictReasonsElement != null)
						{
							verdictReasons = new ArrayList();
							parseVerdictReasons(verdictReasonsElement);
							break;
						}
					}
				}
			}
			
			nodeList = XMLUtil.getChildrenByName(testScriptElement, "testSuite");
			if(nodeList != null)
			{
				for(int i=0; i<nodeList.getLength(); i++)
				{
					if(nodeList.item(i) instanceof Element)
					{
						Element testSuiteElement = (Element)nodeList.item(i);
						if(testSuiteElement != null)
						{
							root = new TestSuite();
							root.setExecutionId(0);
							
							namedElementById = new HashMap();
							elementByTestSuite = new HashMap();
							testInvocations = Collections.EMPTY_LIST;
							
							parse(root, testSuiteElement);
							adjust(root);
							testInvocations = getTestInvocations(root);
							
							break;
						}
					}
				}
			}
		}
		
		return root;			
	}
	
	public List getTestInvocations() {
		
		if(testInvocations == null){
			return (Collections.EMPTY_LIST);
		}
		
		return (new ArrayList(testInvocations));
	}

	public List getVerdicts(){
		
		if(verdicts == null){
			return (Collections.EMPTY_LIST);
		}
		
		return (new ArrayList(verdicts));
	}
	
	public List getVerdictReasons(){
		
		if(verdictReasons == null){
			return (Collections.EMPTY_LIST);
		}
		
		return (new ArrayList(verdictReasons));
	}
	
	private List getTestInvocations(IActionOwner actionOwner) {
		
		int initialExecutionId = executionId;
		List list = new ArrayList();
		for(Iterator i = actionOwner.getActions().iterator(); i.hasNext();)
		{
			Action action = (Action)i.next();
			action.setExecutionId(executionId);
			if(action.isSynchronous())
				executionId++;
				
			if(action instanceof IActionOwner)
			{
				list.addAll(getTestInvocations((IActionOwner)action));
			}
			else if(action instanceof TestInvocation)
			{
				TestInvocation testInvocation = (TestInvocation)action;
				list.add(testInvocation);

				if(testInvocation.getTest() == null)
					continue;
					
				if(testInvocation.getTest() instanceof IActionOwner)
				{
					testInvocation.getTest().setExecutionId(testInvocation.getExecutionId());
					list.addAll(getTestInvocations((IActionOwner)testInvocation.getTest()));
				}
			}
		}
		
		if(actionOwner instanceof Action)
		{ 
			if(!((Action)actionOwner).isSynchronous())
				executionId = initialExecutionId;
		}
		else if(actionOwner instanceof TestSuite)
		{
			TestInvocation testInvocation = ((TestSuite)actionOwner).getTestInvocation();
			if((testInvocation != null) && (!testInvocation.isSynchronous()))
				executionId = initialExecutionId;
		}
				
		return list;		
	}

	protected void adjust(IActionOwner actionOwner) {
		
		for (Iterator i = actionOwner.getActions().iterator(); i.hasNext();)
		{
			Action action = (Action)i.next();
			if(action instanceof IActionOwner)
			{
				adjust((IActionOwner)action);
			}
			else if(action instanceof TestInvocation)
			{
				TestInvocation testInvocation = (TestInvocation)action;
				if(testInvocation.getTestId() == null)
					continue;

				if(testInvocation.getTest() != null)
					continue;
					
				Object object = namedElementById.get(testInvocation.getTestId());
				if(object instanceof TestCase)
				{
					testInvocation.setTest((Test)object);
				}
				else if(object instanceof TestSuite)
				{
					TestSuite ts = new TestSuite();
					testInvocation.setTest(ts);
					ts.setTestInvocation(testInvocation);
					
					parse(ts, (Element)elementByTestSuite.get(object));
					adjust(ts);
				}
				else
					continue;
			}
		}
	}
	
	protected void parseVerdicts(Element element) {
		
		NodeList nodeList = XMLUtil.getChildrenByName(element, "verdict");
		if(nodeList != null)
		{
			for(int i=0, maxi=nodeList.getLength(); i<maxi; i++)
			{
				Element verdictElement = (Element)nodeList.item(i);
				try
				{
					verdicts.add(new Verdict(XMLUtil.getValue(verdictElement, "label"), Integer.parseInt(XMLUtil.getValue(verdictElement, "value")), XMLUtil.getBooleanValue(verdictElement, "isDefault")));				
				}
				catch (NumberFormatException e)
				{
				}
			}
		}
	}
	
	protected void parseVerdictReasons(Element element) {
		
		NodeList nodeList = XMLUtil.getChildrenByName(element, "verdictReason");
		if(nodeList != null)
		{
			for(int i=0, maxi=nodeList.getLength(); i<maxi; i++)
			{
				Element verdictReasonElement = (Element)nodeList.item(i);
				try
				{
					verdictReasons.add(new VerdictReason(XMLUtil.getValue(verdictReasonElement, "label"), Integer.parseInt(XMLUtil.getValue(verdictReasonElement, "value")), XMLUtil.getBooleanValue(verdictReasonElement, "isDefault")));				
				}
				catch (NumberFormatException e)
				{
				}
			}
		}
	}
	
	protected void parse(TestSuite testSuite, Element element) {
		
		parse((Test)testSuite, element);
		elementByTestSuite.put(testSuite, element);

		testSuite.setLocation(XMLUtil.getValue(element, "location"));

		if(testSuite.getTestInvocation() == null)
		{
			NodeList referencedTestSuites = XMLUtil.getChildrenByName(element, "referencedTestSuites");
			if(referencedTestSuites != null)
			{
				for(int i=0, maxi=referencedTestSuites.getLength(); i<maxi; i++)
				{
					Node node = referencedTestSuites.item(i);
					if(!(node instanceof Element))
						continue;
				
					NodeList referencedTestSuite = XMLUtil.getChildrenByName((Element)node, "testSuite");
					if(referencedTestSuite != null)
					{
						for(int j=0, maxj=referencedTestSuite.getLength(); j<maxj; j++)
						{
							TestSuite ts = new TestSuite();
							parse(ts, (Element)referencedTestSuite.item(j));
						}
					}
				}
			}
		}
		
		NodeList testCaseGroups = XMLUtil.getChildrenByName(element, "testCases");
		if(testCaseGroups != null)
		{
			for(int i=0, maxi=testCaseGroups.getLength(); i<maxi; i++)
			{
				Node node = testCaseGroups.item(i);
				if(!(node instanceof Element))
					continue;
					
				NodeList testCases = XMLUtil.getChildrenByName((Element)node, "testCase");
				if(testCases != null)
				{
					parseTestCase (testSuite, testCases);
				}
			}
		}

		NodeList behaviorNodeList = XMLUtil.getChildrenByName(element, "behavior");
		if(behaviorNodeList != null)
		{
			for(int i=0, maxi=behaviorNodeList.getLength(); i<maxi; i++)
			{
				NodeList actionNodeList = behaviorNodeList.item(i).getChildNodes();
				if(actionNodeList != null)
					parse(testSuite, actionNodeList);
			}
		}
	}
	
	
	protected void parseTestCase (TestSuite testSuite, NodeList testCases) {
		
		for(int j=0, maxj=testCases.getLength(); j<maxj; j++)
		{
			TestCase testCaseObj = new TestCase();
			testCaseObj.setTestSuite(testSuite);
			parse(testCaseObj, (Element)testCases.item(j));
		}
	}
	
	protected void parseTestInvocation (IActionOwner actionOwner, Element actionElement) {
		
		TestInvocation testInvocation = new TestInvocation();
		testInvocation.setOwner(actionOwner);
		testInvocation.setIteration(((NamedElement)(actionOwner)).getIteration());
		parse(testInvocation, actionElement);
	}
	
	
	protected void parse(IActionOwner actionOwner, NodeList nodeList) {
		
		for(int i=0, maxi=nodeList.getLength(); i<maxi; i++)
		{
			Node node = nodeList.item(i);
			if(!(node instanceof Element))
				continue;
									
			Element actionElement = (Element)node;
			String tagName = actionElement.getTagName();
			
			if("testInvocation".equals(tagName))
			{
				parseTestInvocation (actionOwner, actionElement);
			}
			else if("loop".equals(tagName))
			{
				Loop loop = new Loop();
				loop.setOwner(actionOwner);
				parse(loop, actionElement);
				
				if(loop.getIterations() > 1)
				{
					for(int j=2; j<=loop.getIterations(); j++)
					{
						loop = new Loop();
						loop.setOwner(actionOwner);
						loop.setIteration(j);
						parse(loop, actionElement);
					}
				}
			}
		}
	}
	
	protected void parse(TestCase testCase, Element element) {
		parse((Test)testCase, element);
	}
	
	protected void parse(Test test, Element element) {
		parse((NamedElement)test, element);
	}

	protected void parse(Loop loop, Element element) {
		
		parse((Action)loop, element);
		
		loop.setIterations(XMLUtil.getIntValue(element, "iterations"));
		loop.setSynchronous(XMLUtil.getBooleanValue(element, "synchronous"));
		NodeList nodeList = element.getChildNodes();
		if(nodeList != null)
			parse(loop, nodeList);
	}

	protected void parse(TestInvocation testInvocation, Element element) {
		
		parse((Action)testInvocation, element);
			
		testInvocation.setTestId(XMLUtil.getValue(element, "testId"));		
	}

	protected void parse(Action action, Element element) {
		
		parse((NamedElement)action, element);
	
		action.setSynchronous(XMLUtil.getBooleanValue(element, "synchronous"));
	}

	protected void parse(NamedElement namedElement, Element element) {
		
		namedElement.setId(XMLUtil.getValue(element, "id"));
		namedElement.setName(XMLUtil.getValue(element, "name"));
		namedElement.setDescription(XMLUtil.getValue(element, "description"));
		
		namedElementById.put(namedElement.getId(), namedElement);
	}
}