/********************************************************************** 
 * Copyright (c) 2005, 2007 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: FilesystemResourceProvider.java,v 1.6 2007/03/22 17:44:13 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.common.testservices.resources;

import java.io.File;
import java.io.IOException;

import org.eclipse.hyades.test.common.agent.ServiceInvoker;
import org.eclipse.hyades.test.common.agent.UnconfiguredComptestAgentException;
import org.eclipse.hyades.test.common.agent.UnknownTestServiceException;
import org.eclipse.osgi.util.NLS;

/**
 * A FilesystemResourceProvider is a component that provides services for 
 * inspecting available resources on the filesystem of a remote 
 * workbench machine, and for fetching those resources at runtime.
 * 
 * @author jtoomey
 * @provisional
 *
 */
public class FilesystemResourceProvider implements IRemoteResourceProvider {

	protected static final String ROOT = "##ROOT##";
	protected static final String CHILDREN = "Children:";
	protected static final String FILESYSTEM_RESOURCE_PROVIDER = "org.eclipse.hyades.test.core.FilesystemResourceProvider";
	protected static final String EXCEPTION = "Exception:";

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.testservices.resources.IRemoteResourceProvider#listAvailableResources(org.eclipse.hyades.execution.core.testservices.resources.IRemoteResource)
	 */
	public IRemoteResource[] listAvailableResources(IRemoteResource parent) throws IOException, UnknownTestServiceException, UnsupportedRemoteResourceException{
		IRemoteResource[] children = null;
		if (parent == null)
			parent = getRootRemoteResource();
		if (isValidRemoteResourceType(parent)) {
			String path = parent.getFullyQualifiedRemoteResourcePath();
			Object[] args = {path};
			String buffer = NLS.bind("method=listChildren,args:parent={0}", args);
			String result;
			try {
				result = ServiceInvoker.invokeService(getFileServiceName(), buffer);
			}catch (UnconfiguredComptestAgentException e) {
				throw new UnknownTestServiceException(e);
			}
			if (result.startsWith(EXCEPTION)) {
				throw new IOException(result.substring(EXCEPTION.length()));
			}
			children = parseChildrenList(parent, result);
		}
		else
			throw new UnsupportedRemoteResourceException();

		return children;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.testservices.resources.IRemoteResourceProvider#isAvailable(org.eclipse.hyades.execution.core.testservices.resources.IRemoteResource)
	 */
	public boolean isAvailable(IRemoteResource remoteResource) throws IOException, UnknownTestServiceException, UnsupportedRemoteResourceException {
		boolean bRetVal = false;
		if (isValidRemoteResourceType(remoteResource)) {
			String path = remoteResource.getFullyQualifiedRemoteResourcePath();
			Object[] args = {path};
			String buffer = NLS.bind("method=fileExists,args:file={0}", args);
			String result;
			try {
				result = ServiceInvoker.invokeService(getFileServiceName(), buffer);
			}catch (UnconfiguredComptestAgentException e) {
				throw new UnknownTestServiceException(e);
			}
			if (result.startsWith(EXCEPTION)) {
				throw new IOException(result.substring(EXCEPTION.length()));
			}
			bRetVal = Boolean.valueOf(result).booleanValue();
		}
		else
			throw new UnsupportedRemoteResourceException();
		return bRetVal;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.testservices.resources.IRemoteResourceProvider#fetchRemoteResource(org.eclipse.hyades.execution.core.testservices.resources.IRemoteResource, java.lang.String)
	 */
	public boolean fetchRemoteResource(IRemoteResource remoteResource,
			String destinationPath) throws IOException, UnknownTestServiceException, UnsupportedRemoteResourceException {
		boolean bRetVal = false;
		if (isValidRemoteResourceType(remoteResource)) {
			String sourcePath = remoteResource.getFullyQualifiedRemoteResourcePath();
			Object[] args = {sourcePath, destinationPath};
			String buffer = NLS.bind("method=getFile,args:source={0},dest={1}", args);
			String result;
			try {
				result = ServiceInvoker.invokeService(getFileServiceName(), buffer);
			}catch (UnconfiguredComptestAgentException e) {
				throw new UnknownTestServiceException(e);
			}
			if (result.startsWith(EXCEPTION)) {
				throw new IOException(result.substring(EXCEPTION.length()));
			}
			bRetVal = Boolean.valueOf(result).booleanValue();
		}
		else
			throw new UnsupportedRemoteResourceException();
		return bRetVal;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.execution.core.testservices.resources.IRemoteResourceProvider#fetchRemoteResource(org.eclipse.hyades.execution.core.testservices.resources.IRemoteResource)
	 */
	public File fetchRemoteResource(IRemoteResource remoteResource)
			throws IOException, UnsupportedRemoteResourceException, UnknownTestServiceException {
		File file = null;
		if (isValidRemoteResourceType(remoteResource)) {
			String sourcePath = remoteResource.getFullyQualifiedRemoteResourcePath();
			Object[] args = {sourcePath};
			String buffer = NLS.bind("method=getTempFile,args:source={0}", args);
			String result;
			try {
				result = ServiceInvoker.invokeService(getFileServiceName(), buffer);
			}catch (UnconfiguredComptestAgentException e) {
				throw new UnknownTestServiceException(e);
			}
			if (result.startsWith(EXCEPTION)) {
				throw new IOException(result.substring(EXCEPTION.length()));
			}
			if (!result.equalsIgnoreCase(Boolean.toString(false))) {
				file = new File(result);
			}
		}
		else
			throw new UnsupportedRemoteResourceException();
		return file;
	}
	
	/**
	 * @param parent
	 * @param result
	 * @return
	 */
	protected IRemoteResource[] parseChildrenList(IRemoteResource parent, String children) {
		IRemoteResource[] result = {};
		String[] childArray = null;
		if (children.startsWith(CHILDREN)) {
			String childList = children.substring(CHILDREN.length());
			if (childList.length() > 0) {
				childArray = childList.split(",");
				result = new FilesystemResource[childArray.length];
				String path = null;
				
				for (int i=0; i<childArray.length; i++) {
					path = buildPath(parent, childArray[i]);
					result[i] = new FilesystemResource(path);
				}
			}
		}
		return result;
	}

	/**
	 * @return
	 */
	protected IRemoteResource getRootRemoteResource() {
		// TODO Auto-generated method stub
		return new FilesystemResource(ROOT);
	}

	protected String getFileServiceName() {
		return FILESYSTEM_RESOURCE_PROVIDER;
	}
	
	protected boolean isValidRemoteResourceType(IRemoteResource resource)
	{
		return (resource instanceof FilesystemResource);
	}

	/**
	 * @param parent
	 * @param string
	 * @return
	 */
	private String buildPath(IRemoteResource parent, String child) {
		String retVal = null;
		if ( isRoot(parent)) {
			retVal = child;
		}
		else {
			char sep = getSeparator(parent);
			String path = parent.getFullyQualifiedRemoteResourcePath();
			retVal = path;
			if (retVal.charAt(retVal.length()-1) != sep) {
				retVal += sep;
			}
			retVal += child;
		}
		
		return retVal;
	}


	/**
	 * @param parent
	 * @return
	 */
	private char getSeparator(IRemoteResource parent) {
		char retVal = '\\';
		if (parent != null) {
			String path = parent.getFullyQualifiedRemoteResourcePath();
			if (path.length() >= 1 && path.charAt(0) == '/')
				retVal = '/';
		}
		return retVal;
	}


	/**
	 * @param parent
	 * @return
	 */
	private boolean isRoot(IRemoteResource parent) {
		// TODO Auto-generated method stub
		return parent.getFullyQualifiedRemoteResourcePath().equals(ROOT);
	}


}
