/********************************************************************** 
 * Copyright (c) 2005, 2006 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: IRemoteResourceProvider.java,v 1.2 2006/03/29 20:12:54 jptoomey Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.common.testservices.resources;

import java.io.File;
import java.io.IOException;

import org.eclipse.hyades.test.common.agent.UnknownTestServiceException;


/**
 * An IRemoteResourceProvider is a component which provides services for 
 * inspecting available resources on a remote workbench machine, and for 
 * fetching those resources at runtime.
 *    
 * @author jtoomey
 * @provisional
 */
public interface IRemoteResourceProvider {

	/**
	 * This method returns an array containing the available IRemoteResources
	 * which are children of the specified parent.  If parent is null, this
	 * method returns the IRemoteResources which are at the root of this Remote
	 * Resource hierarchy.
	 * <p><pre>
	 * 	IRemoteResourceProvider provider = RemoteResourceProviderFactory.getInstance().
	 * 		getRemoteResourceProvider(RemoteResourceProviderFactory.FILESYSTEM_RESOURCE_PROVIDER);
	 * 	IRemoteResource[] roots = provider.listAvailableResources(null);
	 * 	for (int i = 0; i < roots.length; i++) {
	 * 		IRemoteResource resource = roots[i];
	 * 		System.out.println(resource.getFullyQualifiedRemoteResourcePath());
	 * 	}
	 * 
	 * 	// List all children of the first filesystem root
	 * 	FilesystemResource resource = (FilesystemResource) roots[0];
	 * 	roots = provider.listAvailableResources(resource);
	 * 	for (int i = 0; i < roots.length; i++) {
	 * 		IRemoteResource res = roots[i];
	 * 		System.out.println(res.getFullyQualifiedRemoteResourcePath());
	 * 	}
	 * </pre></p>
	 * @param parent
	 * @return an array containing the children of parent, or the root remote 
	 * resources if parent is null or an empty array if parent has no children.
	 * @throws UnsupportedRemoteResourceException 
	 * @throws UnknownTestServiceException 
	 * @throws IOException 
	 */
	public IRemoteResource[] listAvailableResources(IRemoteResource parent) throws IOException, UnknownTestServiceException, UnsupportedRemoteResourceException;
	
	/**
	 * This methods returns whether or not the specified remote resource is
	 * available from this remote resource provider.
	 * <p><pre>
	 * 	IRemoteResourceProvider provider = RemoteResourceProviderFactory.getInstance().
	 * 		getRemoteResourceProvider(RemoteResourceProviderFactory.WORKSPACE_RESOURCE_PROVIDER);
	 * 	WorkspaceResource resource = new WorkspaceResource("MyProject", "bin/MyClass.class");
	 * 	assertTrue("File should exist but does not: " + resource.getFullyQualifiedRemoteResourcePath(), 
	 * 		provider.isAvailable(resource));
	 * 		
	 * 	resource = new WorkspaceResource("MyProject", "foobar_file_that_does_not_exist.txt");
	 * 	assertFalse("File should not exist, but does: " + resource.getFullyQualifiedRemoteResourcePath(), 
	 * 		provider.isAvailable(resource));
	 * </pre></p>
	 * @param resource
	 * @return true if the specified remote resource is available, 
	 *  false otherwise
	 * @throws IOException 
	 */
	public boolean isAvailable(IRemoteResource resource) throws UnknownTestServiceException, UnsupportedRemoteResourceException, IOException;
	
	/**
	 * This method fetches the specified remote resource, writing a copy of it
	 * to the specified destination directory.  For example:
	 * <p><pre>
	 * 	IRemoteResourceProvider provider = RemoteResourceProviderFactory.getInstance().
	 *		getRemoteResourceProvider(RemoteResourceProviderFactory.WORKSPACE_RESOURCE_PROVIDER);
	 *	WorkspaceResource resource = new WorkspaceResource("MyProject", "bin/org/eclipse/tptp/test/Test.class");
	 *	assertTrue("File not copied!", provider.fetchRemoteResource(resource, "c:\\Test.class"));
	 * </pre></p>
	 * 
	 * @param remoteResource
	 * @param destinationPath
	 * @return true if the file was successfully fetched, false if the file was
	 * not available from this remote resource provider
	 * @throws IOException - if the file can not be read or written
	 * @throws UnknownTestServiceException 
	 * @throws UnsupportedRemoteResourceException 
	 */
	public boolean fetchRemoteResource(IRemoteResource remoteResource, String destinationPath) throws IOException, UnknownTestServiceException, UnsupportedRemoteResourceException;

	/**
	 * This method fetches the specified remote resource, writing a copy of it
	 * to a temporary location, and returning the resulting File.  For example:
	 * <p><pre>
	 * 	IRemoteResourceProvider provider = RemoteResourceProviderFactory.getInstance().
	 * 		getRemoteResourceProvider(RemoteResourceProviderFactory.WORKSPACE_RESOURCE_PROVIDER);
	 *	WorkspaceResource resource = new WorkspaceResource("MyProject", "bin/org/eclipse/tptp/test/Test.class");
	 *	File file = provider.fetchRemoteResource(resource);
	 *
	 *	if (file != null) {
	 *		System.out.println("file = " + file.getAbsolutePath());
	 *		assertTrue("file was not copied.", file.exists());
	 *	}
	 *	else {
	 *		fail("Tried to fetch file to temp directory, but null was returned.");
	 *	}
	 * </pre></p>
	 * @param remoteResource
	 * @return A File which points to the newly fetched file, or null if the
	 * file was not available from this remote resource provider.
	 * @throws IOException - if the file can not be read or written
	 * @throws UnsupportedRemoteResourceException 
	 * @throws UnknownTestServiceException 
	 */
	public File fetchRemoteResource(IRemoteResource remoteResource) throws IOException, UnsupportedRemoteResourceException, UnknownTestServiceException;
	
//	public boolean fetchRemoteResources(IRemoteResource[] remoteResources, String destinationDirectory) throws IOException;
//	public File[] fetchRemoteResources(IRemoteResource[] remoteResources) throws IOException;
}
