/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: XMLUtil.java,v 1.11 2008/05/30 18:34:16 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.common.util;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXParseException;

/**
 * XMLUtil.java
 * </a>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @version May 27, 2008
 * @since   January 27, 2005
 */
public class XMLUtil
{

	/**
     * Normalizes the parameter string according to the XML specification for
     * attribute-value normalization 
     * (<a href="http://www.w3.org/TR/REC-xml">http://www.w3.org/TR/REC-xml</a>)
     * and valid characters 
     * (<a href="http://www.w3.org/TR/REC-xml#charsets">http://www.w3.org/TR/REC-xml#charsets</a>).
     * <p>
     * Valid characters, according to the XML specification, include any Unicode
     * character, excluding the surrogate blocks, 0xFFFE, and 0xFFFF.
     * <p>
     * In addition, special handling is provided for Microsoft "smart quotes" and 
     * several other characters from Windows-1252 that are auto-inserted by Microsoft 
     * products. These characters are converted to valid characters using the following
     * mapping:
     * <p>
     * <table>
     * <tr><th>Character Name</th><th>Hexadecimal Representation</th><th>Normalized Representation</th></tr>
     * <tr><td>HORIZONTAL ELLIPSIS</td><td>0x0085</td><td>"..."</td></tr>
     * <tr><td>LEFT SINGLE QUOTATION MARK</td><td>0x0091</td><td>"&apos;"</td></tr>
     * <tr><td>RIGHT SINGLE QUOTATION MARK</td><td>0x0092</td><td>"&apos;"</td></tr>
     * <tr><td>LEFT DOUBLE QUOTATION MARK</td><td>0x0093</td><td>"&quot;"</td></tr>
     * <tr><td>RIGHT DOUBLE QUOTATION MARK</td><td>0x0094</td><td>"&quot;"</td></tr>
     * <tr><td>EM DASH</td><td>0x0097</td><td>"--"</td></tr>
     * </table>
     * <p>
     * A <code>null</code> parameter string results in a <code>null</code> return string.  
     * Likewise, an empty parameter string results in a "" return string.
     * <p>
     * Entity reference values are represented in the hexadecimal 
     * (e.g. <code>&amp;#xa;</code> for the Web line separator character) form.
     * <p>
     * 
     * @param  string The string to be normalized.
     * @return The normalized string, <code>null</code>, or an empty string.
     * @see    #removeXMLSymbols(String)
     */
	public static String useXMLSymbols(String string)
	{
        //Return null if the string is null:
		if(string != null){
			
	        int stringLength = string.length();

	        //Return an empty string if the string is empty:
	        if (stringLength > 0) { 

		        int marker = 0;
		        int counter = 0;
		        char[] characters = string.toCharArray();
		        StringBuffer normalizedString = new StringBuffer(stringLength);
	
		        //Check if any characters require normalization or replacement of invalid characters:
		        while (counter < stringLength) {
	
		        	//Only check non-alpha and non-numeric characters
		        	if ((characters[counter] < 0x0028) || ((characters[counter] > 0x003B) && (characters[counter] < 0x003F)) || (characters[counter] > 0x007E)) {
	
		        		//0x003C:
			            if (characters[counter] == '<') {
			
			            	normalizedString.append(characters,marker,(counter - marker));
			            	marker = (counter + 1);
			
			            	normalizedString.append("&lt;");
			            }
			
			            //0x003E:
			            else if (characters[counter] == '>') {
			            	
			            	normalizedString.append(characters,marker,(counter - marker));
			            	marker = (counter + 1);
			
			            	normalizedString.append("&gt;");
			            }
			
			            //0x0026:
			            else if (characters[counter] == '&') {
			
			            	normalizedString.append(characters,marker,(counter - marker));
			            	marker = (counter + 1);
			
			            	normalizedString.append("&amp;");
			            }
			
			            //0x0022, LEFT DOUBLE QUOTATION MARK, and RIGHT DOUBLE QUOTATION MARK:
			            else if ((characters[counter] == '"') || (characters[counter] == 0x0093) || (characters[counter] == 0x0094)) {
			
			            	normalizedString.append(characters,marker,(counter - marker));
			            	marker = (counter + 1);
			
			            	normalizedString.append("&quot;");
			            }
			
			            //0x0027, LEFT SINGLE QUOTATION MARK, and RIGHT SINGLE QUOTATION MARK:
			            else if ((characters[counter] == '\'') || (characters[counter] == 0x0091) || (characters[counter] == 0x0092)) {
			
			            	normalizedString.append(characters,marker,(counter - marker));
			            	marker = (counter + 1);
			
			            	normalizedString.append("&apos;");
			            }
			
			            //0x0009:
			            else if (characters[counter] == '\t') {
			
			            	normalizedString.append(characters,marker,(counter - marker));
			            	marker = (counter + 1);
			
			            	normalizedString.append("&#x9;");
			            }
			
			            //0x000A:
			            else if (characters[counter] == '\n') {
			
			            	normalizedString.append(characters,marker,(counter - marker));
			            	marker = (counter + 1);
			
			            	normalizedString.append("&#xA;");
			            }
			
			            //0x000D:
			            else if (characters[counter] == '\r') {
			
			            	normalizedString.append(characters,marker,(counter - marker));
			            	marker = (counter + 1);
			
			            	normalizedString.append("&#xD;");
			            }

			            //HORIZONTAL ELLIPSIS:
			            else if (characters[counter] == 0x0085) {
			
			            	normalizedString.append(characters,marker,(counter - marker));
			            	marker = (counter + 1);
			
			            	normalizedString.append("...");
			            }

			            //EM DASH:
			            else if (characters[counter] == 0x0097) {
			
			            	normalizedString.append(characters,marker,(counter - marker));
			            	marker = (counter + 1);
			
			            	normalizedString.append("--");
			            }

			            /*
			             * //0x0020: else if (character == ' '){
			             * normalizedString.append("&#x20;"); }
			             * 			
			             * //Handle valid UTF-16 character range:
			             * else if (!(((characters[counter] >= 0x0020) && (characters[counter] <= 0xD7FF)) || ((characters[counter] >= 0xE000) && (characters[counter] <= 0xFFFD)))) {
			             * 
			             * normalizedString.append(characters,marker,(counter - marker));
			             * marker = (counter + 1);
			             * 
			             * normalizedString.append('?');
			             * }
			             */
		        	}
		        	
			        counter++;
		        }
		        		 
		        if(marker == 0){
		        	return string;
		        }
		        
		        if(marker < counter){
		        	normalizedString.append(characters,marker,(counter - marker));
		        }
		        		
		        return (normalizedString.toString());
	        }
		}
		
		return string;
	}
	
    /**
     * De-normalizes the parameter string.
     * <p>
     * Entity reference values may be represented in either the hexadecimal
     * (e.g. <code>&amp;#xa;</code> for the Web line separator character) or 
     * decimal (e.g. <code>&amp;#10;</code> for the Web line separator character) 
     * form.
     * <p>
     * A <code>null</code> parameter string results in a <code>null</code> return string.  
     * Likewise, an empty parameter string results in a "" return string.
     * <p>
     * 
     * @param  string The string to be de-normalized.
     * @return The de-normalized string, <code>null</code>, or an empty string.
     * @see    #useXMLSymbols(String)
     */
	public static String removeXMLSymbols(String string)
	{
		
        //Return null if the xml string is null:
		if(string != null){
			
	        int stringLength = string.length();

	        //Return an empty string if the xml string is empty:
	        if (stringLength > 0) { 

	        	 int marker = 0;
	             int counter = 0;
	             int semiColonIndex = -1;
	             String entityReferenceName = null;
	             char[] characters = string.toCharArray();
	             StringBuffer denormalizedString = new StringBuffer(stringLength);

	             //Check if any entity references require denormalization:
	             while (counter < stringLength) {

	                 //Check if the current character is the start of a possible entity
	                 //reference (e.g. ampersand in &<name>;) and find a possible end to
	                 //the possible entity reference (e.g. semi-solon in &<name>;):
	                 if ((characters[counter] == '&') && ((semiColonIndex = string.indexOf(';', (counter + 1))) != -1)) {

	                     entityReferenceName = string.substring((counter + 1), semiColonIndex).trim();

	                     if (entityReferenceName.equals("lt")){

	                     	denormalizedString.append(characters,marker,(counter - marker));
	                     	denormalizedString.append('<');
	                     }
	                     else if (entityReferenceName.equals("gt")){

	                     	denormalizedString.append(characters,marker,(counter - marker));
	                         denormalizedString.append('>');
	                     }
	                     else if (entityReferenceName.equals("amp")){

	                     	denormalizedString.append(characters,marker,(counter - marker));
	                         denormalizedString.append('&');
	                     }
	                     else if (entityReferenceName.equals("quot")){

	                     	denormalizedString.append(characters,marker,(counter - marker));
	                         denormalizedString.append('"');
	                     }
	                     else if (entityReferenceName.equals("apos")){

	                     	denormalizedString.append(characters,marker,(counter - marker));
	                         denormalizedString.append('\'');
	                     }
	                     else if ((entityReferenceName.equals("#x9")) || (entityReferenceName.equals("#09"))){

	                     	denormalizedString.append(characters,marker,(counter - marker));
	                         denormalizedString.append('\t');
	                     }
	                     else if ((entityReferenceName.equals("#xA")) || (entityReferenceName.equals("#10"))){

	                     	denormalizedString.append(characters,marker,(counter - marker));
	                         denormalizedString.append('\n');
	                     }
	                     else if ((entityReferenceName.equals("#xD")) || (entityReferenceName.equals("#13"))){

	                     	denormalizedString.append(characters,marker,(counter - marker));
	                         denormalizedString.append('\r');
	                     }
	                     
	                     /*
	                      * else if ((entityReferenceName.equals("#x20")) || (entityReferenceName.equals("#32"))){
	                      * 
	                      * 	denormalizedString.append(characters,marker,(counter - marker));
	                      *  denormalizedString.append(' ');
	                      * }
	                      */

	                     //Unsupported entity reference:
	                     else {
	                     	denormalizedString.append(characters,marker,(semiColonIndex + 1 - marker));
	                     }

	                 	counter = semiColonIndex;
	                 	marker = (counter + 1);
	             	}
	             	
	     	        counter++;
	             }
	             		 
	             if(marker == 0){
	             	return string;
	             }
	             
	             if(marker < counter){
	             	denormalizedString.append(characters,marker,(counter - marker));
	             }
	             		
	             return (denormalizedString.toString());
	        }
		}
		
		return string;
	}
	
	/**
	 * Returns the XML string for the specified attribute and value.  The
	 * value is passed to the {@link #useXMLSymbols(String)} method.
	 * 
	 * <p>If the <code>value</code> is <code>null</code> then the return is
	 * an empty StringBuffer.
	 * 
	 * <p>If <code>asChild</code> is <code>true</code> than this method returns 
	 * <PRE><attribute>value</attribute></PRE>.  If it <code>false</code> the
	 * return is " <i>attribute</i>="<i>value</i>""
	 * 
	 * @param attribute
	 * @param value
	 * @param asChild
	 * @return StringBuffer. 
	 */
	public static StringBuffer createXMLAttribute(String attribute, String value, boolean asChild)
	{
		if(value == null)
			return new StringBuffer(0);
		value = useXMLSymbols(value);
		
		if(asChild)
			return new StringBuffer("<").append(attribute).append(">").append(value).append("</").append(attribute).append(">");
		return new StringBuffer(" ").append(attribute).append("=\"").append(value).append("\""); 
	}
	
	
	/*
	 * Xerces utils
	 */


	/**
	 * Returns an Element with a given name that was loaded 
	 * from the specified input stream.
	 * @param inputStream
	 * @param name
	 * @return Element 
	 */
	public static Element loadDom(InputStream inputStream, String name)
	{
		if((inputStream == null) || (name == null))
			return null;
		
		Document doc = null;
		
		try
		{
			DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();

			docBuilderFactory.setIgnoringComments(true);
			docBuilderFactory.setValidating(true);
	
			DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
			docBuilder.setErrorHandler(createErrorHandler(false));
			doc = docBuilder.parse(inputStream);
		}
		catch(Throwable t)
		{
			t.printStackTrace();
		}
		
		if(doc == null)
			return null;
					
		if(name != null)
		{
			NodeList list = doc.getElementsByTagName(name);
			if((list.getLength() > 0) && (list.item(0) instanceof Element))
				return (Element)list.item(0);
			else
				return null;
		}
			
		return (Element)doc.getFirstChild();
	}
	
	/**
	 * Returns an Element with a given name that was loaded 
	 * from the specified string.
	 * @param xml
	 * @param name
	 * @return Element 
	 */
	public static Element loadDom(String xml, String name)
	{
		if((xml == null) || (name == null))
			return null;
		
		try
		{
			return loadDom(new ByteArrayInputStream(xml.getBytes("UTF-8")), name);
		}
		catch(UnsupportedEncodingException e)
		{
			throw new RuntimeException(e);
		}
	}
	
	/**
	 * Returns a value identified by a name in a given <code>element</code>.  This 
	 * value can correspond to a regular XML attribute value or to a child node
	 * value.  In the last case there should be only one element with the given name.
	 * 
	 * <p>This method returns null if the value is a zero-length string.
	 * 
	 * @param element
	 * @param name
	 * @return String
	 */	
	public static String getValue(Element element, String name)
	{
		if((element == null) || (name == null))
			return null;
			
		String value = element.getAttribute(name);
		if((value == null) || (value.length() == 0))
		{
			NodeList list = element.getElementsByTagName(name);
			for(int i=0, max=list.getLength(); i<max; i++)
			{
				Node textNode = list.item(i);
				Node firstChild = list.item(0).getFirstChild();
				if(firstChild != null)
				{
					String auxValue = firstChild.getNodeValue();
					if((auxValue != null) && textNode.getParentNode().equals(element))
					{
						value = auxValue;
						break;
					}
				}
			}
		}
		if(value.length()== 0)
			return null;
			
		return value;
	}
	
	/**
	 * Returns a {@link NodeList} with all the children identified by a name 
	 * in a given <code>element</code>.   
	 * @param element
	 * @param name
	 * @return NodeList
	 */	
	public static NodeList getChildrenByName(Element element, String name)
	{
		if((element == null) || (name == null))
			return null;
			
		List list = new ArrayList();
		
		NodeList children = element.getChildNodes();
		for(int i=0, max=children.getLength(); i<max; i++)
		{
			if((children.item(i) instanceof Element) && name.equals(((Element)children.item(i)).getTagName()))
				list.add(children.item(i));
		}
		
		return createNodeList(list);
	}
	
	/**
	 * Returns a {@link NodeList} with all nodes of a given list.  The 
	 * returned node list will thrown a <code>ClassCastException</code> if there 
	 * is a element in the list that is not a {@link Node}
	 * @param element
	 * @param name
	 * @return NodeList
	 */	
	public static NodeList createNodeList(final List nodes)
	{
		if(nodes == null)
			return null;
			
		NodeList nodeList = new NodeList()
		{
			public int getLength()
			{
				return nodes.size();
			}

			public Node item(int index)
			{
				return (Node)nodes.get(index);
			}
		};
		
		return nodeList;
	}
	
	/**
	 * Returns a document based on the content of the specified file.
	 * @param filePath
	 * @return Document
	 */
	public static Document getXmlDom(String filePath)
	{
		try
		{
	
			DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
			/*
			docBuilderFactory.setExpandEntityReferences(true);
			docBuilderFactory.setCoalescing(true);
			docBuilderFactory.setIgnoringElementContentWhitespace(true);
			docBuilderFactory.setNamespaceAware(true);
			*/
	
			docBuilderFactory.setIgnoringComments(true);
			docBuilderFactory.setValidating(true);
	
			DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
	
			final File xmlFile = new File(filePath);
	
	
			docBuilder.setErrorHandler(createErrorHandler(false));
			return (docBuilder.parse(xmlFile));
		}
		catch (Exception e)
		{
		}
	
		return null;
	}
	
	/**
	 * Creates the error handler used by this class.
	 * @param logMessages
	 * @return ErrorHandler
	 */
	private static ErrorHandler createErrorHandler(final boolean logMessages)
	{
		ErrorHandler errorHandler = new ErrorHandler()
		{
			public void error(SAXParseException exception)
			{
				if(logMessages)
					exception.printStackTrace();
			}

			public void fatalError(SAXParseException exception)
			{
				if(logMessages)
					exception.printStackTrace();
			}

			public void warning(SAXParseException exception)
			{
				if(logMessages)
					exception.printStackTrace();
			}
		};
		
		return errorHandler;
	}
	/**
	 * Get the value of the attribute tagName in element as an integer 
	 * @param element
	 * @param tagName
	 * @return int
	 */
	public static int getIntValue(Element element, String tagName)
	{
		try
		{
			return Integer.parseInt(getValue(element, tagName));
		}
		catch(NumberFormatException nfe)
		{
		}
		
		return 0;
	}
	
	/**
	 * Get the value of the attribute tagName in element as an boolean 
	 * 
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public static boolean getBooleanValue(Element element, String tagName)
	{
		return (Boolean.valueOf(getValue(element, tagName)).booleanValue());
	}
	
	/**
	 * @param name
	 * @param value
	 * @return String
	 * @deprecated use {@link #createXMLAttribute(String, String, boolean)} instead.
	 */
	public static String xmlVariable(String name, String value)
	{
		return createXMLAttribute(name, value, false).toString();
	}
	
	/**
	 * @param value
	 * @return String
	 * @deprecated use {@link #useXMLSymbols(String)} instead.
	 */
	public static String adjustXMLValue(String value)
	{
		return useXMLSymbols(value);
	}
}
