/*******************************************************************************
 * Copyright (c) 2009, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Cookie.java,v 1.2 2010/05/06 18:21:58 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.http.runner;

/**
 * <p>Represents state information for HTTP requests commonly
 * referred to as a cookie.</p>
 * 
 * <p>For more information, see <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.</p>
 *  
 * 
 * @author  Suzanne Dirkers
 * @author  Paul Slauenwhite
 * @version May 6, 2010
 * @since   August 11, 2009
 * @deprecated As of TPTP V4.7.0, use {@link org.eclipse.hyades.test.http.runner.internal.exec.Cookie}.
 */
public class Cookie{

	/**
	 * <p>The name of the cookie.  By default, <code>null</code>.</p>
	 *
	 * <p>If the name of the cookie is <code>null</code>, the value of the cookie is ignored.</p>
	 */
	private String name = null;

	/**
	 * <p>The value of the cookie.  By default, <code>null</code>.</p>
	 * 
	 * <p>If the name of the cookie is <code>null</code>, the value of the cookie is ignored.</p>
	 */
	private String value = null;

	/**
	 * <code>true</code> if the cookie is secure.  By default, <code>false</code>.
	 */
	private boolean isSecure = false;
	
	/**
	 * <p>The expiry date of the cookie.  By default, <code>null</code>.</p>
	 * 
	 * <p>The format of the expiry date is <code>EEE, dd-MMM-yyyy HH:mm:ss z</code> (see {@link java.text.SimpleDateFormat}).  For example, <code>Thu, 01-July-2015 00:00:20 GMT</code>.  For more information, see section 10.1.2 of <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.</p>
	 */
	private String expires = null;

	/**
	 * The host (name or IP address) or domain of the cookie.  By default, <code>null</code>.
	 */
	private String domain = null;

	/**
	 * The path on the host or domain of the cookie.  By default, <code>null</code>.
	 */
	private String path = null;

	/**
	 * The version (<code>0</code> or higher) of the cookie.  By default, <code>0</code>.
	 */
	private int version = 0;

	/**
	 * <code>true</code> if the domain of the cookie contains an IP address.  By default, <code>false</code>.
	 */
	private boolean isDomainIPAddress = false;

	/**
	 * <p>Creates a new cookie with the default properties.</p>
	 * 
	 * <p>For more information on the properties of a cookie, see <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.</p>
	 * 
	 * <p>By default, the new cookie:</p>
	 * 
	 * <ul>
	 * <li>has <code>null</code> string,</li> 
	 * <li>has a <code>0</code> version,</li> 
	 * <li>is not secure,</li>
	 * <li>and the domain is not an IP address.</li>
	 * </ul> 
	 */
	public Cookie() {
		//No-operation.
	}

	/**
	 * <p>Creates a new cookie with the specified properties.</p>
	 * 
	 * <p>For more information on the properties of a cookie, see <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.</p>
	 *  
	 * @param name The name of the cookie, otherwise <code>null</code>.
	 * @param value The value of the cookie, otherwise <code>null</code>.  If the name of the cookie is <code>null</code>, the value of the cookie is ignored.
	 * @param isSecure <code>true</code> if the cookie is secure, otherwise <code>false</code>.
	 * @param expires The expiry date of the cookie, otherwise <code>null</code>.  The format of the expiry date is <code>EEE, dd-MMM-yyyy HH:mm:ss z</code> (see {@link java.text.SimpleDateFormat}).  For example, <code>Thu, 01-July-2015 00:00:20 GMT</code>.  For more information, see section 10.1.2 of <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.
	 * @param domain The host (name or IP address) or domain of the cookie, otherwise <code>null</code>.
	 * @param path The path on the host or domain of the cookie, otherwise <code>null</code>.
	 * @param version The version (<code>0</code> or higher) of the cookie, otherwise <code>0</code>.
	 * @param isDomainIPAddress <code>true</code> if the domain of the cookie contains an IP address, otherwise <code>false</code>.
	 */
	public Cookie(String name, String value, boolean isSecure, String expires, String domain, String path, int version, boolean isDomainIPAddress){

		this.name = name;

		//Invoke the setter after instantiating the name to enforce the non-null name when a non-null value constraint:
		setValue(value);

		this.isSecure = isSecure;
		this.expires = expires;
		this.domain = domain;
		this.path = path;
		this.version = version;
		this.isDomainIPAddress = isDomainIPAddress;
	}

	/**
	 * <p>Returns the name of the cookie.  By default, <code>null</code>.</p>
	 *
	 * <p>If the name of the cookie is <code>null</code>, the value of the cookie is ignored.</p>
	 *
	 * @return The name of the cookie, otherwise <code>null</code>.
	 */
	public String getName() {
		return name;
	}

	/**
	 * <p>Sets the name of the cookie.  By default, <code>null</code>.</p>
	 *
	 * <p>If the name of the cookie is <code>null</code>, the value of the cookie is set to <code>null</code>.</p>
	 * 
	 * @param name The name of the cookie, otherwise <code>null</code>.
	 */
	public void setName(String name) {

		if(name != null){
			this.name = name;
		}
		else{
			this.value = null;
		}
	}

	/**
	 * <p>Returns the value of the cookie.  By default, <code>null</code>.</p>
	 * 
	 * <p>If the name of the cookie is <code>null</code>, the value of the cookie is ignored.</p>
	 *
	 * @return The value of the cookie, otherwise <code>null</code>. 
	 */
	public String getValue() {
		return value;
	}

	/**
	 * <p>Sets the value of the cookie.  By default, <code>null</code>.</p>
	 * 
	 * <p>If the name of the cookie is <code>null</code>, the value of the cookie is ignored.</p>
	 * 
	 * @param value The value of the cookie, otherwise <code>null</code>. 
	 */
	public void setValue(String value) {

		if(name != null){
			this.value = value;
		}
	}

	/**
	 * <p>Returns the name and value pair of the cookie.  For example, <code><i>name</i>=<i>value</i></code>.  
	 * By default, <code>null</code>.</p>
	 * 
	 * <p>If the name of the cookie is <code>null</code>, the method returns <code>null</code>.</p>
	 * 
	 * <p>If the name of the cookie is not <code>null</code> and the value of the cookie is <code>null</code>, 
	 * the method returns <code><i>name</i>=</code>.</p>
	 *
	 * @return The name and value pair of the cookie, otherwise <code>null</code>. 
	 */
	public String getNameValuePair() {

		if(name != null){

			if(value != null){
				return (name + "=" + value); //$NON-NLS-1$				
			}
			else{
				return (name + "="); //$NON-NLS-1$								
			}
		}

		return null;
	}

	/**
	 * Returns if the cookie is secure.  By default, <code>false</code>.
	 *
	 * @return <code>true</code> if the cookie is secure, otherwise <code>false</code>.
	 */
	public boolean isSecure() {
		return isSecure;
	}

	/**
	 * Sets if the cookie is secure.  By default, <code>false</code>.
	 * 
	 * @param isSecure <code>true</code> if the cookie is secure, otherwise <code>false</code>.
	 */
	public void setSecure(boolean isSecure) {
		this.isSecure = isSecure;
	}
	
	/**
	 * <p>Returns the expiry date of the cookie.  By default, <code>null</code>.</p>
	 * 
	 * <p>The format of the expiry date is <code>EEE, dd-MMM-yyyy HH:mm:ss z</code> (see {@link java.text.SimpleDateFormat}).  For example, <code>Thu, 01-July-2015 00:00:20 GMT</code>.  For more information, see section 10.1.2 of <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.</p>
	 *
	 * @return The expiry date of the cookie, otherwise <code>null</code>.
	 */
	public String getExpires() {
		return expires;
	}

	/**
	 * <p>Sets the expiry date of the cookie.  By default, <code>null</code>.</p>
	 * 
	 * <p>The format of the expiry date is <code>EEE, dd-MMM-yyyy HH:mm:ss z</code> (see {@link java.text.SimpleDateFormat}).  For example, <code>Thu, 01-July-2015 00:00:20 GMT</code>.  For more information, see section 10.1.2 of <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.</p>
	 * 
	 * @param expires The expiry date of the cookie, otherwise <code>null</code>.
	 */
	public void setExpires(String expires) {
		this.expires = expires;
	}

	/**
	 * Returns the host (name or IP address) or domain of the cookie.  By default, <code>null</code>.
	 *
	 * @return The host (name or IP address) or domain of the cookie, otherwise <code>null</code>.
	 */
	public String getDomain() {
		return domain;
	}

	/**
	 * Sets the host (name or IP address) or domain of the cookie.  By default, <code>null</code>.
	 * 
	 * @param domain The host (name or IP address) or domain of the cookie, otherwise <code>null</code>.
	 */
	public void setDomain(String domain) {
		this.domain = domain;
	}

	/**
	 * Returns the path on the host or domain of the cookie.  By default, <code>null</code>.
	 *
	 * @return The path on the host or domain of the cookie, otherwise <code>null</code>.
	 */
	public String getPath() {
		return path;
	}

	/**
	 * Sets the path on the host or domain of the cookie.  By default, <code>null</code>.
	 * 
	 * @param path The path on the host or domain of the cookie, otherwise <code>null</code>.
	 */
	public void setPath(String path) {
		this.path = path;
	}

	/**
	 * Returns version (<code>0</code> or higher) of the cookie.  By default, <code>0</code>.
	 *
	 * @return The version (<code>0</code> or higher) of the cookie, otherwise <code>0</code>.
	 */
	public int getVersion() {
		return version;
	}

	/**
	 * Sets the version (<code>0</code> or higher) of the cookie.  By default, <code>0</code>.
	 * 
	 * @param version The version (<code>0</code> or higher) of the cookie, otherwise <code>0</code>.
	 */
	public void setVersion(int version) {
		this.version = version;
	}

	/**
	 * Returns if the domain of the cookie contains an IP address.  By default, <code>false</code>.
	 *
	 * @return <code>true</code> if the domain of the cookie contains an IP address, otherwise <code>false</code>.
	 */
	public boolean isDomainIPAddress() {
		return isDomainIPAddress;
	}

	/**
	 * Sets if the domain of the cookie contains an IP address.  By default, <code>false</code>.
	 * 
	 * @param isDomainIPAddress <code>true</code> if the domain of the cookie contains an IP address, otherwise <code>false</code>.
	 */
	public void setDomainIPAddress(boolean isDomainIPAddress) {
		this.isDomainIPAddress = isDomainIPAddress;
	}
}
