/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HttpCookieCache.java,v 1.19 2010/05/06 18:21:58 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.http.runner;

import java.util.LinkedList;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.hyades.logging.events.cbe.CommonBaseEvent;
import org.eclipse.hyades.logging.events.cbe.ComponentIdentification;
import org.eclipse.hyades.logging.events.cbe.EventFactory;
import org.eclipse.hyades.logging.events.cbe.ReportSituation;
import org.eclipse.hyades.logging.events.cbe.Situation;
import org.eclipse.hyades.logging.events.cbe.impl.EventFactoryContext;
import org.eclipse.hyades.logging.java.CommonBaseEventLogRecord;
import org.eclipse.hyades.test.http.runner.internal.exec.Cookie;
import org.eclipse.hyades.test.http.runner.internal.exec.CookieLL;
import org.eclipse.hyades.test.http.runner.internal.exec.DomLL;
import org.eclipse.hyades.test.http.runner.internal.exec.URILL;
import org.eclipse.osgi.util.NLS;

/**
 * <p>Cache for cookies.</p>
 * 
 * <p>This class stores and retrieves cookies for HTTP requests.</p>
 * 
 * <p>For more information, see <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.</p>
 * 
 * 
 * @author  Suzanne Dirkers
 * @author  Paul Slauenwhite
 * @version May 6, 2010
 * @since   January 27, 2005
 * @provisional This is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
 */
public class HttpCookieCache {

	/**
	 * @deprecated This field is considered private.
	 */
	public LinkedList CookieCache = null;
	
	private Logger logger = null;
	
	/**
	 * <p>Static instance of the cookie <code>Expires</code> date format used in
	 * {@link #get_time_left(String)}.  For example, <code>01 July 2015 00:01:20 GMT</code>.</p>
	 * 
	 * <p><b>Note:</b> The Java {@link java.text.SimpleDateFormat} and {@link java.util.Locale} classes
	 * are used instead of the ICU {@link com.ibm.icu.text.SimpleDateFormat} and {@link com.ibm.icu.util.ULocale} 
	 * classes to work-around the ICU performance bug <a href="http://bugs.icu-project.org/trac/ticket/7081">7081</a> 
	 * (see TPTP defect <a href="https://bugs.eclipse.org/bugs/show_bug.cgi?id=277502">277502</a>).</p>
	 */
	private static final java.text.SimpleDateFormat COOKIE_EXPIRES_DATE_FORMAT = new java.text.SimpleDateFormat("dd MMM yy HH:mm:ss z", java.util.Locale.US); //$NON-NLS-1$

	private static final String[] DOMAINS = new String[]{".com", ".edu", ".net", ".org", ".gov", ".mil", ".int"};

	private static final Object LOCK = new Object();

	/**
	 * InvalidCookieException.java
	 * 
	 * 
	 * @author  Suzanne Dirkers
	 * @author  Paul Slauenwhite
	 * @version July 30, 2009
	 * @since   January 27, 2005
	 */
	public class InvalidCookieException extends Exception {
		
		private static final long serialVersionUID = -318166820860245163L;

		public InvalidCookieException(String message){
			super(message);
		}		
	}

	public HttpCookieCache() {
		CookieCache = new LinkedList();
	}

	/**
	 * This is the routine that sets dynamic cookies into the cache.
	 * @param request  HttpRequest object containing HTTP request
	 * @param response HttpResponse object containing HTTP response
	 */
	public void DynamicSetCookie(HttpRequest request, HttpResponse response)
			throws InvalidCookieException, Exception {
		int rc;
		Cookie chip;
		String setckhdr;
		setckhdr = new String("");
		/* for each Set-Cookie in HttpReponse: */
		// extract setckhdr out of HttpResponse
		// HttpHeader[] headers = response.getHeaders("Set-Cookie");
		HttpHeader[] headers = response.getHeaders("Set-Cookie");
		for (int i = 0; i < headers.length; i++) {
			try {
				setckhdr = new String(headers[i].getValue());
				// is there a set-cookie?
				//  if there is, is it already in the cache?
				//  if so and new one is obsolete, delete old, no replace
				//   if not and new one is obsolete, discard
				//  else: if new one not expired:
				//  if already exists, replace with this new one
				//  otherwise, add this new one
				//    To add this cookie
				//      find domain ( if new, add)
				//      find path ( if new, add)
				//      find cookie ( if new, add, else replace)
				chip = new Cookie();
				rc = parse_cookie(request, chip, setckhdr);
				if (rc == 0)
					rc = search_repl_cache(chip, request.getHost());
			} // try
			catch (InvalidCookieException ice) {

				cookielogger(Level.WARNING, NLS.bind(HttpResourceBundle.COOKIES_EXCEPTION, ice.getMessage()));
				
				throw ice;
			} 
			catch (Exception e) {

				cookielogger(Level.WARNING, e.getMessage());

				throw e;
			}
		}
		return;
	}
	/**
	 * This is the routine to get a dynamic cookie from the Cache
	 * @param request HttpRequest object containing request
	 * 
	 * This routine searches the CookieCache and returns those
	 * matching the request for host/domain and path and security
	 *   if they have not expired by the original Netscape cookie spec
	 *   guidelines.
	 * The cookies are returned by adding a Cookie: hdr to the
	 * request object.
	 * 
	 */
	public void DynamicGetCookie(HttpRequest request) {
		// find all the cookies on path and domain of request
		// collect in a StringBuffer
		// return in a HttpHeader object
		int i, j, k, saven;
		int psize, csize;
		LinkedList fndcookies, clist;
		
		try {
			StringBuffer ckbuf = new StringBuffer();
			fndcookies = new LinkedList();
			saven = -1;
			//search cookie headers in request object, put static cookies in a
			// string array
			for (i = 0; i < CookieCache.size(); i++) {
				if (domain_match(1, request.getHost(),
						((DomLL) CookieCache.get(i)).getdomain(), (DomLL)CookieCache.get(i)) != -1) {
					psize = ((LinkedList) ((DomLL) CookieCache.get(i)).getUriLL())
							.size();
					for (j = 0; j < psize; j++) {
						if (path_match(
								((URILL) ((DomLL) CookieCache.get(i)).getUriLL()
										.get(j)).geturipath(), request
										.getAbsolutePath()) != -1) {
							csize = ((LinkedList) ((URILL) ((DomLL) CookieCache
									.get(i)).getUriLL().get(j)).getcookieptr()).size();
							for (k = 0; k < csize; k++) {
								if (get_time_left(((CookieLL) ((URILL) ((DomLL) CookieCache
										.get(i)).getUriLL().get(j)).getcookieptr().get(k)).getCookie().getexpires()) != 1) {
									// bugzilla 141821, changed logic to correct an error where
									// non-ssl cookies were not included in an SSL connection request
									// The behavior should be to only exclude secure cookies from
									// being transmitted on a non-secure connection
									boolean bCookieSSLFlag = ((CookieLL) ((URILL) ((DomLL) CookieCache
											.get(i)).getUriLL().get(j)).getcookieptr()
											.get(k)).getCookie().getsslflag();
									if (bCookieSSLFlag == false || !(bCookieSSLFlag == true  && request
											.getSecure()== false)) {
										if (((CookieLL) ((URILL) ((DomLL) CookieCache
												.get(i)).getUriLL().get(j)).getcookieptr()
												.get(k)).getCookie().cval
												.endsWith("\r\n") == true) {
											//              make sure cookie ends in ;
											int cvllgn = ((CookieLL) ((URILL) ((DomLL) CookieCache
													.get(i)).getUriLL().get(j)).getcookieptr()
													.get(k)).getCookie().cval
													.length();
											String Trimmed = new String(
													((CookieLL) ((URILL) ((DomLL) CookieCache
															.get(i)).getUriLL()
															.get(j)).getcookieptr()
															.get(k)).getCookie().cval
															.substring(0,
																	cvllgn - 1));
											StringBuffer Trimmed_plus = new StringBuffer(
													Trimmed);
											Trimmed_plus.append(";");
											synchronized (LOCK) {
												((CookieLL) ((URILL) ((DomLL) CookieCache
														.get(i)).getUriLL().get(j)).getcookieptr()
														.get(k)).getCookie().cval = Trimmed_plus.toString();
											}
										}
										// add cookie in to linked list of
										// 'found' cookies
										fndcookies
												.add(((CookieLL) ((URILL) ((DomLL) CookieCache
														.get(i)).getUriLL().get(j)).getcookieptr()
														.get(k)).getCookie());
									}
								} else {
									synchronized (LOCK) {
										((LinkedList) ((URILL) ((DomLL) CookieCache
												.get(i)).getUriLL().get(j)).getcookieptr())
												.remove(k);
									}
								}
							}
						}
					}
				}
			}
			// put static cookies remaining in fndcookies if they aren't
			// duplicates
			HttpHeader[] reqhdrs = request.getHeaders();
			int eqi;
			String cnam = null;
			for (int n = 0; n < reqhdrs.length; n++) {
				if (reqhdrs[n].getName().equalsIgnoreCase("Cookie")) {
					saven = n;
					//separate out the list
					clist = sep_cookies(reqhdrs[n]);
					for (j = 0; j < clist.size(); j++) {
						if ((eqi = ((String) clist.get(j)).indexOf("=")) != -1)
							cnam = new String(((String) clist.get(j))
									.substring(0, eqi).trim());
						if (eqi != -1) {
							int m;
							boolean fnd = false;
							for (m = 0; m < fndcookies.size(); m++) {
								if ((((Cookie) fndcookies.get(m)).cname)
										.equals(cnam)) {
									fnd = true;
									break;
								}
							}
							if (fndcookies.size() == 0 || fnd == false) {
								/* invent new cookie */
								Cookie statc = new Cookie();
								makecapcookie((String) clist.get(j), statc,
										fndcookies, eqi, request, cnam);
							}
						}
					}
				}
			}
			// reorder cookies according to path (most specific to least)
			LinkedList newchain = new LinkedList();
			int flen = fndcookies.size();
			if (fndcookies.size() > 0) {
				newchain.add(fndcookies.get(0));
				int q;
				int qlen;
				for (int l = 1; l < flen; l++) {
					// put longest paths first, insertion sort
					qlen = newchain.size();
					for (q = 0; q < qlen; q++) {
						if (((Cookie) fndcookies.get(l)).getpath().length() >= ((Cookie) newchain
								.get(q)).getpath().length()) {
							newchain.add(q, fndcookies.get(l));
							break;
						}
					}
					if (q >= qlen)
						newchain.add(fndcookies.get(l));
				}
				// turn chain into StringBuffer
				for (int z = 0; z < newchain.size(); z++) {
					ckbuf.append(((Cookie) newchain.get(z)).getcookie());
					ckbuf.append(" ");
				}
			} else
				return;
			// make sure ckbuf ends in 0x0d0a instead of '; '
			ckbuf.delete(ckbuf.length() - 2, ckbuf.length() + 1);
			HttpHeader CookieHdr = new HttpHeader();
			CookieHdr.setName("Cookie");
			CookieHdr.setValue(ckbuf.toString());
			//check to see if there is already a Cookie hdr
			// if so replace
			if (saven >= 0) {
				// make sure we only return ONE header
				request.removeHeaders("Cookie");
				//  	 reqhdrs[saven].setValue(ckbuf.toString());
			}
			//  add CookieHdr to Request and return
			request.addHeader(CookieHdr);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return;
	}
	/**
	 * this routine turns a captured cookie into a cookie object and lists it
	 * with fndcookies
	 * 
	 * @deprecated This method is considered private.
	 */
	public void makecapcookie(String reqhdrs, Cookie statc,
			LinkedList fndcookies, int eqi, HttpRequest request, String cnam) {
		int sb;
		StringBuffer strb = new StringBuffer(reqhdrs.substring(eqi + 1));
		if (strb.toString().endsWith(";") == false) {
			if (strb.toString().endsWith("\r\n") == true) {
				sb = strb.toString().indexOf("\r\n");
				strb.replace(sb, sb + 2, "; ");
				strb.delete(sb + 1, sb + 2);
			}
			// neither \r\n nor ; ??
			else
				strb.append(";");
		}
		statc.setcookie(cnam, strb.toString());
		statc.setsslflag(request.getSecure());
		statc.setdomain(request.getHost());
		statc.setpath(request.getAbsolutePath());
		// add it into chain ( at beginning because path is whole url)
		fndcookies.addFirst(statc);
	}
	/**
	 * this routine separates the captured cookies in the request hdr
	 * 
	 * @deprecated This method is considered private.
	 */
	public LinkedList sep_cookies(HttpHeader reqhdr) {
		LinkedList clist = new LinkedList();
		StringTokenizer st = new StringTokenizer(reqhdr.getValue(), ";");
		while (st.hasMoreTokens()) {
			clist.add(st.nextToken() + ";");
		}
		return clist;
	}
	/**
	 *  This routine is for prepopulation ONLY.
	 * @param Name      String Name part of cookiename=value;
	 * @param Value     String Value of cookiename.
	 * @param Domain    String value of hostname or Domain name.
	 * @param Path      String Value of Path
	 * @param Expires   String value of expires= string
	 * @param secure    Boolean for secure flag 
	 * @return          0 if all went well, -1 for error
	 */
	public int AddCookie(String Name, String Value, String Domain, String Path,
			String Expires, boolean secure) {
		boolean isip;
		int rc;
		// is this cookie already in cache?
		// if so, replace,
		//  else add
		//first make this cookie
		isip = false;
		if (!(Domain == null)) {
			if (!(Domain.equals(""))) {
				if (is_ipaddr(Domain) == 1)
					isip = true;
				else
					isip = false;
			}
		}

		if(Name == null){
			cookielogger(Level.SEVERE, HttpResourceBundle.COOKIES_NONAME);
		}

		if(Domain == null){   			
			cookielogger(Level.SEVERE, HttpResourceBundle.COOKIES_NOHOST);
		}
		
		if (Path == null){   
			cookielogger(Level.SEVERE, HttpResourceBundle.COOKIES_NOPATH);
		}

		Cookie static_cookie = new Cookie(Name, Value, secure, Expires, Domain, Path, 0, isip);

		if (static_cookie.getpath() == null || static_cookie.getdomain() == null
				|| static_cookie.cname == null)
			return -1;
		rc = search_repl_cache(static_cookie, Domain);
		return rc;
	}
	
	/**
	 * <p><b>DO NOT USE!</b></p>
	 * 
	 * <p>This method is empty.</p>
	 * 
	 * @deprecated This method is considered private.
	 */
	public void deleteCookie(String Domain, String Path, String Name, String Value) {
		//No-operation.
	}
	
	/**
	 * this routine is for clearing a populated Cache
	 */
	public void clearCookieCache() {
		int i, CCS = CookieCache.size() - 1;
		int j, Urisize = 0;
		int k, CkCSize = 0;
		synchronized (LOCK) {
			for (i = CCS; i >= 0; i--) {
				Urisize = ((LinkedList) ((DomLL) CookieCache.get(i)).getUriLL())
						.size() - 1;
				for (j = Urisize; j >= 0; j--) {
					CkCSize = ((LinkedList) ((URILL) ((DomLL) CookieCache
							.get(i)).getUriLL().get(j)).getcookieptr()).size() - 1;
					for (k = CkCSize; k >= 0; k--) {
						((LinkedList) ((URILL) ((DomLL) CookieCache.get(i)).getUriLL()
								.get(j)).getcookieptr()).remove(k);
					}
					((LinkedList) ((DomLL) CookieCache.get(i)).getUriLL()).remove(j);
				}
				CookieCache.remove(i);
			}
		}
	}
	
	/**
	 * <p>Searches and replaces the cookie in the cookie cache.</p>
	 * 
	 * <p>The possible values for the return code are:
	 * <ul>
	 * <li><code>0</code>: The cookie is replaced in the cookie cache.</li>
	 * </ul>
	 * 
	 * @param cookie The cookie to be replaced in the cookie cache.
	 * @param host The host of the cookie.
	 * @return The return code.
	 * @deprecated This method is considered private.
	 */
	public int search_repl_cache(Cookie cookie, String host) {

		synchronized (LOCK) {

			int cookieExpiry = get_time_left(cookie.getexpires());
			boolean isDone = false;

			//Check if the cookie's domain is in the cache:
			int domainListSize = CookieCache.size();
			int domainIndex = 0;

			while((domainIndex < domainListSize) && (!isDone)){

				DomLL domainInfo = ((DomLL)(CookieCache.get(domainIndex)));

				if (domain_match(0, domainInfo.getdomain(), cookie.getdomain(), domainInfo) == 0) {

					//The domain was found but not an IP domain so check if host is an IP host and not in the cache:
					if ((is_ipaddr(domainInfo.getdomain()) == 0) && (is_ipaddr(host) == 1) && (!domainInfo.findip(host))){
						domainInfo.setreqhost(host);
					}

					//Check if the cookie's URI is in the cache:
					LinkedList uriList = ((LinkedList)(domainInfo.getUriLL()));
					int uriListSize = uriList.size();
					int uriIndex = 0;

					while((uriIndex < uriListSize) && (!isDone)){

						URILL uriInfo = ((URILL)(uriList.get(uriIndex)));

						if (path_match(uriInfo.geturipath(), cookie.getpath()) == 0) {

							//Check if the cookie is in the cache:
							LinkedList cookieList = ((LinkedList)(uriInfo.getcookieptr()));
							int cookieListSize = cookieList.size();
							int cookieIndex = 0;

							while((cookieIndex < cookieListSize) && (!isDone)){

								CookieLL cookieInfo = ((CookieLL)(cookieList.get(cookieIndex)));

								if (cookie_match(cookieInfo, cookie.cname) == 0) {

									if (cookieExpiry == 0){
										replace_cookie(domainIndex, uriIndex, cookieIndex, cookie);
									}
									else if (cookieExpiry == 1){
										cookieList.remove(cookieIndex);
									}

									isDone = true;
									break;
								}

								cookieIndex++;
							}

							if (cookieIndex >= cookieListSize) {

								if (cookieExpiry == 0){
									add_new_cookie(domainIndex, uriIndex, cookie);
								}

								isDone = true;
								break;
							}
						}

						uriIndex++;
					}

					if ((uriIndex >= uriListSize) && (!isDone)) {

						if (cookieExpiry == 0) {

							add_new_path(domainIndex, cookie);
							add_new_cookie(domainIndex, uriIndex, cookie);
						}

						isDone = true;
						break;
					}
				}

				domainIndex++;
			}

			if ((domainIndex >= domainListSize) && (!isDone) && (cookieExpiry == 0)) {

				add_new_domain(cookie, host);
				add_new_path(domainIndex, cookie);
				add_new_cookie(domainIndex, 0, cookie);
			}
		}

		return 0;
	}
	
	/**
	 * @param path1   String
	 * @param path2   String
	 * @return
	 * @deprecated This method is considered private.
	 */
	public int path_match(String path1, String path2) {

		if((path1 != null) && (path2 != null)){
			
			//does path1 = path 2 exactly?
			if (path1.equals(path2))
				return 0;
			//is path1 lexically contained in path 2?
			// remove trailing slashes from both if they exist
			StringBuffer tmppath1 = new StringBuffer(path1);
			StringBuffer tmppath2 = new StringBuffer(path2);
			if (path1.length() > 1) {
				if (tmppath1.charAt(tmppath1.length() - 1) == '/')
					tmppath1.deleteCharAt(tmppath1.length() - 1);
			}
			if (path2.length() > 1) {
				if (tmppath2.charAt(tmppath2.length() - 1) == '/')
					tmppath2.deleteCharAt(tmppath2.length() - 1);
			}
			String tpath1 = new String(tmppath1);
			String tpath2 = new String(tmppath2);
			if (tpath2.startsWith(tpath1))
				return 1;
		}
		
		return -1;
	}
	/**
	 * @param i  
	 * @param j
	 * @param k
	 * @param chip
	 * @deprecated This method is considered private.
	 */
	public void replace_cookie(int i, int j, int k, Cookie chip) {
		((CookieLL) ((URILL) ((DomLL) CookieCache.get(i)).getUriLL().get(j)).getcookieptr().get(k)).setCookie(chip);
	}
	/**
	 * @param cookiep
	 * @param ckienam
	 * @return
	 * @deprecated This method is considered private.
	 */
	public int cookie_match(CookieLL cookiep, String ckienam) {
		if (cookiep.getCookie().cname.equals(ckienam))
			return 0;
		else
			return -1;
	}
	
	/**
	 * <p>Determines the expiry of a cookie.</p>
	 * 
	 * <p>A valid <code>Expires</code> attribute is 
	 * <code>EEE, dd-MMM-yyyy HH:mm:ss z</code> 
	 * (see {@link java.text.SimpleDateFormat}).  For example, 
	 * <code>Thu, 01-July-2015 00:00:20 GMT</code>.  For more information, see section 10.1.2 
	 * of <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.</p>
	 * 
	 * <p>The expiry of a cookie is determined by parsing the
	 * <code>Expires</code> attribute to determine if expired.
	 * That is, the date and time is <b>greater</b> than the 
	 * date and time this method is invoked.</p>
	 * 
	 * <p>The possible values for the return code are:
	 * <ul>
	 * <li><code>-1</code>: An error occurred when parsing the <code>Expires</code> attribute.</li>
	 * <li><code>0</code>: The <code>Expires</code> attribute is <code>null</b> or <b>not</b> expired.</li>
	 * <li><code>1</code>: The <code>Expires</code> attribute is <code>EXP</b> or expired.</li>
	 * </ul>
	 * 
	 * @param expires The <code>Expires</code> attribute of the cookie.
	 * @return The return code. 
	 * @deprecated This method is considered private.
	 */
	public int get_time_left(String expires) {

		if (expires == null){
			return 0;
		}
		
		if (!expires.trim().equals("EXP")){ //$NON-NLS-1$
			
			int commaIndex = expires.indexOf(","); //$NON-NLS-1$
			
			//Handle an invalid (e.g. no comma) Expires attribute according to the Cookie Specification V0:
			if (commaIndex == -1) {
			
				//Find the first number (assumed to be the first digit of the day) and decrement the counter to second last character: 
				for (int characterIndex = 0; characterIndex < expires.length(); characterIndex++) {
					
					if (Character.isDigit(expires.charAt(characterIndex))){

						if (characterIndex < expires.length()){
							commaIndex = (characterIndex - 1);
						}
						
						break;
					}
				}
			}
			
			try {				
				
				//Remove the name of the day and the comma and handle an invalid (e.g. no dash separator between the day and month and month and year) Expires attribute according to the Cookie Specification V0:
				long expiresMillis = COOKIE_EXPIRES_DATE_FORMAT.parse(expires.substring(commaIndex + 1).trim().replace('-', ' ')).getTime();
				
				if (expiresMillis > System.currentTimeMillis()){
					return 0;
				}
			} 
			catch (Exception e) {
				
				cookielogger(Level.WARNING, NLS.bind(HttpResourceBundle.EXCEPTION_EXPIRES, e.getMessage())); //$NON-NLS-1$
	
				return -1;
			}
		}
		
		return 1;
	}
		
	/**
	 * @param dom1
	 *            first domain to compare
	 * @param dom2
	 *            second domain to compare
	 * @return 0 for perfect match, 1 for tail match, -1 for no match
	 * @deprecated This method is considered private.
	 */
	public int domain_match(int getflag, String dom1, String dom2, DomLL domentry) {
		int retc1;
		int retc2;
		String tdom;
		String domm1= new String(dom1);
		String domm2= new String (dom2);
		
		// is there a port? if so, ignore it
  		if (dom1.indexOf(':')!=-1){
			 domm1 = new String(dom1.substring(0,dom1.indexOf(':')));
		}
		if (dom2.indexOf(':')!= -1)
		{
			 domm2 = new String(dom2.substring(0,dom2.indexOf(':')));
		} 
		
		// do they match exactly?
		if (domm1.equalsIgnoreCase(domm2))
			return 0;
						
		// does domain 2 tail-match domain 1?
		// check that this is NOT an ipaddr...needs an alpha
		retc1 = is_ipaddr(domm1);
		retc2 = is_ipaddr(domm2);
		
		  if (retc1 ==1 & retc2 == 0) {
		 // check list of domain entry to see if this ip is listed
		 // if so, return 1 ( consider it a tail match)
		    if (domentry.findip(domm1))
		          return 1;
		  }
		 
		
		if (retc1 == 0 && retc2 == 0) {
			StringBuffer tempdom = new StringBuffer(".");
			if (!domm2.startsWith(".")) {
				tempdom.append(domm2.toLowerCase());
				tdom = new String(tempdom);
			} else
				tdom = new String(domm2.toLowerCase());
			if (domm1.length() > domm2.length())
				if (domm1.toLowerCase().endsWith(tdom))
					return 1;
		    if (getflag ==1)
		    {
		    	//  allow it if host is x.y.z and domain attribute was .x.y.z
		    	if (domm2.length() - domm1.length() == 1)
		    		if (domm2.toLowerCase().endsWith(domm1.toLowerCase()) && domm2.substring(0,1).equals("."))
		    			return 1;
		    }
		}
		return -1;
	}
	/**
	 * this is a quick check to see if a domain is specified by ip addr it does
	 * not go so far as to check that the actual numbers used in the addr are
	 * valid for an ip address. That is assumed to be the responsibility of the
	 * server
	 */
	protected int is_ipaddr(String dom) {
		char c;
		int ip = 1;
		for (int i = 0; i < dom.length(); i++) {
			if (!((c = dom.charAt(i)) >= '0' && (c <= '9') || c == '.')) {
				ip = 0;
				break;
			}
		}
		return ip;
	}
	
	/**
	 * <p>Adds a new domain to the cookie cache.</p>
	 * 
	 * <p>The possible values for the return code are:
	 * <ul>
	 * <li><code>0</code>: The new domain is added to the cookie cache.</li>
	 * </ul>
	 * 
	 * @param cookie The cookie used to create the new domain.
	 * @param host The host of the cookie.
	 * @return The return code.
	 * @deprecated This method is considered private.
	 */
	public int add_new_domain(Cookie cookie, String host) {
		
		DomLL dom = new DomLL();
		dom.setdomain(cookie.getdomain());
		dom.setipaddr(cookie.isip);
		
		//Set the requesting host if an IP address and the cookie's domain is an IP address:
		if ((is_ipaddr(host) == 1) && (is_ipaddr(cookie.getdomain()) == 0)){
			dom.setreqhost(host);
		}
		
		//Add the domain to the cookie cache:
		CookieCache.add(dom);
		
		return 0;
	}
	
	/**
	 * <p>Adds a new URI to a domain in the cookie cache.</p>
	 * 
	 * <p>The possible values for the return code are:
	 * <ul>
	 * <li><code>0</code>: The new URI is added to the domain in the cookie cache.</li>
	 * </ul>
	 * 
	 * @param domainIndex The index of the domain in the cookie cache.
	 * @param cookie The cookie used to create the new URI.
	 * @return The return code.
	 * @deprecated This method is considered private.
	 */
	public int add_new_path(int domainIndex, Cookie cookie) {
		
		DomLL domainInfo = ((DomLL)(CookieCache.get(domainIndex)));
		
		((LinkedList)(domainInfo.getUriLL())).add(new URILL(cookie.getpath(), domainInfo));
		
		return 0;
	}

	/**
	 * <p>Adds a new cookie to a URI in a domain in the cookie cache.</p>
	 * 
	 * <p>The possible values for the return code are:
	 * <ul>
	 * <li><code>0</code>: The new cookie is added to the URI in the domain in the cookie cache.</li>
	 * </ul>
	 * 
	 * @param domainIndex The index of the domain in the cookie cache.
	 * @param uriIndex The index of the URI in the domain.
	 * @param cookie The cookie used to create the new cookie.
	 * @return The return code.
	 * @deprecated This method is considered private.
	 */
	public int add_new_cookie(int domainIndex, int uriIndex, Cookie cookie) {
		
		URILL uriInfo = ((URILL)(((DomLL)(CookieCache.get(domainIndex))).getUriLL().get(uriIndex)));
		
		((LinkedList)(uriInfo.getcookieptr())).add(new CookieLL(cookie, 0, uriInfo));
		
		return 0;
	}
	
	/**
	 * This routine parses a cookie before setting it
	 * 
	 * @deprecated This method is considered private.
	 */
	public int parse_cookie(HttpRequest req, Cookie chip, String setckhdr)
			throws InvalidCookieException {
		int eqi;
		int rc = 0;
		int semi;
		int nneqi, nseqi, nnsmi = 0;
		boolean cflag = false;
		StringBuffer Message = null;
		Object[] messageArguments = {setckhdr,};
		String cv, ts;
		// save name=value in cookie object
		rc = 0;
		cflag = false;
		Message = new StringBuffer();
		if ((eqi = setckhdr.indexOf("=")) != -1) {
			//      back up to first non-whitespace
			ts = new String(setckhdr.substring(0, eqi).trim());
			//      get cookie value
			if ((semi = setckhdr.indexOf(";")) != -1)
				cv = new String(setckhdr.substring(eqi + 1, semi + 1).trim());
			else
			//           assume ONLY name=value but add ;
			{
				cv = new String(setckhdr.substring(eqi + 1).trim());
				StringBuffer sb1 = new StringBuffer(cv);
				sb1.append(";");
				cv = new String(sb1);
				semi = eqi + cv.length();
			}
		} else {
			Message.append(NLS.bind(HttpResourceBundle.COOKIES_NOVAL, messageArguments));
			InvalidCookieException ecv1 = new InvalidCookieException(Message
					.toString());
			throw ecv1;
		}
		chip.setcookie(ts, cv);
		chip.setversion(0);
		chip.setdomain(req.getHost());
		if (semi <= setckhdr.length()) {
			if ((findattr("Version", setckhdr, semi)) != -1)
			//  if ((vattr=setckhdr.substring(semi).indexOf("Version"))!= -1)
			{
				Message.append(HttpResourceBundle.COOKIES_VERSION.trim());
				InvalidCookieException ecv1 = new InvalidCookieException(
						Message.toString());
				throw ecv1;
			}
			if ((nneqi = findattr("domain", setckhdr, semi)) != -1)
			//  if ((nneqi=setckhdr.substring(semi).indexOf("domain"))!=-1)
			{
				if ((nseqi = setckhdr.substring(nneqi + semi).indexOf("=")) != -1) {
					nnsmi = setckhdr.substring(nneqi + semi).indexOf(";");
					if (nnsmi != -1)
						chip.setdomain(new String(setckhdr.substring(semi
								+ nneqi + nseqi + 1, semi + nneqi + nnsmi)));
					else
						chip.setdomain(new String(setckhdr.substring(
								semi + nneqi + nseqi + 1).trim()));
				}
			}
			else
			 {
                // is there a port? if so, ignore it
              String dom1 = req.getHost();
              if (dom1.indexOf(':')!=-1){
                 dom1 = new String(dom1.substring(0,dom1.indexOf(':')));
                }
              chip.setdomain(dom1);
              }
			if (is_ipaddr(chip.getdomain()) == 1)
				chip.isip = true;
			// if ((nneqi=setckhdr.substring(semi).indexOf("path"))!=-1)
			if ((nneqi = findattr("path", setckhdr, semi)) != -1) {
				if ((nseqi = setckhdr.substring(nneqi + semi).indexOf("=")) != -1) {
					nnsmi = setckhdr.substring(nneqi + semi).indexOf(";");
					if (nnsmi != -1)
						chip.setpath(new String(setckhdr.substring(semi + nneqi
								+ nseqi + 1, semi + nneqi + nnsmi)));
					else
						chip.setpath(new String(setckhdr.substring(
								semi + nneqi + nseqi + 1).trim()));
				}
			} else
			{
				 // make our best guess at what's really the path
				//chip.setpath(req.getAbsolutePath());
				String tmp_path = new String(req.getAbsolutePath());
				if (tmp_path.charAt(tmp_path.length()-1)!= '/')
				{	
					 tmp_path = new String(tmp_path.substring(0,tmp_path.lastIndexOf('/')+1));
				}	
				chip.setpath(tmp_path);
			}
			if ((findattr("secure", setckhdr, semi)) != -1)
				//   if ((vattr=setckhdr.substring(semi).indexOf("secure"))!=-1)
				chip.setsslflag(true);
			else
				chip.setsslflag(false);
			if ((nneqi = findattr("expires", setckhdr, semi)) != -1)
			//              if ((nneqi=setckhdr.substring(semi).indexOf("expires"))!=-1)
			{
				if ((nseqi = setckhdr.substring(semi + nneqi).indexOf("=")) != -1) {
					nnsmi = setckhdr.substring(semi + nneqi).indexOf(";");
					// bugzilla 150885, the expires tag may end without a semi-colon
					// Used Amazon.com for testing, their server does set-cookie without terminating with a semi-colon
					if (nnsmi == -1){
						nnsmi = setckhdr.substring(semi + nneqi).length();
					}
					if (nnsmi != -1){
						chip.setexpires(new String(setckhdr.substring(semi
								+ nneqi + nseqi + 1, semi + nneqi + nnsmi)));
						if (chip.getexpires().length()>0)
						{
							int ci = 0;
							for (ci=0; ci < chip.getexpires().length();ci++)
							{
								if (!chip.getexpires().substring(ci,ci+1).equals(" "))
									cflag = true;
							}
						}
					}
					else
						chip.setexpires(new String(setckhdr.substring(
								semi + nneqi + nseqi + 1).trim()));
					if (chip.getexpires().equals("") || chip.getexpires().equals(null) || cflag==false)
						  chip.setexpires(new String("Wed, 01-Jan-1970 00:00:00 GMT")); 
					// changed the above line to this default value for effectively deleting a cookie
					// in accordance with bugzilla 141852, previously, the string was set to "EXP"
				}
			}
		}
		//  Now that cookie is parsed...
		//    Check that domain is valid for this host
		/* this is commented out because it is contrary
		 * to our mission to police people for not following
		 * the rules
		 */
	  /*	if (chip.domain != null) {
			rc = check_dots(chip, Message);
			if (domset == 1 && rc == 0) {
			//	rc = check_ondom(req, chip, Message);
			}
		}*/
		if (rc != 0) {
			InvalidCookieException ecv = new InvalidCookieException(Message
					.toString());
			throw ecv;
		}
		return rc;
	}
	/**
	 * This routine specifically searches for attribute strings of cookies
	 * 
	 * @deprecated This method is considered private.
	 */
	public int findattr(String attr, String str, int semi) {
		int idx;
		int i;
		int accumidx = 0;
		boolean secfnd = false;
		boolean eqfnd = false;
		String tmpstr = new String(str.substring(semi));
		tmpstr = tmpstr.toLowerCase();
		while ((idx = tmpstr.indexOf(attr.toLowerCase())) != -1) {
			accumidx = accumidx + idx;
			eqfnd = false;
			secfnd = false;
			for (i = idx + attr.length(); i < tmpstr.length(); i++) {
				if (tmpstr.substring(i, i + 1).equals(" "))
					continue;
				if (attr.toLowerCase().equals("secure")) {
					// parse back looking for blanks then ;
					secfnd = parse_secure(idx, tmpstr);
					if (secfnd == true)
						break;
				} else if (tmpstr.substring(i, i + 1).equals("="))
					eqfnd = true;
				break;
			}
			if (eqfnd == true || parse_secure(idx, tmpstr) == true)
				return accumidx;
			if (i >= tmpstr.length())
				break;
			accumidx = accumidx + attr.length();
			tmpstr = new String(tmpstr.substring(i));
		}
		return -1;
	}
	/**
	 * This routine is checking that a 'secure' attribute is really an attribute
	 * 
	 * @param idx
	 * @param tmpstr
	 * @return
	 * @deprecated This method is considered private.
	 */
	public boolean parse_secure(int idx, String tmpstr) {
		int j;
		j = idx;
		if (j < 1)
			return false;
		while (tmpstr.substring(j - 1, j).equals(" ") && j > 0)
			j = j - 1;
		if (tmpstr.substring(j - 1, j).equals(";"))
			return true;
		else
			return false;
	}
	/**
	 * This routine checks that a host setting a cookie belongs to the domain of 
	 * the cookie.
	 * 
	 * @deprecated This method is considered private.
	 */
	public int check_ondom(HttpRequest req, Cookie chip, StringBuffer Message) {
		int retc1;
		String tdom;
		Object[] messageArguments = {req.getHost(), chip.getdomain(),};
		// Do the domains match exactly?
		if (chip.getdomain().toLowerCase().equals(req.getHost().toLowerCase()))
			return 0;
		// Is the Host on the domain that was specified? (better not be an
		// ipaddr)
		retc1 = is_ipaddr(req.getHost());
		StringBuffer tempdom = new StringBuffer(".");
		if (!chip.getdomain().startsWith(".")) {
			tempdom.append(chip.getdomain());
			tdom = new String(tempdom);
		} else
			tdom = new String(chip.getdomain());
		if (chip.getdomain().length() < req.getHost().length()) {
			if ((retc1 == 0 && !chip.isip)
					&& (req.getHost().toLowerCase()
							.endsWith(tdom.toLowerCase())))
				return 0;
			else {
				Message.append(NLS.bind(HttpResourceBundle.COOKIES_NOTONDOMAIN, messageArguments));
				return -1;
			}
		} else {
			Message.append(NLS.bind(HttpResourceBundle.COOKIES_NOTONDOMAIN, messageArguments));
			return -1;
		}
	}
	/**
 	 * this routine checks that the domain of a cookie has an appropriate number
	 * of dots.
	 * 
	 * @deprecated This method is considered private.
	 */
	public int check_dots(Cookie chip, StringBuffer Message) {
		Object[] messageArguments = {chip.getdomain(),};
		if (chip.getdomain().length() >= 4) {
			for (int i = 0; i < DOMAINS.length; i++) {
				if (chip.getdomain().substring(chip.getdomain().length() - 4)
						.equalsIgnoreCase(DOMAINS[i])) {
					if (chip.getdomain().indexOf(".") < chip.getdomain().lastIndexOf("."))
						return 0;
					else {
						//  this is being commented out because IE doesn't care
						// even though this is AGAINST the cookie spec
						//  Je proteste! Someday some virus will exploit this, I
						// predict...
						//  Message.append(" Error: Domain must have at least 2
						// dots for special domain = " + chip.domain);
						// return -1;
						return 0; // allow anyway
					}
				}
			}
		}
		int cnt = 0;
		for (int i = 0; i < chip.getdomain().length(); i++) {
			if (chip.getdomain().substring(i, i + 1).equals("."))
				cnt++;
		}
		// if NO dots, assume domain is local
		if (cnt < 3 && cnt > 0) {
			Message.append(NLS.bind(HttpResourceBundle.COOKIES_3DOTS, messageArguments));
			return -1;
		}
		return 0;
	}
	
	/**
	 * <p>Logs a message at a logging level to the <code>CookieLog</code> logger.</p>
	 * 
	 * <p>The default logging level is {@link Level#WARNING}.
	 * 
	 * <p>The <code>logging.properties</code> file (see {@link Logger}) may define a logger named <code>CookieLog</code> and its logging properties.</p>
	 * 
	 * @param level The logging level ({@link Level#ALL}, {@link Level#CONFIG}, {@link Level#FINE}, {@link Level#FINER}, {@link Level#FINEST}, {@link Level#INFO}, {@link Level#OFF}, {@link Level#SEVERE}, or {@link Level#WARNING}).
	 * @param message The log message.
	 * @deprecated This method is considered private.
	 */
	public void cookielogger(Level level, String message) {

		try {

			//Initialize the CookieLog logger:
			if(logger == null){

				//Retrieve the CookieLog logger:
				logger = Logger.getLogger("CookieLog"); //$NON-NLS-1$

				//Set the logging level to WARNING:
				logger.setLevel(Level.WARNING);
			}

			if(logger != null){

				if(logger.isLoggable(level)){

					//Retrieve the instance of the Simple Event Factory:
					EventFactory eventFactory = EventFactoryContext.getInstance().getEventFactoryHome("org.eclipse.hyades.logging.events.cbe.impl.SimpleEventFactoryHomeImpl").getEventFactory("org.eclipse.hyades.logging.test.http.runner.HttpCookieCache");  //$NON-NLS-1$ //$NON-NLS-2$

					//Create a new instance of a report situation:
					ReportSituation reportSituation = eventFactory.createReportSituation();
					reportSituation.setReasoningScope("INTERNAL"); //$NON-NLS-1$
					reportSituation.setReportCategory("LOG"); //$NON-NLS-1$

					//Create a new instance of a situation:
					Situation situation = eventFactory.createSituation();
					situation.setCategoryName("ReportSituation"); //$NON-NLS-1$
					situation.setSituationType(reportSituation);

					//Create a new instance of a Source Component:
					ComponentIdentification sourceComponentIdentification = eventFactory.createComponentIdentification();
					sourceComponentIdentification.setLocation("localhost"); //$NON-NLS-1$
					sourceComponentIdentification.setLocationType("IPV4"); //$NON-NLS-1$
					sourceComponentIdentification.setExecutionEnvironment("Java"); //$NON-NLS-1$
					sourceComponentIdentification.setComponent("HyadesTestHttp"); //$NON-NLS-1$
					sourceComponentIdentification.setSubComponent("HyadesTestHttpCookieCache"); //$NON-NLS-1$
					sourceComponentIdentification.setComponentIdType("Application"); //$NON-NLS-1$
					sourceComponentIdentification.setComponentType("HyadesHttpCookieCache"); //$NON-NLS-1$

					//Create a new instance of a Common Base Event:
					CommonBaseEvent CommonBaseEvent = eventFactory.createCommonBaseEvent();
					CommonBaseEvent.setSourceComponentId(sourceComponentIdentification);
					CommonBaseEvent.setSituation(situation);
					CommonBaseEvent.setCreationTimeAsLong(System.currentTimeMillis());
					CommonBaseEvent.setSeverity(((short) (20)));
					CommonBaseEvent.setMsg(message);

					//Create a new instance of a Common Base Event log record:
					CommonBaseEventLogRecord commonBaseEventLogRecord = new CommonBaseEventLogRecord(CommonBaseEvent);
					commonBaseEventLogRecord.setLoggerName("CookieLog"); //$NON-NLS-1$
					commonBaseEventLogRecord.setLevel(level);

					//Log the Common Base Event log record:
					logger.log(commonBaseEventLogRecord);
				}
			}
			else if((level == null) || (Level.WARNING.equals(level) || (Level.SEVERE.equals(level)) || (Level.ALL.equals(level)))){
				System.out.println(message);
			}
		} 
		catch (Exception e) {
			e.printStackTrace();
		}
	}	
}