/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HttpElement.java,v 1.2 2005/02/25 22:17:53 hleung Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.http.runner;

import java.util.ArrayList;

/**
 * Abstract class with the common attributes of an HTTP request and response.
 * 
 * @author marcelop
 * @author jnevicos
 * @since 1.0.2
 */
public abstract class HttpElement
{
	private String id;
	private String version;
	
	/**header collection has been changed from Map because there can be multiple headers with the same name,
	* for example, multiple "Set-Cookie"s.  Searching has become less optimized by removing the Map, but in general ALL headers will
	* be returned at once.
	* 
	* <b>headers</b> is an unsorted array of HttpHeader objects. 
	*/
	private ArrayList headers;
	
	
	private String body;
					
	public HttpElement()
	{
		id = Integer.toString(hashCode());
	}
	
	public void addHeader(HttpHeader header)
	{
		if(HttpHeader.isValid(header))
		{
			if(headers == null)
				headers = new ArrayList();
			headers.add(header);
			header.setElement(this);
		}		
	}
	
	public String getId()
	{
		return id;
	}
	
	public String getVersion()
	{
		return version;
	}
	
	public void setVersion(String version)
	{
		this.version = version;
	}

	/** Method to return all headers
	 * 
	 * @return all headers currently in the HttpElement in a HttpHeader[].
	 * If no headers exist, it returns HttpHeader[] of length 0.
	 */
	public HttpHeader[] getHeaders()
	{
		if(headers == null)
			return new HttpHeader[0];
		/*
		 * return (HttpHeader[])headers.values().toArray(new HttpHeader[headers.values().size()]);
		*/		
		return (HttpHeader[]) headers.toArray(new HttpHeader[headers.size()]);		
	}
	
	
	/**
	 * Search for a particular type of header.
	 * @param name - String to search headers with
	 * @return An HttpHeader[] containing all HttpHeaders with this name
	 */
	public HttpHeader[] getHeaders(String name)
	{
		if (name == null || headers == null)
			return new HttpHeader[0];
		
		ArrayList found = new ArrayList();
		for (int i = 0; i < headers.size(); i++)
		{
			if ( ((HttpHeader)headers.get(i)).getName().equalsIgnoreCase(name))
				found.add((HttpHeader)headers.get(i));
		}
		return (HttpHeader[]) found.toArray(new HttpHeader[found.size()]);
		
		
	}
	
	 /**
	 /* this routine is used to make sure only 1 modified cookie header is on request
	  * 
	  * @param name  the name of the headers to remove
	  */
	public void removeHeaders(String name)
	  {
	  	if (name == null || headers == null)
	  	   return;
	  	   
		
	    for (int i = headers.size()-1; i >=0; i--)
		{
	       if ( ((HttpHeader)headers.get(i)).getName().equalsIgnoreCase(name))
				headers.remove((HttpHeader)headers.get(i));
		}
	  	
	  	return;
	  }
	/** 
	 * @param name - String to search headers for
	 * @return String[] containing all header values with the matching name.  
	 * If none are found, it returns a String[] of length 0.
	 */
	public String[] getHeaderValues(String name)
	{
		HttpHeader[] myheaders = getHeaders(name);
		if(myheaders == null)
			return new String[0];
		String[] myvals = new String[myheaders.length];
		for (int i = 0; i < myheaders.length; i++)
			myvals[i] = myheaders[i].getValue();
			
		return myvals;
	}
	
	public int[] getHeaderIntValues(String name)
	{
		HttpHeader[] myheaders = getHeaders(name);
		if(myheaders == null)
			return new int[0];
		int[] myvals = new int[myheaders.length];
		for (int i = 0; i < myheaders.length; i++)
		{
			try
			{
				myvals[i] = Integer.parseInt(myheaders[i].getValue());
			}
			catch(NumberFormatException nfe)
			{
				myvals[i] = -1;
			}
		}
		return myvals;
	}
	
	public void setBody(String body)
	{
		this.body = body;
	}

	public String getBody()
	{
		return body;
	}		
}
