/*******************************************************************************
 * Copyright (c) 2009, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Cookie.java,v 1.7 2010/05/06 18:22:02 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.http.runner.internal.exec;

/**
 * <p>Represents state information for HTTP requests commonly
 * referred to as a cookie.</p>
 * 
 * <p>For more information, see <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.</p>
 * 
 * <p><b>Note</b>: This class' APIs will NOT change to maintain forward and backward compatibility.</p>
 * 
 * 
 * @author  Suzanne Dirkers
 * @author  Paul Slauenwhite
 * @version May 6, 2010
 * @since   August 11, 2009
 */
public class Cookie{

	/**
	 * <p>The cookie in the form of:</p>
	 * 
	 * <code>&lt;name&gt; = &lt;value&gt;</code>
	 * 
	 * <p>By default, <code>null</code>.</p>
	 */
	public String cookie = null;
	
	/**
	 * <p>The name of the cookie.  By default, <code>null</code>.</p>
	 *
	 * <p>If the name of the cookie is <code>null</code>, the value of the cookie is ignored.</p>
	 */
	public String cname = null;

	/**
	 * <p>The value of the cookie.  By default, <code>null</code>.</p>
	 * 
	 * <p>If the name of the cookie is <code>null</code>, the value of the cookie is ignored.</p>
	 */
	public String cval = null;

	/**
	 * <code>true</code> if the cookie is secure.  By default, <code>false</code>.
	 */
	public boolean sslflag = false;
	
	/**
	 * <p>The expiry date of the cookie.  By default, <code>null</code>.</p>
	 * 
	 * <p>The format of the expiry date is <code>EEE, dd-MMM-yyyy HH:mm:ss z</code> (see {@link java.text.SimpleDateFormat}).  For example, <code>Thu, 01-July-2015 00:00:20 GMT</code>.  For more information, see section 10.1.2 of <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.</p>
	 */
	public String expires = null;

	/**
	 * The host (name or IP address) or domain of the cookie.  By default, <code>null</code>.
	 */
	public String domain = null;

	/**
	 * The path on the host or domain of the cookie.  By default, <code>null</code>.
	 */
	public String path = null;

	/**
	 * The version (<code>0</code> or higher) of the cookie.  By default, <code>0</code>.
	 */
	public int version = 0;

	/**
	 * <code>true</code> if the domain of the cookie contains an IP address.  By default, <code>false</code>.
	 */
	public boolean isip = false;

	/**
	 * <p>Creates a new cookie with the default properties.</p>
	 * 
	 * <p>For more information on the properties of a cookie, see <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.</p>
	 * 
	 * <p>By default, the new cookie:</p>
	 * 
	 * <ul>
	 * <li>has <code>null</code> string,</li> 
	 * <li>has a <code>0</code> version,</li> 
	 * <li>is not secure,</li>
	 * <li>and the domain is not an IP address.</li>
	 * </ul> 
	 */
	public Cookie() {
		//No-operation.
	}

	/**
	 * <p>Creates a new cookie with the specified properties.</p>
	 * 
	 * <p>For more information on the properties of a cookie, see <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.</p>
	 *  
	 * @param name The name of the cookie, otherwise <code>null</code>.
	 * @param value The value of the cookie, otherwise <code>null</code>.  If the name of the cookie is <code>null</code>, the value of the cookie is ignored.
	 * @param isSecure <code>true</code> if the cookie is secure, otherwise <code>false</code>.
	 * @param expires The expiry date of the cookie, otherwise <code>null</code>.  The format of the expiry date is <code>EEE, dd-MMM-yyyy HH:mm:ss z</code> (see {@link java.text.SimpleDateFormat}).  For example, <code>Thu, 01-July-2015 00:00:20 GMT</code>.  For more information, see section 10.1.2 of <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.
	 * @param domain The host (name or IP address) or domain of the cookie, otherwise <code>null</code>.
	 * @param path The path on the host or domain of the cookie, otherwise <code>null</code>.
	 * @param version The version (<code>0</code> or higher) of the cookie, otherwise <code>0</code>.
	 * @param isDomainIPAddress <code>true</code> if the domain of the cookie contains an IP address, otherwise <code>false</code>.
	 */
	public Cookie(String name, String value, boolean isSecure, String expires, String domain, String path, int version, boolean isDomainIPAddress){

		setcookie(name, value);
         
		this.sslflag = isSecure;
		this.expires = expires;
		this.domain = domain;
		this.path = path;
		this.version = version;
		this.isip = isDomainIPAddress;
	}

	/**
	 * <p>Separates the cookie into the name and value.  The cookie is 
	 * in the form of:</p>
	 * 
	 * <code>&lt;name&gt; = &lt;value&gt;</code>
	 * 
	 * <p>By default, <code>null</code>.</p>
	 *
	 * <p>If the cookie is not well-formed, the cookie (including the name and value of the cookie) are set to <code>null</code>.</p>
	 * 
	 * @param cookie The cookie, otherwise <code>null</code>.
	 * @return <code>0</code> if the cookie was set successfully, otherwise <code>1</code>.
	 */
	public int sepcookie(String cookie){

		if(cookie != null){
			
			int equalsIndex = cookie.indexOf('=');

			if (equalsIndex != -1){

				cname = cookie.substring(0, equalsIndex);
				cval = cookie.substring(equalsIndex + 1);

				return 0;
			}
		}

		cookie = null;
		cname = null;
		cval = null;

		return 1;
    }

	/**
	 * <p>Returns the cookie in the form of:</p>
	 * 
	 * <code>&lt;name&gt; = &lt;value&gt;</code>
	 * 
	 * <p>By default, <code>null</code>.</p>
	 * 
	 * @return The cookie, otherwise <code>null</code>.
	 */
	public String getcookie(){
		return cookie;
    }

    /**
	 * <p>Sets the cookie in the form of:</p>
	 * 
	 * <code>&lt;name&gt; = &lt;value&gt;</code>
	 * 
	 * <p>By default, <code>null</code>.</p>
	 *
	 * <p>If the cookie is not well-formed, the cookie (including the name and value of the cookie) are set to <code>null</code>.</p>
	 * 
	 * @param cookie The cookie, otherwise <code>null</code>.
	 * @return <code>0</code> if the cookie was set successfully, otherwise <code>1</code>.
	 */
    public int setcookie(String cookie){
        
    	this.cookie = cookie;
    	
    	return (sepcookie(cookie));
    }

    /**
	 * <p>Sets the cookie (including the namd and value of the cookie) in the form of:</p>
	 * 
	 * <code>&lt;name&gt; = &lt;value&gt;</code>
	 * 
	 * <p>By default, <code>null</code>.</p>
	 *
	 * <p>If the name of the cookie is <code>null</code>, the cookie (including the value of the cookie) is set to <code>null</code>.</p>
	 * 
	 * @param name The name of the cookie, otherwise <code>null</code>.
	 * @param value The value of the cookie, otherwise <code>null</code>. 
	 */
    public void setcookie(String name, String value){
        
		this.cname = name;
		this.cval = value;

         if (name != null){
            cookie = new String(name + "=" + value); //$NON-NLS-1$
         }
         else{
        	 cookie = null;
        	 cval = null;   
         }
    }
    
	/**
	 * Returns if the cookie is secure.  By default, <code>false</code>.
	 *
	 * @return <code>true</code> if the cookie is secure, otherwise <code>false</code>.
	 */
	public boolean getsslflag() {
		return sslflag;
	}

	/**
	 * Sets if the cookie is secure.  By default, <code>false</code>.
	 * 
	 * @param isSecure <code>true</code> if the cookie is secure, otherwise <code>false</code>.
	 */
	public void setsslflag(boolean isSecure) {
		this.sslflag = isSecure;
	}
	
	/**
	 * <p>Returns the expiry date of the cookie.  By default, <code>null</code>.</p>
	 * 
	 * <p>The format of the expiry date is <code>EEE, dd-MMM-yyyy HH:mm:ss z</code> (see {@link java.text.SimpleDateFormat}).  For example, <code>Thu, 01-July-2015 00:00:20 GMT</code>.  For more information, see section 10.1.2 of <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.</p>
	 *
	 * @return The expiry date of the cookie, otherwise <code>null</code>.
	 */
	public String getexpires() {
		return expires;
	}

	/**
	 * <p>Sets the expiry date of the cookie.  By default, <code>null</code>.</p>
	 * 
	 * <p>The format of the expiry date is <code>EEE, dd-MMM-yyyy HH:mm:ss z</code> (see {@link java.text.SimpleDateFormat}).  For example, <code>Thu, 01-July-2015 00:00:20 GMT</code>.  For more information, see section 10.1.2 of <a href="http://www.w3.org/Protocols/rfc2109/rfc2109.txt">HTTP State Management Mechanism</a>.</p>
	 * 
	 * @param expires The expiry date of the cookie, otherwise <code>null</code>.
	 */
	public void setexpires(String expires) {
		this.expires = expires;
	}

	/**
	 * Returns the host (name or IP address) or domain of the cookie.  By default, <code>null</code>.
	 *
	 * @return The host (name or IP address) or domain of the cookie, otherwise <code>null</code>.
	 */
	public String getdomain() {
		return domain;
	}

	/**
	 * Sets the host (name or IP address) or domain of the cookie.  By default, <code>null</code>.
	 * 
	 * @param domain The host (name or IP address) or domain of the cookie, otherwise <code>null</code>.
	 */
	public void setdomain(String domain) {
		this.domain = domain;
	}

	/**
	 * Returns the path on the host or domain of the cookie.  By default, <code>null</code>.
	 *
	 * @return The path on the host or domain of the cookie, otherwise <code>null</code>.
	 */
	public String getpath() {
		return path;
	}

	/**
	 * Sets the path on the host or domain of the cookie.  By default, <code>null</code>.
	 * 
	 * @param path The path on the host or domain of the cookie, otherwise <code>null</code>.
	 */
	public void setpath(String path) {
		this.path = path;
	}

	/**
	 * Returns version (<code>0</code> or higher) of the cookie.  By default, <code>0</code>.
	 *
	 * @return The version (<code>0</code> or higher) of the cookie, otherwise <code>0</code>.
	 */
	public int getversion() {
		return version;
	}

	/**
	 * Sets the version (<code>0</code> or higher) of the cookie.  By default, <code>0</code>.
	 * 
	 * @param version The version (<code>0</code> or higher) of the cookie, otherwise <code>0</code>.
	 */
	public void setversion(int version) {
		this.version = version;
	}
}
