/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RequestHelper.java,v 1.3 2007/05/02 19:36:24 paules Exp $
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.tools.core.http.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.models.common.facades.behavioral.IMethod;
import org.eclipse.hyades.models.common.facades.behavioral.IProperty;
import org.eclipse.hyades.models.common.facades.behavioral.ITargetInvocation;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.impl.HyadesFactory;
import org.eclipse.hyades.test.core.util.CoreUtil;
import org.eclipse.hyades.test.http.runner.HttpHeader;
import org.eclipse.hyades.test.http.runner.HttpRequest;
import org.eclipse.hyades.test.tools.core.common.TestCommon;

/**
 * @author marcelop
 * @since 1.0.2
 */
public class RequestHelper
{
	public final static int ATT_HOST = 1;
	public final static int ATT_PORT = 2;
	public final static int ATT_ABS_PATH = 3;
	public final static int ATT_BODY = 4;
	public final static int ATT_METHOD = 5;
	public final static int ATT_VERSION = 6;
	// aberns:
	public final static int ATT_THINK_TIME = 7;
	
	//jnevicos
	public final static int ATT_PAGE_NUM = 8;
	public final static int ATT_PAGE_ORDER = 9;
	public final static int ATT_CONN_ID = 10;
	public final static int ATT_CONN_ORDER = 11;
	public final static int ATT_IS_SSL = 12;
		
	private final static String PROPERTY_DESC_ATT = "DESC_ATT"; 
	private final static String PROPERTY_DESC_HEADER = "DESC_HEADER";
	
	private ITestCase testCase;
	private ITargetInvocation targetInvocation;
	private boolean testCaseModified = false;
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		testCase = null;
		targetInvocation = null;
		testCaseModified = false;
	}
	
	public void setTestCase(ITestCase testCase)
	{
		dispose();
		if(TestCommon.HTTP_JUNIT_TEST_CASE_TYPE.equals(testCase.getType()))
			this.testCase = testCase;
	}
	
	public boolean isDirty()
	{
		return testCaseModified;
	}

	public void clearDirty()
	{
		testCaseModified = false;
	}
	
	public ITestCase getTestCase()
	{
		return testCase;
	}
	
	public ITargetInvocation getTargetInvocation()
	{
		if(targetInvocation != null)
			return targetInvocation;
			
		if((testCase != null) && (testCase.getImplementor() != null) && (testCase.getImplementor().getBlock() != null))
		{
			for(Iterator i = testCase.getImplementor().getBlock().getActions().iterator(); i.hasNext();)
			{
				Object action = i.next();
				if(action instanceof ITargetInvocation)
				{
					targetInvocation = (ITargetInvocation)action;
					break;			
				}
			}						
		}
		return targetInvocation;
	}
	
	public ITargetInvocation createTargetInvocation()
	{
		if(testCase == null)
			return null;
		
		if(testCase.getImplementor() == null)
			HyadesFactory.INSTANCE.createImplementor(testCase, false);
			
		targetInvocation = HyadesFactory.INSTANCE.createTargetInvocation(null);
		testCase.getImplementor().getBlock().getActions().add(targetInvocation); 
		testCaseModified = true;
		return targetInvocation;
	}
	
	public IMethod getMethod()
	{
		ITargetInvocation targetInvocation = getTargetInvocation();
		if(targetInvocation == null)
			return null;
			
		return targetInvocation.getInvocation();
	}
	
	public void setMethod(IMethod method)
	{
		if(testCase == null)
			return;

		ITargetInvocation targetInvocation = getTargetInvocation();
		if(targetInvocation == null)
		{
			if(method == null)
				return;
			targetInvocation = createTargetInvocation();
		}
		else
		{
			IMethod currentMethod = getMethod();
			if(currentMethod != null)
				testCase.getImplementor().getMethods().remove(currentMethod);
		}
		
		if(method != null)
			testCase.getImplementor().getMethods().add(method);
			
		targetInvocation.setInvocation(method);
		testCaseModified = true;
	}
	
	public String getAttribute(int flag)
	{
		ITargetInvocation targetInvocation = getTargetInvocation();
		if(targetInvocation == null)
			return null;
		
		String attribute = null;
		switch(flag)
		{
			case ATT_ABS_PATH:
				attribute = "abs_path";
				break;
				
			case ATT_BODY:
				attribute = "body";
				break;
				
			case ATT_HOST:
				attribute = "host";
				break;
				
			case ATT_PORT:
				attribute = "port";
				break;

			case ATT_VERSION:
				attribute = "version";
				break;

			case ATT_METHOD:
				IMethod method = getMethod();
				if(method != null)
					return method.getName();
				return null;

			// aberns:
			case ATT_THINK_TIME:
				attribute = "think_time";
				break;
			
			//jnevicos							
			case ATT_PAGE_NUM :
				attribute = "page_num";
				break;
			case ATT_PAGE_ORDER :
				attribute = "page_order";
				break;
			case ATT_CONN_ID :
				attribute = "conn_id";
				break;
			case ATT_CONN_ORDER :
				attribute = "conn_order";
				break;
			case ATT_IS_SSL :
				attribute = "is_ssl";
				break;							
			default:
				return null;
		}
		
		for (Iterator i = targetInvocation.getActionProperties().getProperties().iterator(); i.hasNext();)
		{
			IProperty prop = (IProperty)i.next();
			if(PROPERTY_DESC_ATT.equals(prop.getDescription()) && attribute.equals(prop.getName()))
				return prop.getValue();
		}
		
		return null;	
	}
	
	public void setAttribute(int flag, String value)
	{
		if(testCase == null)
			return;
			
		if("".equals(value))
			value = null;
			
		String attribute = null;
		switch(flag)
		{
			case ATT_ABS_PATH:
				attribute = "abs_path";
				break;
				
			case ATT_BODY:
				attribute = "body";
				break;
				
			case ATT_HOST:
				attribute = "host";
				break;
				
			case ATT_PORT:
				attribute = "port";
				int port = parseInteger(value);
				if(port <= 0)
					value = null;
				break;

			case ATT_VERSION:
				attribute = "version";
				break;

			case ATT_METHOD:
				if(value == null)
				{
					setMethod(null);
				}
				else
				{
					IMethod method = getMethod();
					if(method == null)
					{
						method = HyadesFactory.INSTANCE.createMethod();
						setMethod(method);
					}
					method.setName(value);
					testCaseModified = true;
				}
				return;
			//aberns:
			case ATT_THINK_TIME :
				attribute = "think_time";
				int nThink = parseInteger(value);
				if( nThink < -1 )
					value = "0";
				break;
								
			//jnevicos
			case ATT_PAGE_NUM :
				attribute = "page_num";
				
				int nPage = parseInteger(value);
				if (nPage < 0)
					value = "0"; 
				break;
			case ATT_PAGE_ORDER :
				attribute = "page_order";
				int nPageOrd = parseInteger(value);
				if (nPageOrd != HttpRequest.PAGE_START && nPageOrd!= HttpRequest.PAGE_MID 
				 && nPageOrd != HttpRequest.PAGE_END && nPageOrd != HttpRequest.PAGE_ONLY)
					value = Integer.toString(HttpRequest.PAGE_UNDEFINED);
				break;
			case ATT_CONN_ID :
				attribute = "conn_id";
				int nconn = parseInteger(value);
				if (nconn < 0)
					value = "0"; 
				break;
				
			case ATT_CONN_ORDER :
				attribute = "conn_order";
				int nconnOrd = parseInteger(value);
				if (nconnOrd != HttpRequest.CONN_START && nconnOrd != HttpRequest.CONN_MID && nconnOrd != HttpRequest.CONN_END)
					value = Integer.toString(HttpRequest.CONN_UNDEFINED);
				break;

			case ATT_IS_SSL :
				attribute = "is_ssl";
				break;
			default:
				return;
		}

		ITargetInvocation targetInvocation = getTargetInvocation();
		if(targetInvocation == null)
		{
			if(value == null)
				return;
			targetInvocation = createTargetInvocation();
		}

		IProperty property = null;
		for (Iterator i = targetInvocation.getActionProperties().getProperties().iterator(); i.hasNext();)
		{
			IProperty prop = (IProperty)i.next();
			if(PROPERTY_DESC_ATT.equals(prop.getDescription()) && attribute.equals(prop.getName()))
			{
				property = prop;
				break;
			}				
		}

		if(property != null)
		{
			if(value != null)
				property.setValue(value);
			else
				targetInvocation.getActionProperties().getProperties().remove(property);
				
			testCaseModified = true;
		}
		else if(value != null)
		{
			IProperty prop = HyadesFactory.INSTANCE.createProperty();
			prop.setName(attribute);
			prop.setValue(value);
			prop.setDescription(PROPERTY_DESC_ATT);
			targetInvocation.getActionProperties().getProperties().add(prop);
			testCaseModified = true;
		}
	}

	
	public String getHeader(String header)
	{
		ITargetInvocation targetInvocation = getTargetInvocation();
		if(targetInvocation == null)
			return null;

		if(header != null)
		{
			for (Iterator i = targetInvocation.getActionProperties().getProperties().iterator(); i.hasNext();)
			{
				IProperty prop = (IProperty)i.next();
				if(PROPERTY_DESC_HEADER.equals(prop.getDescription()) && header.equals(prop.getName()))
					return prop.getValue();
			}
		}
		
		return null;	
	}

	public IProperty[] getHeaders()
	{
		ITargetInvocation targetInvocation = getTargetInvocation();
		if(targetInvocation == null)
			return new IProperty[0];

		List properties = new ArrayList();
		for (Iterator i = targetInvocation.getActionProperties().getProperties().iterator(); i.hasNext();)
		{
			IProperty prop = (IProperty)i.next();
			if(PROPERTY_DESC_HEADER.equals(prop.getDescription()))
				properties.add(prop);
		}
		return (IProperty[])properties.toArray(new IProperty[properties.size()]);	
	}

	public void swapHeaders(IProperty header1, IProperty header2)
	{
		if((header1 == null) || (header2 == null))
			return;

		ITargetInvocation targetInvocation = getTargetInvocation();
		if(targetInvocation == null)
			return;
		
		List properties = targetInvocation.getActionProperties().getProperties();
			
		int index1 = properties.indexOf(header1);
		if(index1 < 0)
			return;

		int index2 = properties.indexOf(header2);
		if(index2 < 0)
			return;
			
		CoreUtil.swap(properties, index1, index2);
	}
	
	public void addHeader(IProperty header)
	{
		if(header == null)
			return;		

		ITargetInvocation targetInvocation = getTargetInvocation();
		if(targetInvocation == null)
		{
			targetInvocation = createTargetInvocation();
		}
		else
		{
			if(targetInvocation.getActionProperties().getProperties().contains(header))
				return;
		}
		
		header.setDescription(PROPERTY_DESC_HEADER);
		targetInvocation.getActionProperties().getProperties().add(header);
		testCaseModified = true;
	}

	public void removeHeader(IProperty header)
	{
		if(header == null)
			return;		

		ITargetInvocation targetInvocation = getTargetInvocation();
		if(targetInvocation == null)
			return;
		
		testCaseModified = targetInvocation.getActionProperties().getProperties().remove(header);
	}
	
	public void setHeader(String header, String value)
	{
		if(header == null)
			return;
			
		if("".equals(value))
			value = null;
			
		ITargetInvocation targetInvocation = getTargetInvocation();
		if(targetInvocation == null)
		{
			//if(value == null)  //jnevicos - value can be empty, but header cannot
			if(header == null)
				return;
			targetInvocation = createTargetInvocation();
		}

		for (Iterator i = targetInvocation.getActionProperties().getProperties().iterator(); i.hasNext();)
		{
			IProperty prop = (IProperty)i.next();
			if(PROPERTY_DESC_HEADER.equals(prop.getDescription()) && header.equals(prop.getName()))
			{
				prop.setValue(value);
				testCaseModified = true;
				return;
			}				
		}
		
		if(value != null)
		{
			IProperty prop = HyadesFactory.INSTANCE.createProperty();
			prop.setName(header);
			prop.setValue(value);
			prop.setDescription(PROPERTY_DESC_HEADER);
			targetInvocation.getActionProperties().getProperties().add(prop);
			testCaseModified = true;
		}
	}
		
	/**
	 * Creates a request object that is used by the HTTP runner. 
	 * @return HttpRequest
	 */
	public HttpRequest createHttpRequest()
	{
		if(getTargetInvocation() == null)
			return null;
			
		HttpRequest request = new HttpRequest();
		
		request.setAbsolutePath(getAttribute(ATT_ABS_PATH));
		request.setHost(getAttribute(ATT_HOST));
		request.setMethod(getAttribute(ATT_METHOD));
		request.setBody(getAttribute(ATT_BODY));
		request.setVersion(getAttribute(ATT_VERSION));
		try
		{
			request.setPort(Integer.parseInt(getAttribute(ATT_PORT)));	
		}
		catch (NumberFormatException e)
		{
		}
		
		//aberns
		try
		{
			request.setThinkTime(Long.parseLong(getAttribute(ATT_THINK_TIME)));	
		}
		catch (NumberFormatException e)
		{
			request.setThinkTime( HttpConstants.DEFAULT_THINK_TIME );
		}
		
		//jnevicos
		try 
		{
			request.setPageNumber(Integer.parseInt(getAttribute(ATT_PAGE_NUM)));
		}
		catch (NumberFormatException e) {
			request.setPageNumber(0);
		}
		
		try
		{
			request.setPageOrder(Integer.parseInt(getAttribute(ATT_PAGE_ORDER)));
		}
		catch (NumberFormatException e) {
			request.setPageOrder(HttpRequest.PAGE_UNDEFINED);
		}
		
		try {
			request.setConnID(Integer.parseInt(getAttribute(ATT_CONN_ID)));
			
		}
		catch (NumberFormatException e) {
			request.setConnID(0);
		}
		try {
		
		request.setConnOrder(Integer.parseInt(getAttribute(ATT_CONN_ORDER)));
		}
		catch (NumberFormatException e) {
			request.setConnOrder(HttpRequest.CONN_UNDEFINED);			
		}
		request.setSecure("true".equalsIgnoreCase(getAttribute(ATT_IS_SSL)));
		
		IProperty[] headers = getHeaders();
		for (int i = 0, maxi = headers.length; i < maxi; i++)
		{
			HttpHeader header = new HttpHeader();
			header.setName(headers[i].getName());
			header.setValue(headers[i].getValue());
			request.addHeader(header);
		}
		
		return request;
	}

	//convenience method to parseInt
	protected int parseInteger(String value) {
		try
		{
			return Integer.parseInt(value);
		}
		catch (NumberFormatException e)
		{
			return Integer.MIN_VALUE;
		}

	}

}
