/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HyadesJUnitRunner.java,v 1.16 2009/05/22 17:01:41 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.java.runner;

import java.io.PrintStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Vector;

import junit.extensions.RepeatedTest;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import org.eclipse.hyades.execution.core.IControlMessage;
import org.eclipse.hyades.internal.execution.local.common.CustomCommand;
import org.eclipse.hyades.internal.execution.remote.RemoteComponentSkeleton;
import org.eclipse.hyades.test.common.event.ExecutionEvent;
import org.eclipse.hyades.test.common.event.MessageEvent;
import org.eclipse.hyades.test.common.junit.DefaultTestArbiter;
import org.eclipse.hyades.test.common.junit.HyadesTestRunner;
import org.eclipse.hyades.test.common.junit.HyadesTestSuite;
import org.eclipse.hyades.test.common.runner.internal.util.AgentConsoleStream;
import org.eclipse.hyades.test.common.util.BaseString;
import org.eclipse.osgi.util.NLS;

/**
 * <p>HyadesJUnitRunner.java</p>
 * 
 * 
 * @author  Paul Slauenwhite
 * @version May 22, 2009
 * @since   January 27, 2005
 */
public class HyadesJUnitRunner extends HyadesTestRunner {
	
	protected AgentConsoleStream stdout = null;
	protected AgentConsoleStream stderr = null;
	protected static final String UTF8 = "UTF-8"; //$NON-NLS-1$
	public static final String LOADTESTSUITEPREFIX = "_@USERGROUP_"; //$NON-NLS-1$
	
	public static Test getTest(Class clazz) {
		
		try {

			//Note: Ignore the following compiler warning (see defect #277479):
			//      The argument of type null should explicitly be cast to <type> for the invocation of the varargs method <mthod> from type <type>. It could alternatively be cast to <type> for a varargs invocation
			return ((Test)(clazz.getMethod("suite", null).invoke(null, null))); //$NON-NLS-1$
		} 
		catch (InvocationTargetException e) {
			// The method was succesfully invoked, but it has thrown an exception.
			// Report it.
			return createErrorTest("Exception thrown by suite() invocation", e.getCause());
		} 
		catch (Exception e) {
			// The class has no "suite" method, or something prevented from using
			// it. Use the reflection feature of TestSuite constructor instead. This
			// method will return appropriate diagnostic if the passed argument is invalid.
			return new TestSuite(clazz);
		}
	}
	
	protected static Test createFailTest(String testName, final String message) {
		TestSuite ts = new TestSuite();
		ts.addTest(new TestCase(testName) {
			protected void runTest() {
				fail(message);
			}
		});
		return ts;
	}
	
	protected static Test createErrorTest(String testName, final Throwable e) {
		TestSuite ts = new TestSuite();
		ts.addTest(new TestCase(testName) {
			protected void runTest() throws Throwable {
				throw e;
			}
		});
		return ts;
	}
	

	public static void main(String[] args)
	{
		HyadesJUnitRunner runner = new HyadesJUnitRunner(args);

		try
		{
			/* Setup stdout and stder */
			/* bugzilla_140715: this must be done before the call to getTest(), because
			 * this may invoke user code (in suite() method) that has stdout/stderr output.
			 */
			runner.stdout = new AgentConsoleStream(runner.agent, AgentConsoleStream.OUT, null, runner._peekParentEventID());
			runner.stderr = new AgentConsoleStream(runner.agent, AgentConsoleStream.ERR, null, runner._peekParentEventID());
			System.setOut(new PrintStream(runner.stdout, false, UTF8));
			System.setErr(new PrintStream(runner.stderr, false, UTF8));

			/* We need to locate the testsuite for our IDs */
			// bugzilla 84790: changed forName to use runner.testClass rather than arg[0] directly
			Class c = Class.forName(runner.testClass, true, HyadesJUnitRunner.class.getClassLoader());
			Test test = getTest(c);

			/* Is this a LoadTest? */
			int instances = 1;
			String sInstances = System.getProperty("hyades.loadtest.nusers"); //$NON-NLS-1$
			if (sInstances != null) {
				try {
					instances = Integer.valueOf(sInstances).intValue();
				}
				catch (Exception e) {
					instances = 1;
				}
				
			}
			
			if (instances <= 1) {
				/* Run the testcase */
				runner.run(test);
			}
			else if (test instanceof HyadesTestSuite) {
				/* This is a LoadTest. Use the ThreadRunner to run multiple
				 * instances simultaneously.
				 */
				HyadesTestSuite suite = (HyadesTestSuite)test;
				runner.agent.sendMessageToAttachedClient("Running a " + instances + " instance test", 0);
				
				// Create a loadTestSuite
				HyadesTestSuite loadTest = createLoadTestSuite(suite, instances);
				
				// Run the loadtest Suite
				runner.run(loadTest);
			}

			runner.agent.sendMessageToAttachedClient("Testcase completed successfuly", 0);
			runner.agent.sendMessageToAttachedClient(IControlMessage.STOP, 0);
			
			//Flush out the system output and error stream:
			System.out.flush();
			System.err.flush();
		}
		catch(Throwable e)
		{
			
			//Flush out the system output and error stream:
			System.out.flush();
			System.err.flush();

			/* We could not find the class (or something else, report the error */
			runner.reportException(e);
		}
	}
	
	protected void reportException(Throwable t) {
		MessageEvent messageEvent = new MessageEvent();
		messageEvent.setText(BaseString.getStackTrace(t));
		messageEvent.setSeverity(MessageEvent.ERROR);
		//bugzilla 84790: changed setOwnerId to use runner.testId rather than args[1] directly;
		messageEvent.setOwnerId(testID);
		messageEvent.setParentId("ROOT"); //$NON-NLS-1$
		agent.logMessageUTF8(messageEvent.toString());
	}
	
	public void handleCustomCommand(CustomCommand command){	}
	
	/**
	 * Flush any data that is cached in the sysout and syserr consoles
	 */
	protected void flushConsoles()
	{
		if ( stdout != null )
			stdout.flush();
		
		if ( stderr != null )
			stderr.flush();
	}

	/**
	 * Constructor for HyadesJUnitRunner
	 * @param agent
	 */
	public HyadesJUnitRunner(RemoteComponentSkeleton agent)
	{
		super(agent);
	}
	
	public HyadesJUnitRunner(String[] args)
	{
		this(args, null);
		
	}

	public HyadesJUnitRunner(String[] args, Vector commandHandler)
	{
		super(args, commandHandler);	
	}
	
	/**
	 * @see org.eclipse.hyades.test.common.junit.HyadesTestRunner#runnerStarted()
	 */
	protected void runnerStarted()
	{
		writeExecEvent("<EXECUTION>"); //$NON-NLS-1$
		super.runnerStarted();
	}

	/**
	 * @see org.eclipse.hyades.test.common.junit.HyadesTestRunner#runnerExit(boolean, long)
	 */
	protected void runnerExit(boolean runnerStopped, long elapsedTime)
	{
		super.runnerExit(runnerStopped, elapsedTime);
		writeExecEvent("</EXECUTION>"); //$NON-NLS-1$
	}

	/**
	 * @see org.eclipse.hyades.test.common.junit.HyadesTestRunner#writeEvent(org.eclipse.hyades.test.common.event.ExecutionEvent)
	 */
	public void writeEvent(ExecutionEvent executionEvent)
	{
		flushConsoles();
		
		writeExecEvent(executionEvent);
	}
	
	/**
	 * @see org.eclipse.hyades.test.common.junit.HyadesTestRunner#getLastEventText(boolean)
	 */
	protected String getLastEventText(boolean runnerStopped, long elapsedTime)
	{
		String[] args = new String[]
		{
			new Integer(getSucceedTests().size() + getFailureTests().size() + getErrorTests().size()).toString(), 
			new Integer(getFailureTests().size()).toString(), 
			new Integer(getErrorTests().size()).toString(),
			new Long(elapsedTime).toString()
		};

		if(runnerStopped)
			return NLS.bind(JUnitResourceBundle.execution_TestStoped, args); 
		return NLS.bind(JUnitResourceBundle.execution_TestsFinished, args); 
	}
	
	protected static HyadesTestSuite createLoadTestSuite(HyadesTestSuite suite, int instances) {
		
		String id = suite.getId();
//		HyadesTestSuite root = new HyadesTestSuite("root");
//		root.setArbiter(DefaultTestArbiter.INSTANCE).setId(id).setSynchronous(false);
//		
//		HyadesTestSuite usergroup = new HyadesTestSuite("usergroup");
//
//		root.addTest(new RepeatedTest(suite, instances));
//
//		return root;
		
		HyadesTestSuite root = new HyadesTestSuite("root"); //$NON-NLS-1$
		root.setArbiter(DefaultTestArbiter.INSTANCE).setId(id + "HTTP_ROOTID"); //$NON-NLS-1$

		HyadesTestSuite usergroup = new HyadesTestSuite(LOADTESTSUITEPREFIX + instances);
		root.addTest(new RepeatedTest(usergroup, instances));
		usergroup.setId(id + "_HTTP_USERGROUP").setSynchronous(false); //$NON-NLS-1$
		usergroup.addTest(suite);
		
		return root;
	
	}

}