/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ASTHelper.java,v 1.3 2007/05/02 19:36:26 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.core.internal.common.codegen;

import java.util.Iterator;

import org.eclipse.hyades.test.tools.core.internal.common.codegen.Helper.LineIterator;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTMatcher;
import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.ASTParser;
import org.eclipse.jdt.core.dom.ASTVisitor;
import org.eclipse.jdt.core.dom.AbstractTypeDeclaration;
import org.eclipse.jdt.core.dom.BodyDeclaration;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.ImportDeclaration;
import org.eclipse.jdt.core.dom.Javadoc;
import org.eclipse.jdt.core.dom.MemberRef;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.MethodRef;
import org.eclipse.jdt.core.dom.MethodRefParameter;
import org.eclipse.jdt.core.dom.Modifier;
import org.eclipse.jdt.core.dom.Name;
import org.eclipse.jdt.core.dom.QualifiedName;
import org.eclipse.jdt.core.dom.SimpleName;
import org.eclipse.jdt.core.dom.SingleVariableDeclaration;
import org.eclipse.jdt.core.dom.TagElement;
import org.eclipse.jdt.core.dom.TextElement;
import org.eclipse.jdt.core.dom.TypeDeclaration;

/**
 * Utilities for manipulating AST trees.
 * @author jcanches
 * @since 4.3
 */
public class ASTHelper {

	/**
	 * Resolves a name to a qualified name, in the context of a given compilation unit,
	 * by looking at the imports.
	 */
	public static Name resolveName(CompilationUnit cu, Name name) {
		if (name.isSimpleName()) {
			SimpleName simpleName = (SimpleName)name;
			Iterator it = cu.imports().iterator();
			while (it.hasNext()) {
				ImportDeclaration id = (ImportDeclaration) it.next();
				if (matchName(simpleName, id.getName())) return id.getName(); 
			}
			return null;
		}
		return name;
	}

	private static boolean matchName(SimpleName simple, Name complete) {
		SimpleName unqualifiedName = null;
		if (complete.isSimpleName()) {
			unqualifiedName = (SimpleName)complete;
		}
		if (complete.isQualifiedName()) {
			QualifiedName qn = (QualifiedName) complete;
			unqualifiedName = qn.getName();
		}
		if (unqualifiedName != null) {
			return simple.getIdentifier().equals(unqualifiedName.getIdentifier());
		}
		return false;
	}
	
	/**
	 * Determines whether a compilation unit has a given import.
	 */
	public static boolean hasImport(CompilationUnit cu, String qualifiedName) {
		for (Iterator it = cu.imports().iterator(); it.hasNext();) {
			ImportDeclaration id = (ImportDeclaration)it.next();
			if (qualifiedName.equals(id.getName().getFullyQualifiedName())) {
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Parses a method declaration and returns the resulting AST element.
	 * @param ast The AST that the result must belong to.
	 * @param methodDeclaration A method declaration (as a string).
	 * @return An AST MethodDeclaration object.
	 */
	public static MethodDeclaration parseMethod(AST ast, String methodDeclaration) {
		ASTParser parser = ASTParser.newParser(ast.apiLevel());
		parser.setSource(methodDeclaration.toCharArray());
		parser.setKind(ASTParser.K_CLASS_BODY_DECLARATIONS);
		TypeDeclaration localType = (TypeDeclaration) parser.createAST(/*progress monitor*/null);
		MethodDeclaration method = (MethodDeclaration) localType.bodyDeclarations().get(0);
		// The resulting method must be cloned to the proper AST owner
		return (MethodDeclaration) ASTNode.copySubtree(ast, method);
	}
	
	/**
	 * Returns the method with 0 parameters matching the given name in the given type.
	 * @param type
	 * @param name
	 * @return
	 */
	public static MethodDeclaration findMethodWithNoParameter(TypeDeclaration type, String name) {
		MethodDeclaration[] methods = type.getMethods();
		for (int i = 0; i < methods.length; i++) {
			if (name.equals(methods[i].getName().getIdentifier()) && methods[i].parameters().isEmpty()) {
				return methods[i];
			}
		}
		return null;
	}
	
	/**
	 * Returns the method matching the name and the parameter types of a pattern.
	 * Return type, nor parameter names are taken into account.
	 * @param type The type where method is searched
	 * @param pattern A pattern. Only method name, and parameters types are taken into account.
	 * @return The method within <code>type</code> that matches pattern, or <code>null</code>
	 * if not found.
	 */
	public static MethodDeclaration findMethod(TypeDeclaration type, MethodDeclaration pattern) {
		MethodDeclaration[] methods = type.getMethods();
		for (int i = 0; i < methods.length; i++) {
			if (pattern.getName().getIdentifier().equals(methods[i].getName().getIdentifier()) && pattern.parameters().size() == methods[i].parameters().size()) {
				// Same name, same number of formal parameters:
				for (Iterator it1 = pattern.parameters().iterator(),
					          it2 = methods[i].parameters().iterator(); it1.hasNext();) {
					SingleVariableDeclaration patDecl = (SingleVariableDeclaration)it1.next();
					SingleVariableDeclaration matDecl = (SingleVariableDeclaration)it2.next();
					if (!patDecl.getType().subtreeMatch(new ASTMatcher(), matDecl.getType())) {
						break;
					}
				}
				return methods[i];
			}
		}
		return null;
	}
	
	/**
	 * Returns the index, among all the type members, of the first method within a
	 * type declaration.
	 * @param type A type declaration.
	 * @return The index of the first method, or -1 if there is no method within <code>type</code>.
	 */
	public static int getFirstMethodIndex(TypeDeclaration type) {
		int cpt;
		Iterator it;
		for (cpt = 0, it = type.bodyDeclarations().iterator(); it.hasNext(); cpt++) {
			BodyDeclaration bd = (BodyDeclaration)it.next();
			if (bd instanceof MethodDeclaration) {
				return cpt;
			}
		}
		return -1;
	}
	
	/**
	 * Returns the only public, first-level type declaration found in a compilation unit.
	 * @param cu A Compilation Unit.
	 * @return The "main" type, or <code>null</code> if no such type exists.
	 */
	public static TypeDeclaration getMainType(CompilationUnit cu) {
		for (Iterator it = cu.types().iterator(); it.hasNext(); ) {
			AbstractTypeDeclaration atd = (AbstractTypeDeclaration)it.next();
			if (atd.isPackageMemberTypeDeclaration()
				&& (atd.getModifiers() & Modifier.PUBLIC) != 0
				&& atd instanceof TypeDeclaration) {
				return (TypeDeclaration)atd;
			}
		}
		return null;
	}
	
	private static TagElement getJavadocDescription (String methodSource) {
		ASTParser parser = ASTParser.newParser(AST.JLS3);
		parser.setKind(ASTParser.K_CLASS_BODY_DECLARATIONS);
		parser.setSource(methodSource.toCharArray());
		TypeDeclaration type = (TypeDeclaration) parser.createAST(null);
		MethodDeclaration md = (MethodDeclaration) type.bodyDeclarations().get(0);
		Javadoc javadoc = md.getJavadoc();
		if (javadoc != null && javadoc.tags().size() > 0) {
			TagElement tag = (TagElement) javadoc.tags().get(0);
			if (tag.getTagName() == null) {
				return tag;
			}
		}
		return null;
	}
	
	public static TagElement getJavadocDescription(IMethod method) throws JavaModelException {
		return getJavadocDescription(method.getSource());
	}
	
	public static TagElement parseJavadocDescription(AST ast, String javadoc) {
		String input = "/**" + javadoc + "*/void f(){}"; //$NON-NLS-1$ //$NON-NLS-2$
		TagElement tag = getJavadocDescription(input);
		return (TagElement)ASTNode.copySubtree(ast, tag);
	}
	
	public static boolean compareJavadocDescriptions(TagElement tag1, TagElement tag2) {
		return (tag1 == null && tag2 == null)
			|| (tag1 != null && tag1.subtreeMatch(new ASTMatcher(true), tag2));
	}
	
	public static String extractDescription(TagElement tag) {
		JavatagFlattener flattener = new JavatagFlattener();
		tag.accept(flattener);
		return flattener.getResult();
	}
	
	static class JavatagFlattener extends ASTVisitor {

		private StringBuffer buf = new StringBuffer();
		
		public String getResult() {
			return buf.toString();
		}
		
		public boolean visit(MemberRef node) {
			if (node.getQualifier() != null) {
				node.getQualifier().accept(this);
			}
			buf.append("#");//$NON-NLS-1$
			node.getName().accept(this);
			return false;
		}

		public boolean visit(MethodRef node) {
			if (node.getQualifier() != null) {
				node.getQualifier().accept(this);
			}
			buf.append("#");//$NON-NLS-1$
			node.getName().accept(this);
			buf.append("(");//$NON-NLS-1$
			for (Iterator it = node.parameters().iterator(); it.hasNext(); ) {
				MethodRefParameter e = (MethodRefParameter) it.next();
				e.accept(this);
				if (it.hasNext()) {
					buf.append(",");//$NON-NLS-1$
				}
			}
			buf.append(")");//$NON-NLS-1$
			return false;
		}

		public boolean visit(SimpleName node) {
			buf.append(node.getIdentifier());
			return false;
		}
		
		public boolean visit(QualifiedName node) {
			node.getQualifier().accept(this);
			buf.append(".");//$NON-NLS-1$
			node.getName().accept(this);
			return false;
		}

		public boolean visit(TagElement node) {
			if (node.isNested()) {
				// nested tags are always enclosed in braces
				buf.append("{");//$NON-NLS-1$
			}
			boolean previousRequiresWhiteSpace = false;
			if (node.getTagName() != null) {
				buf.append(node.getTagName());
				previousRequiresWhiteSpace = true;
			}
			boolean previousRequiresNewLine = false;
			for (Iterator it = node.fragments().iterator(); it.hasNext(); ) {
				ASTNode e = (ASTNode) it.next();
				// assume text elements include necessary leading and trailing whitespace
				// but Name, MemberRef, MethodRef, and nested TagElement do not include white space
				boolean currentIncludesWhiteSpace = (e instanceof TextElement);
				if (previousRequiresNewLine && currentIncludesWhiteSpace) {
					buf.append("\n");//$NON-NLS-1$
				}
				previousRequiresNewLine = currentIncludesWhiteSpace;
				// add space if required to separate
				if (previousRequiresWhiteSpace && !currentIncludesWhiteSpace) {
					buf.append(" "); //$NON-NLS-1$
				}
				e.accept(this);
				previousRequiresWhiteSpace = !currentIncludesWhiteSpace && !(e instanceof TagElement);
			}
			if (node.isNested()) {
				buf.append("}");//$NON-NLS-1$
			}
			return false;
		}

		public boolean visit(TextElement node) {
			buf.append(node.getText());
			return false;
		}
		
	}

	public static void addDescriptionToJavadoc(Javadoc javadoc, String string) {
		LineIterator it = new Helper.LineIterator(string, " \t"); //$NON-NLS-1$
		int idx = 0;
		while(it.hasNextLine()) {
			TagElement tag = javadoc.getAST().newTagElement();
			TextElement text = javadoc.getAST().newTextElement();
			text.setText(it.getNextLine());
			tag.fragments().add(text);
			javadoc.tags().add(idx++, tag);
		}
		
	}
	
}
