/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AutomaticDependencyUpdater.java,v 1.6 2007/04/26 18:49:54 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.core.internal.common.codegen;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.test.tools.core.internal.util.PluginProjectUtil;

/**
 * Dependency updater that selects automatically the best-suited updater for the
 * project to update.
 * @author jcanches
 * @since 4.2
 */
public class AutomaticDependencyUpdater implements IProjectDependencyUpdater {

	private class PluginJarPair {
		public PluginJarPair(String pluginId, String jarName) {
			this.pluginId = pluginId;
			this.jarName = jarName;
		}
		public String pluginId;
		public String jarName;
	}
	
	private Collection requiredPluginJars;
	private Collection requiredProjects;
	
	public void addRequiredPlugin(String pluginId, String jarName) {
		if (requiredPluginJars == null) requiredPluginJars = new HashSet();
		requiredPluginJars.add(new PluginJarPair(pluginId, jarName));
	}

	public void addRequiredProject(IProject project) {
		if (requiredProjects == null) requiredProjects = new HashSet();
		requiredProjects.add(project);
	}

	private IProjectDependencyUpdater createDelegate(IProject project) {
		IProjectDependencyUpdater delegate;
		if (PluginProjectUtil.isStrictPluginProject(project)) {
			delegate = new PluginProjectDependencyUpdater();
		} else {
			delegate = new JavaProjectDependencyUpdater();
		}
		if (requiredPluginJars != null) {
			for (Iterator it = requiredPluginJars.iterator(); it.hasNext();) {
				PluginJarPair pair = (PluginJarPair)it.next();
				delegate.addRequiredPlugin(pair.pluginId, pair.jarName);
			}
		}
		if (requiredProjects != null) {
			for (Iterator it = requiredProjects.iterator(); it.hasNext();) {
				delegate.addRequiredProject((IProject)it.next());
			}
		}
		return delegate;
	}
	
	public void adjustProject(IProject project, IProgressMonitor monitor) throws CoreException {
		if (requiredPluginJars == null && requiredProjects == null) return;
		IProjectDependencyUpdater delegate = createDelegate(project);
		delegate.adjustProject(project, monitor);
	}

	public Collection previewAdjustProject(IProject project) {
		if (requiredPluginJars == null && requiredProjects == null) return Collections.EMPTY_SET;
		IProjectDependencyUpdater delegate = createDelegate(project);
		return delegate.previewAdjustProject(project);
	}

}
