/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ClasspathContainerInitializer.java,v 1.5 2007/05/03 01:09:08 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.core.internal.common.codegen;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.hyades.test.tools.core.internal.common.CommonPluginMessages;
import org.eclipse.jdt.core.IClasspathContainer;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.osgi.util.ManifestElement;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleException;
import org.osgi.framework.Constants;

/**
 * This class implements the "TPTP Libraries" container initializer. It is responsible
 * of converting container entries of the form
 * "org.eclipse.hyades.test.tools.core.TPTP_LIBRARIES/list",
 * where list is a comma-separated list of plugin ids, into a ClasspathContainer.
 * @author jcanches
 * @since 4.3
 */
public class ClasspathContainerInitializer extends org.eclipse.jdt.core.ClasspathContainerInitializer {
	
	public final static String CONTAINER_ID = "org.eclipse.hyades.test.tools.core.TPTP_LIBRARIES"; //$NON-NLS-1$

	/**
	 * The "TPTP Libraries" classpath container.
	 * This container is parametrized by a list of plugins, i.e. it returns, as classpath
	 * entries, all jars declared in the runtime section of each plugin.
	 */
	protected static class ClasspathContainer implements IClasspathContainer {

		private IClasspathEntry[] classpathEntries;
		private String args;
		
		public ClasspathContainer(String args) {
			Collection ret = new ArrayList();
			this.args = args;
			String[] plugins = getPluginNames();
			for (int i = 0; i < plugins.length; i++) {
				Bundle bundle = Platform.getBundle(plugins[i]);
				if (bundle != null) {
					try {
						IPath[] paths = getPluginLibraries(bundle);
						for (int j = 0; j < paths.length; j++) {
							IClasspathEntry entry = JavaCore.newLibraryEntry(paths[j], null, null);
							ret.add(entry);
						}
					} catch (Throwable t) {
						CorePlugin.logError(t);
					}
				}
			}
			classpathEntries = (IClasspathEntry[]) ret.toArray(new IClasspathEntry[ret.size()]);
		}
		
		public String[] getPluginNames() {
			return args.split(","); //$NON-NLS-1$
		}
		
		public boolean contains(Collection pluginNames) {
			String[] plugins = getPluginNames();
			return Arrays.asList(plugins).containsAll(pluginNames);
		}
		
		public String getArgs() {
			return args;
		}
		
		private static IPath[] getPluginLibraries(Bundle bundle) throws BundleException, IOException {
			URL url = FileLocator.resolve(bundle.getEntry("")); //$NON-NLS-1$
			IPath pluginPath = new Path(url.getFile());
			if (url.getProtocol().equals("jar")) { //jar in the plugin root //$NON-NLS-1$
				String file = url.getFile();
				if (file.startsWith("file:")) //$NON-NLS-1$
					file = file.substring("file:".length()); //$NON-NLS-1$
				if (file.endsWith("!/") || file.endsWith("!\\")) //$NON-NLS-1$ //$NON-NLS-2$
					file = file.substring(0, file.length()-2);
				return new IPath[] { new Path(file) };
			}
			Collection libraryNames = getLibraryNames(bundle);
			ArrayList ret = new ArrayList(libraryNames.size());
			for (Iterator it = libraryNames.iterator(); it.hasNext();) {
				String libName = (String) it.next();
				IPath path = pluginPath.append(new Path(libName));
				ret.add(path);
			}
			return (IPath[])ret.toArray(new IPath[ret.size()]);
		}
		
		private static Collection getLibraryNames(Bundle bundle) throws BundleException {
			ArrayList ret = new ArrayList();
			String requires = (String)bundle.getHeaders().get(Constants.BUNDLE_CLASSPATH);
			ManifestElement[] elements = ManifestElement.parseHeader(Constants.BUNDLE_CLASSPATH, requires);
			if (elements == null) return Collections.EMPTY_LIST;
			for(int i = 0, maxi = elements.length; i < maxi; i++) {
				ret.add(elements[i].getValue());
			}
			return ret;
		}
	
		
		public IClasspathEntry[] getClasspathEntries() {
			return classpathEntries;
		}

		public String getDescription() {
			return CommonPluginMessages.TPTP_CONTAINER_NAME;
		}

		public int getKind() {
			return K_APPLICATION;
		}

		public IPath getPath() {
			return new Path(CONTAINER_ID).append(new Path(args));
		}
		
	}
	public void initialize(IPath containerPath, IJavaProject project) throws CoreException {
		if (containerPath.segmentCount() != 2) return;
		JavaCore.setClasspathContainer(
				containerPath,
				new IJavaProject[] { project },
				new IClasspathContainer[] { new ClasspathContainer(containerPath.lastSegment()) },
				new NullProgressMonitor());

	}
	
	/**
	 * Returns the TPTP container entry of a given Java project, if any.
	 * @param javaProject
	 * @return
	 * @throws JavaModelException
	 */
	public static IClasspathEntry getTPTPContainerEntry(IJavaProject javaProject) throws JavaModelException {
		IClasspathEntry[] rawClasspath = javaProject.getRawClasspath();
		for (int i = 0; i < rawClasspath.length; i++) {
			IClasspathEntry entry = rawClasspath[i];
			if (entry.getEntryKind() == IClasspathEntry.CPE_CONTAINER) {
				if (entry.getPath().segmentCount() > 1
					&& ClasspathContainerInitializer.CONTAINER_ID.equals(entry.getPath().segment(0))) {
					return entry;
				}
			}
		}
		return null;
	}

	/**
	 * Creates a new TPTP container, for the given list of plugins. 
	 * @param pluginNames
	 * @return
	 */
	public static IClasspathEntry newContainerEntry(Collection pluginNames) {
		return JavaCore.newContainerEntry(
			new Path(CONTAINER_ID).append(newArgument(pluginNames)));
	}
	
	private static String newArgument(Collection pluginNames) {
		StringBuffer buf = new StringBuffer();
		Iterator it = pluginNames.iterator();
		if (it.hasNext()) {
			buf.append((String)it.next());
		}
		while (it.hasNext()) {
			buf.append(',');
			buf.append((String)it.next());
		}
		return buf.toString();
	}

}
