/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Generator.java,v 1.12 2009/05/04 16:25:56 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.tools.core.internal.common.codegen;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.core.internal.changes.CreateContainerChange;
import org.eclipse.hyades.test.core.internal.changes.DeleteFileChange;
import org.eclipse.hyades.test.core.internal.changes.PotentialChange;
import org.eclipse.hyades.test.core.util.JavaUtil;
import org.eclipse.hyades.test.tools.core.internal.common.CommonPluginMessages;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;
import org.eclipse.ltk.core.refactoring.NullChange;
import org.eclipse.ltk.core.refactoring.Refactoring;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;

/**
 * <p>Abstract code generator/code updater.</p>
 * 
 * <p>The typical use of the generator to generate the code for a test suite is:</p>
 * 
 * <code>
 * Generator generator = new Generator(testSuite);
 * if (generator.checkAllConditions(new SubProgressMonitor(pm, 1)).isOk()) {
 *   Change change = generator.createChange(new SubProgressMonitor(pm, 1));
 *   change.perform(new SubProgressMonitor(pm, 1));
 * }
 * </code>
 * 
 * <p>Since this class implements Refactoring, it can also be used in the context of a refactor
 * wizard (see {@link org.eclipse.ltk.ui.refactoring.RefactoringWizardOpenOperation}).</p>
 * 
 * <p>As of release 4.3, the Generator class has been completely redesigned, and now extends
 * the Refactoring framework provided by Eclipse SDK. This allows the user to preview
 * the changes implied by code generation/code update, and also to revert these changes.</p>
 * 
 * <p>This class is abstract. Sub-classes must provide {@link #generateCode(IProgressMonitor)}.
 * Optionally, if specific actions are needed before and/or after code generation, they can
 * extend {@link #createPreCodegenerationChange(IProgressMonitor)} and
 * {@link #createPostCodegenerationChange(IProgressMonitor)}.
 * If code update is desired they must override {@link #createUpdateChange(IFile, IProgressMonitor)}.
 * If additional checks are needed, sub-classes may extend {@link #checkInitialConditions(IProgressMonitor)},
 * {@link #checkFinalConditions(IProgressMonitor)} and {@link #checkAllConditions(IProgressMonitor)}.</p>
 * 
 * <p>The location and name of the generated file may be customized by overriding the methods
 * {@link #getFileHandle(ITestSuite)} and {@link #getSourceContainerHandle(ITestSuite)}.</p>
 * 
 * <p>A couple of helper methods are available to sub-classes: to manipulate Changes, see 
 * {@link #addPotentialChange(Change, CompositeChange)} and {@link #packCompositeChange(CompositeChange)}.</p>
 * 
 * <p>The package {@link org.eclipse.hyades.test.tools.core.internal.common.codegen.changes} provides a
 * library of pre-defined Changes for achieving various actions related to code-generation.</p>
 * 
 * <p>For classes that generate java code, consider extending {@link JavaGenerator}.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Julien Canches
 * @author  Paul E. Slauenwhite
 * @version April 30, 2009
 * @since   March 18, 2005
 */
abstract public class Generator extends Refactoring
{
	public static final String CHARSET_UTF8 = "UTF-8"; //$NON-NLS-1$
	
	private ITestSuite testSuite;
	private ITestSuite cachedRepositoryTestSuite;
	private boolean cachedRepositoryComputed = false;
	
	/**
	 * Instantiates a generator for the specified test suite.
	 * @param testSuite
	 */
	protected Generator(ITestSuite testSuite) {
		this.testSuite = testSuite;
	}
	
	/**
	 * Returns the test suite for which this generator is generating or updating code.
	 * @return
	 */
	final public ITestSuite getTestSuite() {
		return testSuite;
	}
	
	/**
	 * Returns the test suite as it is currently persisted on disk. This allows
	 * to compare models before and after modification.
	 * @return A test suite model, or <code>null</code> if the test suite model is
	 * new and has never been persisted.
	 */
	final protected ITestSuite getRepositoryTestSuite() {
		if (!cachedRepositoryComputed) {
			TPFTestSuite ts = (TPFTestSuite)getTestSuite();
			Resource res = ts.eResource();
			if (res != null) {
				URI uri = EcoreUtil.getURI(ts);
				ResourceSet rs = new ResourceSetImpl();
				try {
					EObject object = rs.getEObject(uri, /*loadOnDemand*/true);
					if (object != null && object instanceof ITestSuite) {
						cachedRepositoryTestSuite = (ITestSuite)object;
					}
				} catch (Throwable t) {
					// In 99% this is because the resource does not exist.
					cachedRepositoryTestSuite = null;
				}
			}
			cachedRepositoryComputed = true;
		}
		return cachedRepositoryTestSuite;
	}
	
	/**
	 * Add a change to a composite change, if this change has a real effect.
	 * A change is considered to have an effect if it not <code>null</code>, if it is not
	 * an instance of {@link NullChange}, and if it is not an instance of {@link PotentialChange},
	 * unless this potential change reports to be required.
	 * @param change A change or <code>null</code>.
	 * @param composite The composite change that will receive the change if necessary.
	 */
	protected final void addPotentialChange(Change change, CompositeChange composite) {
		if (change != null) {
			if (change instanceof NullChange) return;
			if (change instanceof PotentialChange && !((PotentialChange)change).hasEffect()) return;
			composite.add(change);
		}
	}
	
	/**
	 * Pack a composite change. If the composite change
	 * has zero children, a {@link NullChange} is returned. Otherwise, the composite itself is
	 * returned.
	 * @param change
	 * @return
	 */
	protected final Change packCompositeChange(CompositeChange change) {
		if (change.getChildren().length > 0) return change;
		return new NullChange();
	}
	
	/**
	 * This implementation always return a OK status.
	 */
	public RefactoringStatus checkFinalConditions(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		pm.beginTask("", 1); //$NON-NLS-1$
		try {
			return new RefactoringStatus();
		} finally {
			pm.done();
		}
	}

	/**
	 * This implementation checks existence of the source container, and the accessibility of its project.
	 */
	public RefactoringStatus checkInitialConditions(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		
		pm.beginTask("", 2); //$NON-NLS-1$
		
		try {
		
			RefactoringStatus status = new RefactoringStatus();
			
			String errorMessage = JavaUtil.validateSourceFolder(getSourceContainerHandle(testSuite));
			
			pm.worked(1);
			
			if (errorMessage != null){
				status.addFatalError(NLS.bind(CommonPluginMessages.CODEGEN_SOURCE_FOLDER_ERROR, errorMessage));
			} 
							
			pm.worked(1);
			
			return status;
		} 
		finally {
			pm.done();
		}
	}

	final public Change createChange(IProgressMonitor monitor) throws CoreException, OperationCanceledException {
		Change change = null;
		IFile file = getFileHandle(testSuite);
		ITestSuite previous = getRepositoryTestSuite();
		if (previous != null) {
			file = getFileHandle(previous);
		}
		if(file != null && file.exists()) {
			try {
				change = createUpdateChange(file, monitor);
			} catch (UnsupportedOperationException e) {
				//Fall back to code generation.
			}
		}
		if (change == null) {
			change = createGenerateChange(monitor);
		}
		return change;
	}

	public String getName() {
		return CommonPluginMessages.GENERIC_GENERATOR;
	}

	private Change createGenerateChange(IProgressMonitor monitor) throws CoreException {
		monitor.beginTask("", 6); //$NON-NLS-1$
		try {
			CompositeChange change = new CompositeChange(CommonPluginMessages.GENERATE_CODE);
			Change subChange = createPreCodegenerationChange(new SubProgressMonitor(monitor, 1));
			if (subChange != null) {
				change.add(subChange);
			}
			IFile file = getFileHandle(testSuite);
			if (file.exists()) {
				change.add(new DeleteFileChange(file));
			}
			monitor.worked(1);
			change.add(createGenerateCodeChange(file, monitor));
			monitor.worked(3);
			subChange = createPostCodegenerationChange(new SubProgressMonitor(monitor, 1));
			if (subChange != null) {
				change.add(subChange);
			}
			return change;
		} finally {
			monitor.done();
		}
	}
	
	/**
	 * This implementation creates the source container, if it does not exist.
	 * @param monitor
	 * @return
	 * @throws CoreException
	 */
	protected Change createPreCodegenerationChange(IProgressMonitor monitor) throws CoreException {
		monitor.beginTask("", 1); //$NON-NLS-1$
		try {
			IContainer sourceContainer = getSourceContainerHandle(testSuite);
			if (!sourceContainer.exists()) {
				return new CreateContainerChange(sourceContainer);
			}
			return new NullChange();
		} finally {
			monitor.done();
		}
	}
	
	/**
	 * This implementation does nothing.
	 * @param monitor
	 * @return
	 * @throws CoreException
	 */
	protected Change createPostCodegenerationChange(IProgressMonitor monitor) throws CoreException {
		monitor.beginTask("", 1); //$NON-NLS-1$
		try {
			return new NullChange();
		} finally {
			monitor.done();
		}
	}
	
	/**
	 * Creates the change that describes the update of the specified file.
	 * <p>This implementation throws a {@link UnsupportedOperationException}. If code update
	 * is desired, the sub-class must override this implementation.</p>
	 * @param file The file to update
	 * @param monitor A progress monitor
	 * @return A change describing the update.
	 * @throws CoreException if the update creation failed, UnsupportedOperationException if code
	 * update is not supported.
	 */
	protected Change createUpdateChange(IFile file, IProgressMonitor monitor) throws CoreException, UnsupportedOperationException {
		throw new UnsupportedOperationException();
	}
		
	/**
	 * Returns the generated file. This is handle-only method, i.e. the returned file is
	 * not required to exist. 
	 * @return The generated file.
	 */
	public IFile getFileHandle(ITestSuite ts) {
		IContainer sourceContainer = getSourceContainerHandle(ts);
		return sourceContainer.getFile(new Path(ts.getImplementor().getResource()));
	}
	
	/**
	 * <p>Returns the container of the generated class for the test suite.</p>
	 * 
	 * @param testSuite The test suite of the generated class. 
	 * @return The container of the generated class for the test suite, otherwise <code>null</code>.
	 */
	final protected IContainer getSourceContainerHandle(ITestSuite testSuite) {
		
		IContainer container = null;
		
		try {

			String location = testSuite.getImplementor().getLocation();
			
			if (location != null) {
				
				IPath path = new Path(location);
				
				if(path.segmentCount() == 1) {
					container = ResourcesPlugin.getWorkspace().getRoot().getProject(path.toString());
				} 
				else {
					container = ResourcesPlugin.getWorkspace().getRoot().getFolder(path);
				}
			}
		} 
		catch (Exception e) {
			//Ignore and return null.
		}
		
		return container;
	}
	
	/**
	 * Generates the file for the test suite.
	 * @param file The file that must be generated.
	 * @param monitor A progress monitor
	 * @return A change.
	 * @throws CoreException
	 */
	abstract protected Change createGenerateCodeChange(IFile file, IProgressMonitor monitor) throws CoreException;
	
}
