/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: Helper.java,v 1.20 2010/05/13 14:55:25 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.tools.core.internal.common.codegen;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.facades.behavioral.IDeployableComponent;
import org.eclipse.hyades.models.common.facades.behavioral.IImplementor;
import org.eclipse.hyades.models.common.facades.behavioral.INamedElement;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.test.core.internal.changes.CompilationUnitChange;
import org.eclipse.hyades.test.core.internal.changes.DeleteFileChange;
import org.eclipse.hyades.test.core.internal.changes.MoveCompilationUnitChange;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.core.util.JavaUtil;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.hyades.test.tools.core.internal.common.CommonPluginMessages;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.changes.AdjustTestCasePropertiesChange;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.changes.ProjectAdjustChange;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.ToolFactory;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.ImportDeclaration;
import org.eclipse.jdt.core.formatter.CodeFormatter;
import org.eclipse.jface.text.Document;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;
import org.eclipse.text.edits.TextEdit;

/**
 * <p>Helper.java</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @author  Marcelo Paternostro
 * @version May 13, 2010
 * @since   March 18, 2005
 */
public class Helper{
	
	public final static String JUNIT_TEST_CLASS_NAME = "junit.framework.Test"; //$NON-NLS-1$
	public final static String JUNIT_TEST_CASE_CLASS_NAME = "junit.framework.TestCase"; //$NON-NLS-1$
	public final static String JUNIT_TEST_SUITE_CLASS_NAME = "junit.framework.TestSuite"; //$NON-NLS-1$
	public final static String HYADES_TEST_CASE_CLASS_NAME = "org.eclipse.hyades.test.common.junit.HyadesTestCase"; //$NON-NLS-1$
	public final static String HYADES_TEST_SUITE_CLASS_NAME = "org.eclipse.hyades.test.common.junit.HyadesTestSuite"; //$NON-NLS-1$
	public final static String HYADES_DEFAULT_TEST_ARBITER_CLASS_NAME = "org.eclipse.hyades.test.common.junit.DefaultTestArbiter"; //$NON-NLS-1$
	public final static String HYADES_JUNIT_RUNNER = "org.eclipse.hyades.test.java.runner.HyadesJUnitRunner"; //$NON-NLS-1$
	public final static String EXCEPTION_CLASS_NAME = "java.lang.Exception"; //$NON-NLS-1$
	public final static String JUNIT_TEST_METHOD_PREFIX = "test"; //$NON-NLS-1$
	
	private static final String DEFAULT_NAMESPACE = "DEFNS#"; //$NON-NLS-1$
	private Set namesInWorkspace;
	private Map nameByObjectKey;
	private Map newTestCaseMethodNames;

	private ImportManager importManager;
	private StringBuffer importStringBuffer;
	private int importInsertionPoint;
	
	private String superClassName = HYADES_TEST_CASE_CLASS_NAME;

	public static String formatContent(String content) {
		return formatContent(content, 0);
	}
	
	public static String formatContent(String content, int offset)
	{
		if(content == null)
			return null;
		String code = ""; //$NON-NLS-1$
		Document doc = new Document(content);
		TextEdit tEdit = ToolFactory.createCodeFormatter(null).format(CodeFormatter.K_UNKNOWN, doc.get(), 0, doc.get().length(), offset, null);
        try {
            tEdit.apply(doc);
        } catch (Exception e) {
            CorePlugin.logError(e);
        }
        code = doc.get();
		return code;
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		if (namesInWorkspace != null)
			namesInWorkspace.clear();

		if (nameByObjectKey != null)
			nameByObjectKey.clear();
	}

	public String getPackageName(Object object)
	{
		String cls = null;
		if (object instanceof ITest)
			object = ((ITest)object).getImplementor();

		if (object instanceof IDeployableComponent)
			cls = ((IDeployableComponent)object).getResource();

		if (cls != null)
		{
			int index = cls.lastIndexOf('.');
			if (index >= 0)
				return cls.substring(0, index);
		}

		return ""; //$NON-NLS-1$
	}

	/**
	 * This method does nothing and only returns an empty string "". 
	 * Use {@link org.eclipse.hyades.test.ui.internal.util.TestUIUtil TestUIUtil} instead since this is a UI utility.
	 * @param object
	 * @return
	 * @deprecated
	 */
	public String getLabel(Object object)
	{
		return ""; //$NON-NLS-1$
	}
	
	public String getFilePath(Object object)
	{
		if(object instanceof EObject)
			return EMFUtil.getWorkspaceFilePath(((EObject)object).eResource()); 
		
		return null;		
	}

	public String retrieveClassName(INamedElement namedElement)
	{
		String name = getUniqueName("CLASS#", namedElement); //$NON-NLS-1$
		if (name != null)
			return name;

		IDeployableComponent deployableComponent = null;
		if (namedElement instanceof ITestSuite)
			deployableComponent = ((ITestSuite)namedElement).getImplementor();
		else if (namedElement instanceof IDeployableComponent)
			deployableComponent = (IDeployableComponent)namedElement;

		if(deployableComponent != null)
		{
			name = deployableComponent.getResource();
			name = name.substring(name.lastIndexOf('.')+1);
		}
		else
			name = namedElement.getName();
			 

		name = JavaUtil.getValidClassName(name, false);

		name = retrieveUniqueName("CLASS#", namedElement, name, true); //$NON-NLS-1$

		if(deployableComponent != null)
		{
            String packageName = getPackageName(deployableComponent);
            String newName = packageName.length() > 0 ? packageName + "." + name : name; //$NON-NLS-1$
            if (!newName.equals(deployableComponent.getResource())) {
                deployableComponent.setResource(newName);
            }
		}
		return name;
	}

	/**
	 * Returns a modified identifier with first letter lower case.<br>
	 * E.g. lowerCaseIdentifier("MyIdentifier") returns "myIdentifier". The same
	 * identifier is returned if it already starts with a lower case letter.
	 * @param identifier An identifier. <code>null</code> is allowed.
	 * @return An identifier starting with a lower case, or <code>null</code> if
	 * identifier was <code>null</code>.
	 */
	public String lowerCaseIdentifier(String identifier) {
		if (identifier == null) return null;
		if (identifier.length() == 0) return identifier;
		char c = identifier.charAt(0);
		if (Character.isLowerCase(c)) {
			return identifier;
		} else {
			return Character.toString(Character.toLowerCase(c)) + identifier.substring(1);
		}
	}
	
	/**
	 * Returns a modified identifier with first letter lower case.<br>
	 * E.g. lowerCaseIdentifier("MyIdentifier") returns "myIdentifier". The same
	 * identifier is returned if it already starts with a lower case letter.
	 * @param identifier An identifier. <code>null</code> is allowed.
	 * @return An identifier starting with a lower case, or <code>null</code> if
	 * identifier was <code>null</code>.
	 */
	public String upperCaseIdentifier(String identifier) {
		if (identifier == null) return null;
		if (identifier.length() == 0) return identifier;
		char c = identifier.charAt(0);
		if (Character.isUpperCase(c)) {
			return identifier;
		} else {
			return Character.toString(Character.toUpperCase(c)) + identifier.substring(1);
		}
	}
	
	public String retrieveIdentifierName(INamedElement namedElement, Object namespace)
	{
		String name = getUniqueName("IDENTIFIER#" + namespace, namedElement); //$NON-NLS-1$
		if (name != null)
			return name;

		name = JavaUtil.getValidIdentifierName(namedElement.getName());

		name = retrieveUniqueName("IDENTIFIER#" + namespace, namedElement, name, true); //$NON-NLS-1$
		// JC: commenting out the following: a getter should not modify the object!
		// (it is up to the caller to decide what to do with the returned value)
//		if(namedElement instanceof ITestCase)
//		{
//			ITestCase testCase = (ITestCase)namedElement;
//			if(!name.equals(testCase.getImplementor().getResource()))
//				testCase.getImplementor().setResource(name);
//		}
//		
		return name;
	}
	
	/**
	 * Returns a unique name for a given object derived from the the base name
	 * argument.  The namespace argument defines the context in which the returned
	 * name must be unique.  If <code>null</code> then a default namespace is default. 
	 *
	 * <p>If a name has been already created for a given object in the same	namespace 
	 * this name is returned.
	 * 
	 * @param namespace
	 * @param object
	 * @param name
	 * @return a not <code>null</code> String
	 */
	protected synchronized String retrieveUniqueName(String namespace, INamedElement namedElement, String baseName, boolean caseSensitive)
	{
		if ((namedElement == null) || (baseName == null))
			return ""; //$NON-NLS-1$

		if(!caseSensitive)
			baseName = baseName.toLowerCase();

		if (namesInWorkspace == null)
			namesInWorkspace = new HashSet();
		if (nameByObjectKey == null)
			nameByObjectKey = new HashMap();

		if (namespace == null)
			namespace = DEFAULT_NAMESPACE;

		String key = namespace + ">"; //$NON-NLS-1$
		String objectKey = key + namedElement.getId();
		String uniqueName = (String)nameByObjectKey.get(objectKey);
		if (uniqueName == null)
		{
			int count = 0;
			uniqueName = baseName;
			while (namesInWorkspace.contains(key + uniqueName))
				uniqueName = baseName + (++count);

			namesInWorkspace.add(key + uniqueName);
			nameByObjectKey.put(objectKey, uniqueName);
		}

		return uniqueName;
	}

	/**
	 * Returns a created unique name for a given object derived from the the base name
	 * argument.  The namespace argument defines the context in which the returned
	 * name must be unique.  If <code>null</code> then a default namespace is default. 
	 *
	 * @param namespace
	 * @param object
	 * @return a unique name or null if none was created by the 
	 * {@link #retrieveUniqueName(String, Object, String).
	 */
	protected String getUniqueName(String namespace, INamedElement namedElement)
	{
		if ((namedElement == null) || (nameByObjectKey == null))
			return null;

		if (namespace == null)
			namespace = DEFAULT_NAMESPACE;

		String key = namespace + ">"; //$NON-NLS-1$
		String objectKey = key + namedElement.getId();
		return (String)nameByObjectKey.get(objectKey);
	}

	public void markImportLocation(StringBuffer stringBuffer)
	{
		importStringBuffer = stringBuffer;
		importInsertionPoint = stringBuffer.length();
		importManager.addCompilationUnitImports(stringBuffer.toString());
	}

	public void emitSortedImports()
	{
		String NL = System.getProperties().getProperty("line.separator"); //$NON-NLS-1$
		StringBuffer imports = new StringBuffer();

		String previousPackageName = null;
		for (Iterator iter = importManager.getImports().iterator(); iter.hasNext();)
		{
			String importName = (String)iter.next();
			int index = importName.lastIndexOf("."); //$NON-NLS-1$
			if (index != -1)
			{
				String packageName = importName.substring(0, index);
				if (previousPackageName != null && !previousPackageName.equals(packageName))
				{
					imports.append(NL);
				}
				previousPackageName = packageName;
			}
			imports.append(NL + "import " + importName + ";"); //$NON-NLS-1$ //$NON-NLS-2$
		}

		importStringBuffer.insert(importInsertionPoint, imports);
	}
	
	/**
	 * Update the specified compilation unit with the list of imports that were
	 * recorded by the helper using addImport.
	 * @throws JavaModelException 
	 * @author jcanches
	 */
	public void emitSortedImports(ICompilationUnit cu) throws JavaModelException {
		Iterator it = importManager.getImports().iterator();
		while (it.hasNext()) {
			String importName = (String) it.next();
			cu.createImport(importName, /*sibling*/null, /*progressMonitor*/null);
		}
	}
	
	public void emitSortedImports(CompilationUnit cu) {
		Iterator it = importManager.getImports().iterator();
		while (it.hasNext()) {
			String importName = (String) it.next();
			if (!ASTHelper.hasImport(cu, importName)) {
				ImportDeclaration id = cu.getAST().newImportDeclaration();
				id.setName(cu.getAST().newName(importName));
				cu.imports().add(id);
			}
		}
		// TODO Remove unused imports
	}

	public String getImportedName(String qualifiedName)
	{
		int index = qualifiedName.indexOf("$"); //$NON-NLS-1$
		importManager.addImport(index == -1 ? qualifiedName : qualifiedName.substring(0, index));
		return importManager.getImportedName(qualifiedName);
	}

	public void addImport(String qualifiedName)
	{
		importManager.addImport(qualifiedName);
	}

	protected ImportManager getImportManager()
	{
		return importManager;
	}

	public void setImportManager(ImportManager importManager)
	{
		this.importManager = importManager;
	}
	
	/**
	 * Computes a valid unique name for a test method that would implement the
	 * specified test case.
	 * @param testCase
	 * @param strictJUnitConventions
	 * @return
	 */
	public String computeTestMethodName(ITestCase testCase, boolean strictJUnitConventions) {
		String identifier = retrieveIdentifierName(testCase, "METHOD"); //$NON-NLS-1$
		// The above call never returns null
		if (strictJUnitConventions) {
			if (!identifier.startsWith(JUNIT_TEST_METHOD_PREFIX)) {
				return JUNIT_TEST_METHOD_PREFIX + upperCaseIdentifier(identifier);
			}
		}
		return identifier;
	}
	
	/**
	 * Returns the test method name that implements the specified testCase
	 * in the JUnit source code, if this value was set.
	 * @param testCase
	 * @return The test method name, or <code>null</code> if this value was not
	 * set.
	 * @see Helper#computeTestMethodName(ITestCase, boolean)
	 */
	public String getTestMethodName(ITestCase testCase, boolean directModelAccess) {
		if (!directModelAccess && newTestCaseMethodNames != null) {
			String name = (String)newTestCaseMethodNames.get(testCase);
			if (name != null) return name;
		}
		return getTestMethodName(testCase);
	}
	
	public static String getTestMethodName(ITestCase testCase) {
		IImplementor implementor = testCase.getImplementor();
		return implementor != null ? implementor.getResource() : null;
	}

	/**
	 * Records the name of the test method associated to a testCase.
	 * @param testCase
	 * @param name
	 * @see Helper#computeTestMethodName(ITestCase, boolean)
	 * @see Helper#getTestMethodName(ITestCase, boolean)
	 */
	public void setTestMethodName(ITestCase testCase, String name, boolean directModelAccess) {
		if (directModelAccess) {
			setTestMethodName(testCase, name);
		} else {
			if (newTestCaseMethodNames == null) {
				newTestCaseMethodNames = new HashMap();
			}
			newTestCaseMethodNames.put(testCase, name);
		}
	}
	
	public static void setTestMethodName(ITestCase testCase, String name) {
		testCase.getImplementor().setResource(name);
	}
	
	public Change getMethodNamesChange() {
		if (newTestCaseMethodNames != null) {
			CompositeChange change = new CompositeChange(CommonPluginMessages.Helper_MODEL_ADJUST_CHANGE);
			for (Iterator it = newTestCaseMethodNames.entrySet().iterator(); it.hasNext();) {
				Map.Entry entry = (Map.Entry) it.next();
				change.add(new AdjustTestCasePropertiesChange((ITestCase)entry.getKey(), (String)entry.getValue()));
			}
			return change;
		}
		return null;
	}
	
	public void setSuperclassName(String name) {
		this.superClassName = name;
	}
	
	public String getSuperclassName() {
		return superClassName;
	}
	
	/**
	 * Compares two strings appearing within a java comment/javadoc comment and return true
	 * if they contain the same strings, ignoring differences such as line breaks, tabs and
	 * space indentation.
	 * @param comment1
	 * @param comment2
	 * @return
	 */
	public static boolean compareJavaComments(CharSequence comment1, String comment2) {
		// Compare character by character, ignoring separating characters
		final String separators = "\n\r\t "; //$NON-NLS-1$
		StringIterator i1 = new StringIterator(comment1, separators);
		StringIterator i2 = new StringIterator(comment2, separators);
		while(i1.hasNextCharacter() && i2.hasNextCharacter()) {
			if (i1.getNextCharacter() != i2.getNextCharacter()) return false;
		}
		return !i1.hasNextCharacter() && !i2.hasNextCharacter();
	}
	
	/**
	 * String iterator that iterates character by character through a string. It has
	 * the ability to group a certain class of characters, replacing a sequence of one or 
	 * more of such characters by a single space. 
	 * @author jcanches
	 */
	public static class StringIterator {
		private int currentIndex;
		private int length;
		private CharSequence string;
		private String spaceCharacters;
		private boolean issueSpace;
		public StringIterator(CharSequence string, String spaceCharacters) {
			this.string = string;
			this.length = string.length();
			this.spaceCharacters = spaceCharacters;
			this.currentIndex = 0;
			this.issueSpace = false;
			skipIgnoredCharacters();
		}
		private void skipIgnoredCharacters() {
			while (currentIndex < length
				   && spaceCharacters.indexOf(string.charAt(currentIndex)) != -1) {
				issueSpace = true;
				currentIndex++;
			}
		}
		public boolean hasNextCharacter() {
			return issueSpace || currentIndex < length;
		}
		public char getNextCharacter() {
			if (issueSpace) {
				issueSpace = false;
				return ' ';
			}
			char ret = string.charAt(currentIndex++);
			skipIgnoredCharacters();
			return ret;
		}
	}

	public static class LineIterator {
		private final static String lineDelimiters = "\n\r"; //$NON-NLS-1$
		private StringIterator iterator;
		private String nextLine;
		public LineIterator(CharSequence string, String spaceCharacters) {
			this.iterator = new StringIterator(string, spaceCharacters);
			computeNextLine();
		}
		public boolean hasNextLine() {
			return nextLine != null;
		}
		public String getNextLine() {
			String ret = nextLine;
			computeNextLine();
			return ret;
		}
		private void computeNextLine() {
			StringBuffer line = new StringBuffer();
			boolean lineBegins = true;
			while (iterator.hasNextCharacter()) {
				char c = iterator.getNextCharacter();
				if (lineBegins) {
					if (c == ' ' || lineDelimiters.indexOf(c) != -1) {
						continue;
					}
					lineBegins = false;
				}
				if (lineDelimiters.indexOf(c) == -1) {
					line.append(c);
				} else {
					break;
				}
			}
			nextLine = line.toString().trim();
			if (nextLine.length() == 0) nextLine = null;
		}
	}
	
	public static boolean isDestructiveChange(Change change) {
		if (change instanceof ProjectAdjustChange) {
			return true;
		}
		if (change instanceof CompilationUnitChange) {
			return ((CompilationUnitChange)change).isDestructive();
		}
		if (change instanceof MoveCompilationUnitChange) {
			return true;
		}
		if (change instanceof DeleteFileChange) {
			return true;
		}
		if (change instanceof CompositeChange) {
			CompositeChange cchange = (CompositeChange)change;
			Change[] children = cchange.getChildren();
			for (int i = 0; i < children.length; i++) {
				Change c = children[i];
				if (isDestructiveChange(c)) return true;
			}
			return false;
		}
		return false;
	}

}
