/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JavaGenerator.java,v 1.19 2009/05/04 16:25:56 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.tools.core.internal.common.codegen;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.test.core.internal.changes.CreatePackageFragmentChange;
import org.eclipse.hyades.test.core.internal.changes.MoveCompilationUnitChange;
import org.eclipse.hyades.test.core.internal.changes.PotentialChange;
import org.eclipse.hyades.test.core.util.JavaUtil;
import org.eclipse.hyades.test.tools.core.internal.common.CommonPluginMessages;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.changes.AdjustSourceContainerChange;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.changes.AssociateModelSourceChange;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.changes.ProjectAdjustChange;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;
import org.eclipse.ltk.core.refactoring.NullChange;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;

/**
 * <p>Abstract Java code generator.</p>
 * 
 * <p>Although this abstract class does not implement any code generation
 * or code update, it provides additional features for generating java code, and also
 * makes additional assumptions about the test suite behavior model.</p>
 * 
 * <p>Sub-classes must implement {@link Generator#generateCode(IProgressMonitor)}, and
 * need to redefine {@link #createSourceUpdateChange(SubProgressMonitor)} if code update
 * is desired.</p>
 * 
 * <p>Sub-classes may also redefine {@link #computeTestMethodNames(ITestSuite, boolean, Helper)},
 * and extend {@link #createAdjustProjectClasspathChange(IProgressMonitor)},
 * {@link #createPreCodegenerationChange(IProgressMonitor)},
 * {@link Generator#createPostCodegenerationChange(IProgressMonitor)}, and all conditions
 * checking methods.</p>
 * 
 *
 * @author  Marcelo Paternostro
 * @author  Julien Canches
 * @author  Paul E. Slauenwhite
 * @version April 30, 2009
 * @since   March 18, 2005 
 */
abstract public class JavaGenerator extends Generator {
	
	private IProjectDependencyUpdater projectDependencyUpdater;
	
	/**
	 * Creates a JavaGenerator for a Java test suite.
	 * @param testSuite
	 * @param updater A Project dependency updater that will perform project
	 * adjustment to comply with additional classpath requirements.
	 */
	protected JavaGenerator(ITestSuite testSuite, IProjectDependencyUpdater updater) {
		super(testSuite);
		this.projectDependencyUpdater = updater;
	}
	
	/**
	 * Creates the change that adjusts the project classpath.
	 * <p>This implementation creates a change that invokes the generator's project
	 * dependency updater.</p>
	 * <p>Sub-classes may extend or redefine this implementation.</p> 
	 * @param monitor
	 * @return
	 * @throws CoreException
	 */
	protected Change createAdjustProjectClasspathChange(IProgressMonitor monitor) throws CoreException {
		monitor.beginTask("", 1); //$NON-NLS-1$
		try {
			IProject project = getFileHandle(getTestSuite()).getProject();
			Collection dependentProjects = getDependentProjects(getTestSuite(), project);
			for(Iterator it = dependentProjects.iterator(); it.hasNext(); ) {
				projectDependencyUpdater.addRequiredProject((IProject)it.next());
			}
			if (!projectDependencyUpdater.previewAdjustProject(project).isEmpty()) {
				monitor.worked(1);
				return new ProjectAdjustChange(projectDependencyUpdater, project);
			} else {
				monitor.worked(1);
				return null;
			}
		} finally {
			monitor.done();
		}
	}
	
	protected IPackageFragmentRoot getPackageFragmentRoot(ITestSuite ts) {
		IContainer container = getSourceContainerHandle(ts);
		if (container == null) return null;
		IJavaElement javaElement = JavaCore.create(container);
		if (javaElement instanceof IJavaProject) {
			IJavaProject jproject = (IJavaProject)javaElement;
			return jproject.getPackageFragmentRoot(container);
		} else if (javaElement instanceof IPackageFragmentRoot) {
			return (IPackageFragmentRoot)javaElement;
		}
		return null;
	}
	
	protected IPackageFragment getPackageFragment(ITestSuite ts) {
		IPackageFragmentRoot root = getPackageFragmentRoot(ts);
		if (root == null) return null;
		return root.getPackageFragment(getPackageName(ts));
	}
	
	protected ICompilationUnit getCompilationUnit(ITestSuite ts) {
		IPackageFragment pack = getPackageFragment(ts);
		if (pack != null) {
			return pack.getCompilationUnit(getClassName(ts) + ".java"); //$NON-NLS-1$
		}
		return null;
	}
	
	/**
	 * <p>Returns the package name of the generated class for the test suite.</p>
	 * 
	 * @param testSuite The test suite of the generated class. 
	 * @return The package name of the generated class for the test suite, otherwise an empty string
	 */
	protected String getPackageName(ITestSuite testSuite) {
		
		String packageName = "";
		
		String resource = testSuite.getImplementor().getResource();
		
		if (resource != null){
			
			int dotIndex = resource.lastIndexOf('.');
			
			if (dotIndex != -1) {
				packageName = resource.substring(0, dotIndex);
			}
		}

		return packageName;
	}
	
	/**
	 * <p>Returns the class name of the generated class for the test suite.</p>
	 * 
	 * @param testSuite The test suite of the generated class. 
	 * @return The class name of the generated class for the test suite, otherwise <code>null</code>.
	 */
	protected String getClassName(ITestSuite testSuite) {

		String className = null;
		
		String resource = testSuite.getImplementor().getResource();
		
		if (resource != null){
			
			int dotIndex = resource.lastIndexOf('.');
			
			if (dotIndex != -1) {
				className = resource.substring(dotIndex + 1);
			}
			else{
				className = resource;
			}
		}

		return className;
	}
	
	/**
	 * In addition to its super implementation, this implementation creates a change that ensures 
	 * that the source container is a source folder, creates the package, and adjusts the project
	 * classpath if necessary.
	 */
	protected Change createPreCodegenerationChange(IProgressMonitor monitor) throws CoreException
	{
		monitor.beginTask("", 10); //$NON-NLS-1$
		try {
			CompositeChange change = new CompositeChange(CommonPluginMessages.SETUP_CONTEXT);
			addPotentialChange(super.createPreCodegenerationChange(monitor), change);
			addPotentialChange(createAdjustSourceContainerChange(), change);
			addPotentialChange(createCreatePackageChange(), change);
			addPotentialChange(createAdjustProjectClasspathChange(new SubProgressMonitor(monitor, 1)), change);
			return packCompositeChange(change);
		} finally {
			monitor.done();
		}
	}
	
	/**
	 * Helper method to create a change that makes sure that the source container is a java
	 * source folder.
	 * @return
	 */
	final protected Change createAdjustSourceContainerChange() {
		PotentialChange change = new AdjustSourceContainerChange(getSourceContainerHandle(getTestSuite()));
		if (change.hasEffect()) return change;
		else return new NullChange();
	}
	
	/**
	 * Helper method to create a change that creates the package, if necessary.
	 */
	final protected Change createCreatePackageChange() {
        IContainer container = getSourceContainerHandle(getTestSuite());
		IJavaProject jproject = JavaCore.create(container.getProject());
		IPackageFragmentRoot sourceFolder = jproject.getPackageFragmentRoot(container);
		PotentialChange change = new CreatePackageFragmentChange(sourceFolder, getPackageName(getTestSuite()));
		if (change.hasEffect()) return change;
		else return new NullChange();
	}

	protected final Change createUpdateChange(IFile file, IProgressMonitor monitor) throws CoreException, UnsupportedOperationException {
		monitor.beginTask("", 2); //$NON-NLS-1$
		try {
			Change sourceUpdateChange = createSourceUpdateChange(file, new SubProgressMonitor(monitor, 1));
			CompositeChange change = new CompositeChange(CommonPluginMessages.UPDATE_CODE);
			addPotentialChange(createAdjustProjectClasspathChange(new SubProgressMonitor(monitor, 1)), change);
			addPotentialChange(sourceUpdateChange, change);
			addPotentialChange(createCreatePackageChange(), change);
			addPotentialChange(createMoveCompilationUnitChange(), change);
			addPotentialChange(createPostCodegenerationChange(new SubProgressMonitor(monitor, 1)), change);
			change.markAsSynthetic();
			return packCompositeChange(change);
		} finally {
			monitor.done();
		}
	}
	
	/**
	 * Creates a move compilation unit change, if necessary (i.e. if the classes in the
	 * previous and in the current versions of the model are different).
	 * @return
	 */
	private Change createMoveCompilationUnitChange() {
		ITestSuite previous = getRepositoryTestSuite();
		if (previous != null) {
			IFile file = getFileHandle(previous);
			IFile newFile = getFileHandle(getTestSuite());
			if (!newFile.equals(file)) {
				// Change destination
				if (file.exists()) {
					ICompilationUnit cu = JavaCore.createCompilationUnitFrom(file);
					if (cu.exists()) {
						IPackageFragment newPackage = getPackageFragment(getTestSuite());
						return new MoveCompilationUnitChange(cu, newPackage, newFile.getName());
					}
				}
			}
		}
		return null;
	}

	/**
	 * Creates the source update change. This implementation throws a {@link UnsupportedOperationException}.
	 * Sub-classes need to redefine this method in order to be code-update-capable.
	 * @file The file to update. This is not always equal to getFileHandle(), when there
	 * is a source file move involved during the generation.
	 * @param monitor A progress monitor
	 * @return
	 * @throws CoreException, UnsupportedOperationException
	 */
	protected Change createSourceUpdateChange(IFile file, SubProgressMonitor monitor) throws CoreException, UnsupportedOperationException {
		throw new UnsupportedOperationException();
	}

	protected Change createPostCodegenerationChange(IProgressMonitor monitor) throws CoreException {
		PotentialChange change = new AssociateModelSourceChange(getFileHandle(getTestSuite()), getTestSuite());
		if (change.hasEffect()) {
			return change;
		}
		return new NullChange();
	}

	/**
	 * In addition to its super implementation, this implementation checks that the package name
	 * and class name are valid.
	 */
	public RefactoringStatus checkInitialConditions(IProgressMonitor monitor) throws CoreException, OperationCanceledException {
		
		monitor.beginTask("", 6); //$NON-NLS-1$
		
		try {
			
			//Generator (source folder) validation:
			RefactoringStatus status = super.checkInitialConditions(new SubProgressMonitor(monitor, 2));
			
			//Package name validation:
			String errorMessage = JavaUtil.validatePackageName(getPackageName(getTestSuite()));
					
			monitor.worked(1);
			
			if (errorMessage != null){
				status.addFatalError(NLS.bind(CommonPluginMessages.CODEGEN_PACKAGE_NAME_ERROR, errorMessage));
			} 
			
			monitor.worked(1);
			
			//Class name validation:
			errorMessage = JavaUtil.validateClassName(getClassName(getTestSuite()));
			
			monitor.worked(1);
			
			if (errorMessage != null){
				status.addFatalError(NLS.bind(CommonPluginMessages.CODEGEN_CLASS_NAME_ERROR, errorMessage));
			} 
			
			monitor.worked(1);
			
			//Compilation unit validation:
			ITestSuite previous = getRepositoryTestSuite();
			ICompilationUnit cu;
			if (previous != null) {
				cu = getCompilationUnit(previous);
			} else {
				cu = getCompilationUnit(getTestSuite());
			}
			if (cu != null && cu.hasUnsavedChanges()) {
				status.addFatalError(NLS.bind(CommonPluginMessages.JavaGenerator_W_COMP_UNIT_UNSAVED_CHANGES, cu.getElementName()));
			}
			return status;
		} finally {
			monitor.done();
		}
	}
	
	/**
	 * Computes the names of the test methods associated to the test cases of the specified test
	 * suite. Test methods names should be typically stored in the model.
	 * @param testSuite
	 * @param isExternalBehavior
	 * @param helper
	 * @see Helper#setTestMethodName(ITestCase, String, boolean)
	 */
	protected void computeTestMethodNames(ITestSuite testSuite, boolean isExternalBehavior, Helper helper) {
		Iterator it = testSuite.getITestCases().iterator();
		while (it.hasNext()) {
			ITestCase testCase = (ITestCase) it.next();
			String methodName = helper.computeTestMethodName(testCase, isExternalBehavior);
			helper.setTestMethodName(testCase, methodName, /*directModelAccess*/false);
		}
	}
	
	public IFile getFileHandle(ITestSuite ts) {
		IContainer sourceContainer = getSourceContainerHandle(ts);
		if (sourceContainer == null) return null;
		return sourceContainer.getFile(new Path(ts.getImplementor().getResource().replace('.','/') + ".java")); //$NON-NLS-1$
	}
			
	protected static Collection getDependentProjects(ITestSuite testSuite, IProject project)
	{
		if((testSuite.getIReferencedSuites() == null) || (testSuite.getIReferencedSuites().isEmpty()))
			return Collections.EMPTY_SET;
			
		Set entries = new HashSet(testSuite.getIReferencedSuites().size()+1);
		for (Iterator i = testSuite.getIReferencedSuites().iterator(); i.hasNext();)
		{
			ITestSuite referenced = (ITestSuite)i.next();
			if(referenced.getImplementor() == null)
				continue;
				
			String location = referenced.getImplementor().getLocation();
			if(location == null)
				continue;
				
			IPath path = new Path(location);
			IProject referencedProject = ResourcesPlugin.getWorkspace().getRoot().getProject(path.segment(0));
			if((referencedProject != null) && referencedProject.exists() && (!project.equals(referencedProject)))
				entries.add(referencedProject);
		}		
		return entries;
	}

}