/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JavaProjectDependencyUpdater.java,v 1.11 2010/03/17 16:01:53 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.core.internal.common.codegen;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.models.common.util.IDisposable;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.hyades.test.tools.core.internal.util.PluginProjectUtil;
import org.eclipse.jdt.core.IClasspathContainer;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.junit.JUnitCore;

/**
 * <p>JavaProjectDependencyUpdater.java</p>
 * 
 * 
 * @author  Paul Slauenwhite
 * @version March 17, 2010
 * @since   May 16, 2006
 */
public class JavaProjectDependencyUpdater implements IProjectDependencyUpdater, IDisposable {

	private Collection requiredPlugins;
	private Collection requiredProjects;
	
	public void addRequiredPlugin(String pluginId, String jarName) {
		if(requiredPlugins == null)
			requiredPlugins = new HashSet();
		requiredPlugins.add(pluginId);
	}

	public void addRequiredProject(IProject project) {
		if (requiredProjects == null) {
			requiredProjects = new HashSet();
		}
		requiredProjects.add(project);
	}

	/**
	 * Utility class used to describe the changes that need to be applied to a project classpath.
	 * @author jcanches
	 */
	private class ProjectClasspathChange {
		
		private Collection addedEntries;
		private Collection removedEntries;
		IJavaProject project;

		public ProjectClasspathChange(IJavaProject project, Collection addedEntries, Collection removedEntries) {
			this.addedEntries = addedEntries;
			this.removedEntries = removedEntries;
			this.project = project;
		}
		
		public void perform() throws JavaModelException {
			List rawClasspath = new ArrayList(Arrays.asList(project.getRawClasspath()));
			rawClasspath.removeAll(removedEntries);
			int pos = rawClasspath.size()-2;
			if(pos < 0)	pos = 0;
				
			rawClasspath.addAll(pos, addedEntries);
			
			project.setRawClasspath((IClasspathEntry[])rawClasspath.toArray(new IClasspathEntry[rawClasspath.size()]), null);
		}
		

        public Collection getPreview(){
            Collection list = addedEntries.isEmpty() ? removedEntries : addedEntries;
            if (!list.isEmpty()) {
                ArrayList ret = new ArrayList(addedEntries.size());
                for (Iterator it = addedEntries.iterator(); it.hasNext();) {
                    IClasspathEntry entry = (IClasspathEntry) it.next();
                    if (entry.getEntryKind() == IClasspathEntry.CPE_CONTAINER) {
                        try {
                            IClasspathContainer container = JavaCore.getClasspathContainer(entry.getPath(), project);
                            IClasspathEntry[] cpentries = container.getClasspathEntries();
                            for (int i = 0, max = cpentries.length; i < max; i++) {
                                ret.add(computePath(cpentries[i].getPath()));
                            }
                        } catch (JavaModelException e) {
                            CorePlugin.logError(e);
                            continue;
                        }
                    } else {
                        ret.add(entry.getPath().lastSegment());
                    }
                }
                return ret;
            }
            return Collections.EMPTY_LIST;
        }

        private String computePath(IPath path) {
            
        	if (path == null){
        		return null;
        	}
        	
            IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(path.makeAbsolute());
            String version = null;

            if (resource != null){
            	
            	path = resource.getLocation().makeAbsolute();
            	version = PluginProjectUtil.getBundleVersion(resource.getProject());
            }
                        
            return (version != null ? (path.lastSegment() + "<" + version + ">") : (path.lastSegment())); //$NON-NLS-1$ //$NON-NLS-2$
        }
    }

	
	private ProjectClasspathChange computeChange(IProject project) {
		if((requiredPlugins == null || requiredPlugins.isEmpty())
			&& (requiredProjects == null || requiredProjects.isEmpty())) return null;
				
		List additionalClasspathEntries = new ArrayList();
		List removedClasspathEntries = new ArrayList();
		
		IJavaProject javaProject = JavaCore.create(project);
		try
		{
			if (requiredProjects != null) {
				for(Iterator it = requiredProjects.iterator(); it.hasNext();) {
					IClasspathEntry entry = getProjectEntry((IProject)it.next());
					if(getConflictingEntry(javaProject, entry) == null) {
						additionalClasspathEntries.add(entry);
					}
				}
			}
			
			if (requiredPlugins != null) {
				ArrayList otherRequiredPlugins = new ArrayList(requiredPlugins);
				if (otherRequiredPlugins.remove("org.junit")) { //$NON-NLS-1$
					// For requirement org.junit, we need to add the JUnit container,
					// instead of adding all jars of this plugin to the TPTP container.
					IClasspathEntry newContainerEntry = JavaCore.newContainerEntry(JUnitCore.JUNIT4_CONTAINER_PATH);
					addContainer(javaProject, newContainerEntry, null, additionalClasspathEntries, removedClasspathEntries);
				}
				
				IClasspathEntry tptpContainerEntry = ClasspathContainerInitializer.getTPTPContainerEntry(javaProject);
				Set addedPlugins = new HashSet();
				if (tptpContainerEntry != null) {
					IClasspathContainer classpathContainer = JavaCore.getClasspathContainer(tptpContainerEntry.getPath(), javaProject);
					if (classpathContainer instanceof ClasspathContainerInitializer.ClasspathContainer) {
						ClasspathContainerInitializer.ClasspathContainer container = (ClasspathContainerInitializer.ClasspathContainer)classpathContainer;
						if (!container.contains(otherRequiredPlugins)) {
							addedPlugins.addAll(otherRequiredPlugins);
							addedPlugins.addAll(Arrays.asList(container.getPluginNames()));
							removedClasspathEntries.add(tptpContainerEntry);
						}
					}
				} else {
					addedPlugins.addAll(otherRequiredPlugins);
				}
				if (!addedPlugins.isEmpty()) {
					IClasspathEntry newContainerEntry = ClasspathContainerInitializer.newContainerEntry(addedPlugins);
					addContainer(javaProject, newContainerEntry, tptpContainerEntry, additionalClasspathEntries, removedClasspathEntries);
				}
			}
			
			if (additionalClasspathEntries.isEmpty() && removedClasspathEntries.isEmpty()) return null;
			return new ProjectClasspathChange(javaProject, additionalClasspathEntries, removedClasspathEntries);
		}
		catch(JavaModelException e)
		{
			CorePlugin.logError(e);
			return null;
		}
				
	}
	
	/**
	 * Adds a container entry to a Java project classpath. If there are simple entries (i.e.
	 * non-container entries) that conflict the container entries, removes them. If there
	 * are other containers whose entries conflict with the specified container, then the
	 * container is not added, and all entries that do not conflict with the existing
	 * java project classpath are added. In any scenario, the project is guaranted to
	 * contain all necessary jars specified in newContainerEntry.
	 * @param project A Java project
	 * @param newContainerEntry The container entry to add.
	 * @param oldContainerEntry A previously existing container entry that should be
	 * removed from the classpath, before adding the new entry. May be null.
	 * @param additionalClasspathEntries This list is filled by this method with all entries
	 * that need to be added to the classpath.
	 * @param removedClasspathEntries This list is filled by this method with all entries
	 * that need to be removed from the classpath. 
	 * @throws JavaModelException
	 */
	private void addContainer(IJavaProject project, IClasspathEntry newContainerEntry, IClasspathEntry oldContainerEntry, Collection additionalClasspathEntries, Collection removedClasspathEntries) throws JavaModelException {
		if (!containsContainerEntry(project, newContainerEntry)) {
			// The project does not contain already the new container
			// Search for entries in the project classpath that conflict with the new container
			Collection conflictingEntries = getConflictingEntries(project, newContainerEntry);
			Collection conflictingContainerEntries = getContainerEntries(conflictingEntries);
			if (conflictingContainerEntries.isEmpty()
				|| (oldContainerEntry != null
					&& conflictingContainerEntries.size() == 1
					&& oldContainerEntry.equals(conflictingContainerEntries.iterator().next()))) {
				// All conflicting entries are simple entries and can be removed
				// and replaced by the container
				removedClasspathEntries.addAll(conflictingEntries);
				additionalClasspathEntries.add(newContainerEntry);
			} else {
				// At least one conflicting entry belongs to a container, so we cannot
				// add the JUnit container, otherwise it would conflict.
				additionalClasspathEntries.addAll(getNonConflictingClasspathEntries(project, newContainerEntry));
			}
		}
	}
	
	public void adjustProject(IProject project, IProgressMonitor monitor) {
		try {
			ProjectClasspathChange change = computeChange(project);
			if (change != null) {
				change.perform();
			}
		} catch(JavaModelException e) {
			CorePlugin.logError(e);
			return;
		}
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		requiredPlugins.clear();
		requiredProjects.clear();
	}
	
	/**
	 * Returns the project classpath (raw) entry that conflicts with a given entry, if any.
	 * @param classpath A java project.
	 * @param entry An entry that is of any type except CPE_CONTAINER.
	 * @return
	 * @throws JavaModelException 
	 */
	private static IClasspathEntry getConflictingEntry(IJavaProject project, IClasspathEntry entry) throws JavaModelException {
		Assert.isLegal(entry.getEntryKind() != IClasspathEntry.CPE_CONTAINER);
		IClasspathEntry resolvedEntry = JavaCore.getResolvedClasspathEntry(entry);
		if (resolvedEntry == null) return null;
		IClasspathEntry[] rawEntries = project.getRawClasspath();
		for (int i = 0; i < rawEntries.length; i++) {
			IClasspathEntry rawEntry = rawEntries[i];
			if (rawEntry.getEntryKind() == IClasspathEntry.CPE_CONTAINER) {
				IClasspathContainer container = JavaCore.getClasspathContainer(rawEntry.getPath(), project);
				if (container != null) {
					IClasspathEntry[] entries = container.getClasspathEntries();
					for (int j = 0; j < entries.length; j++) {
						IClasspathEntry ref = JavaCore.getResolvedClasspathEntry(entries[j]);
						if (ref.getPath().equals(resolvedEntry.getPath())) {
							return rawEntry;
						}
					}
				}
			} else {
				IClasspathEntry ref = JavaCore.getResolvedClasspathEntry(rawEntry);
				if (ref.getPath().equals(resolvedEntry.getPath())) {
					return rawEntry;
				}
			}
		}
		return null;
	}
	
	/**
	 * Returns whether a project contains a given container entry in its classpath.
	 * @param project A Java project
	 * @param entry A container entry.
	 * @return <code>true</code> if the project contains the entry.
	 * @throws JavaModelException
	 */
	private static boolean containsContainerEntry(IJavaProject project, IClasspathEntry entry) throws JavaModelException {
		Assert.isLegal(entry.getEntryKind() == IClasspathEntry.CPE_CONTAINER);
		IClasspathEntry[] rawEntries = project.getRawClasspath();
		for (int i = 0; i < rawEntries.length; i++) {
			IClasspathEntry rawEntry = rawEntries[i];
			if (rawEntry.getEntryKind() == IClasspathEntry.CPE_CONTAINER
				&& rawEntry.getPath().equals(entry.getPath())) {
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Returns the list of raw entries, in a project classpath, that conflict with the
	 * entries of a container entry.
	 * @param project A Java project
	 * @param entry A container entry.
	 * @return the list of raw entries, in the project classpath, that conflict with the
	 * entries of entry.
	 * @throws JavaModelException
	 */
	private static Collection getConflictingEntries(IJavaProject project, IClasspathEntry entry) throws JavaModelException {
		Assert.isLegal(entry.getEntryKind() == IClasspathEntry.CPE_CONTAINER);
		HashSet ret = new HashSet();
		IClasspathContainer container = JavaCore.getClasspathContainer(entry.getPath(), project);
		IClasspathEntry[] entries = container.getClasspathEntries();
		for (int i = 0; i < entries.length; i++) {
			IClasspathEntry containerEntry = entries[i];
			IClasspathEntry conflict = getConflictingEntry(project, containerEntry);
			if (conflict != null) {
				ret.add(conflict);
			}
		}
		return ret;
	}
	
	/**
	 * Returns the entries that are container, within a list of classpath entries.
	 * @param entries
	 * @return
	 * @throws JavaModelException
	 */
	private static Collection getContainerEntries(Collection entries) {
		ArrayList ret = new ArrayList(entries.size());
		for (Iterator it = entries.iterator(); it.hasNext();) {
			IClasspathEntry entry = (IClasspathEntry) it.next();
			if (entry.getEntryKind() == IClasspathEntry.CPE_CONTAINER) {
				ret.add(entry);
			}
		}
		return ret;
	}
	
	/**
	 * Returns the non-conflicting entries of a container entry that do not conflict with
	 * a project classpath.
	 * @param project A java project
	 * @param entry A container entry.
	 * @return The entries, within entry, that do not conflict with project classpath.
	 * @throws JavaModelException
	 */
	private static Collection getNonConflictingClasspathEntries(IJavaProject project, IClasspathEntry entry) throws JavaModelException {
		Assert.isLegal(entry.getEntryKind() == IClasspathEntry.CPE_CONTAINER);
		ArrayList ret = new ArrayList();
		IClasspathContainer container = JavaCore.getClasspathContainer(entry.getPath(), project);
		IClasspathEntry[] entries = container.getClasspathEntries();
		for (int i = 0; i < entries.length; i++) {
			IClasspathEntry containerEntry = entries[i];
			IClasspathEntry conflict = getConflictingEntry(project, containerEntry);
			if (conflict == null) {
				ret.add(containerEntry);
			}
		}
		return ret;
	}
	
	protected static IClasspathEntry getProjectEntry(IProject referencedProject)
	{
		return JavaCore.newProjectEntry(referencedProject.getFullPath());
	}

	/**
	 * This implementation returns the collection of entries to be added, if any, or the 
	 * collection of entries to be removed, if there are only entries to remove. This is
	 * to conform with calls to {@link IProjectDependencyUpdater#previewAdjustProject(IProject)}
	 * that test whether the returned list is empty or not to decide if the project needs
	 * to be updated. See comments in the interface for more information.
	 */
	public Collection previewAdjustProject(IProject project) {
		ProjectClasspathChange change = computeChange(project);
		if (change != null) {
			return change.getPreview();
		}
		return Collections.EMPTY_LIST;
	}

}
