/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AssociateModelSourceChange.java,v 1.6 2007/05/02 19:36:26 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.core.internal.common.codegen.changes;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.test.core.internal.changes.PotentialChange;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.hyades.test.tools.core.internal.common.CommonPluginMessages;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.JUnitExtensionsRegistry;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.JUnitModelUpdater;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.JUnitProjectBuilder;
import org.eclipse.hyades.test.tools.core.java.IJUnitTestSuiteFactory;
import org.eclipse.hyades.test.tools.core.java.IJUnitTestSuiteUpdateFactory;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;

/**
 * Change that adds a JUnit association from a java file to a test suite model.
 * @author jcanches
 * @since 4.3
 */
public class AssociateModelSourceChange extends PotentialChange {
	
	private IFile javaFile;
	private ITestSuite testSuite;
	
	/**
	 * @param javaFile
	 * @param testSuite
	 */
	public AssociateModelSourceChange(IFile javaFile, ITestSuite testSuite) {
		this.javaFile = javaFile;
		this.testSuite = testSuite;
	}

	public Object getModifiedElement() {
		return javaFile;
	}

	public String getName() {
		return NLS.bind(CommonPluginMessages.ASSOCIATE_TEST_TO_SOURCE, new Object[] { testSuite.getName(), javaFile.getFullPath().toString() });
	}

	public void initializeValidationData(IProgressMonitor pm) {
		// NOP
	}

	public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException,	OperationCanceledException {
		return new RefactoringStatus();
	}

	public Change perform(IProgressMonitor pm) throws CoreException {
		pm.beginTask("", 3); //$NON-NLS-1$
		try {
			if (testSuite instanceof EObject) {
				EObject eObject = (EObject) testSuite;
				IFile testSuiteFile = EMFUtil.getWorkspaceFile(eObject);
				if (testSuiteFile != null && testSuiteFile.exists()) {
					String type = testSuite.getType();
					IJUnitTestSuiteFactory factory = JUnitExtensionsRegistry.getInstance().getFactory(type);
					if (factory != null && factory instanceof IJUnitTestSuiteUpdateFactory) {
						// Add the Junit builder to the file's project (if not already installed)
						JUnitProjectBuilder.installBuilder(javaFile.getProject());
					}
					pm.worked(1);
					// Also add the JUnit builder to the test suite's project (the source file
					// and the test suite usually are under the same project, but this is not
					// mandatory).
					JUnitProjectBuilder.installBuilder(testSuiteFile.getProject());
					pm.worked(1);
					// Record the link from the java file to the testsuite file
					JUnitModelUpdater.associateTestSuiteToJUnitSourceFile(javaFile, testSuiteFile, testSuite);
					pm.worked(1);
				}
			}
		} catch (Throwable t) {
			CorePlugin.logError(t);
		} finally {
			pm.done();
		}
		return null;
	}

	public boolean hasEffect() {
		if (!javaFile.exists()) return true;
		try {
			if (!JUnitProjectBuilder.isBuilderInstalled(javaFile.getProject())) return true;
			if (testSuite instanceof EObject) {
				EObject eObject = (EObject) testSuite;
				IFile testSuiteFile = EMFUtil.getWorkspaceFile(eObject);
				if (testSuiteFile != null && testSuiteFile.exists()) {
					if (!JUnitProjectBuilder.isBuilderInstalled(testSuiteFile.getProject())) return true;
				}
				IFile registeredTestSuiteFile = JUnitModelUpdater.getRegisteredTestSuiteFile(javaFile);
				if (registeredTestSuiteFile == null || !registeredTestSuiteFile.equals(testSuiteFile)) return true;
				return false;
			}
			return true;
		} catch (CoreException e) {
			CorePlugin.logError(e);
			return false;
		}
	}

}
