/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestLaunchConfigurationValidator.java,v 1.6 2008/04/18 20:12:35 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.core.internal.common.launch;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended;
import org.eclipse.hyades.execution.core.file.IFileManagerExtended.FileIdentifierList;
import org.eclipse.hyades.execution.local.file.FileManagerFactory;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.NodeFactory;
import org.eclipse.hyades.models.common.configuration.CFGArtifactLocationPair;
import org.eclipse.hyades.models.common.configuration.CFGClass;
import org.eclipse.hyades.models.common.configuration.CFGComparableProperty;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.configuration.CFGPropertyGroup;
import org.eclipse.hyades.models.common.configuration.util.ConfigurationUtil;
import org.eclipse.hyades.models.common.facades.behavioral.IImplementor;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.interactions.BVRProperty;
import org.eclipse.hyades.models.common.testprofile.TPFBehavior;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.TestCorePreferences;
import org.eclipse.hyades.test.core.internal.launch.extensions.LaunchConfigurationExtensionsManager;
import org.eclipse.hyades.test.core.internal.launch.extensions.LaunchDisposeListener;
import org.eclipse.hyades.test.core.launch.configurations.DeploymentLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.configurations.TestLaunchConfigurationFacade;
import org.eclipse.hyades.test.core.launch.extensions.ITestLaunchConfigurationValidator;
import org.eclipse.hyades.test.core.util.LocationUtil;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.hyades.test.tools.core.internal.common.CommonPluginMessages;
import org.eclipse.jdt.core.IJavaModelMarker;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.osgi.util.NLS;

/**
 * @see ITestLaunchConfigurationValidator
 */
public class TestLaunchConfigurationValidator implements ITestLaunchConfigurationValidator, LaunchDisposeListener {
    private Map diagnosticMap; //- Use for deployment validation that may be not so quick

    public TestLaunchConfigurationValidator() {
        LaunchConfigurationExtensionsManager.getInstance().addDisposeListener(this);
        this.diagnosticMap = new HashMap();
    }

    private static TPFTestSuite getTestSuite(ILaunchConfiguration launchConfiguration, final ResourceSet resourceSet) throws CoreException {
    	TPFTest test = TestLaunchConfigurationFacade.getTest(launchConfiguration, resourceSet);
    	if (test instanceof TPFTestCase) {
            return ((TPFTestCase) test).getTestSuite();
        } else if (test instanceof TPFTestSuite) {
            return (TPFTestSuite) test;
        }
    	// Unreachable
    	return null;
    }
    /**
     * @see ITestLaunchConfigurationValidator#validate
     */
    public Diagnostic validate(final ILaunchConfiguration launchConfiguration, final ResourceSet resourceSet) {
        List diagnostics = new LinkedList();
        try {
        	TPFTestSuite testSuite = getTestSuite(launchConfiguration, resourceSet);
            if (testSuite != null) {
            	IFile sourceFile = getBehaviorSourceFile(testSuite);
            	if (sourceFile == null || !sourceFile.exists()) {
            		// The code has not been generated yet. Issue an error
            		return createErrorDiagnostic(CommonPluginMessages._ERROR_LAUNCH_VALIDATOR);
            	}
            	IProject testProject = sourceFile.getProject();
            	if (isClasspathBroken(testProject)) {
            		diagnostics.add(createWarningDiagnostic(NLS.bind(CommonPluginMessages._WARNING_BUILD_PATH_PROBLEM, testProject.getName())));
            	}
            	if (hasCompileErrors(testProject)) {
            		diagnostics.add(createWarningDiagnostic(NLS.bind(CommonPluginMessages._WARNING_COMPILATION_PROBLEM, testProject.getName())));
            	}
            }
        } catch (CoreException e) {
            return createErrorDiagnostic(CommonPluginMessages._ERROR_LAUNCH_CONFIG_VALIDATION);
        }
        //- Now validate the deployment:
        diagnostics.add(validateDeployment(launchConfiguration, resourceSet));
        return findMostSevere(diagnostics);
    }
    
    private Diagnostic findMostSevere(List diagnosticList) {
        Diagnostic ret = createOKDiagnostic();
        Iterator it = diagnosticList.iterator();
        while (it.hasNext()) {
            ret = findMostSevere(ret, (Diagnostic) it.next());
        }
        return ret;
    }

    private Diagnostic findMostSevere(Diagnostic diagnostic1, Diagnostic diagnostic2) {
        if (diagnostic1.getSeverity() <= diagnostic2.getSeverity()) {
            return diagnostic1;
        }
        return diagnostic2;
    }

    private boolean isClasspathBroken(IProject p) throws CoreException {
        IMarker[] markers = p.findMarkers(IJavaModelMarker.BUILDPATH_PROBLEM_MARKER, false, IResource.DEPTH_ZERO);
        for (int i = 0, l = markers.length; i < l; i++) {
            if (((Integer) markers[i].getAttribute(IMarker.SEVERITY)).intValue() == IMarker.SEVERITY_ERROR) return true;
        }
        return false;
    }

    private boolean hasCompileErrors(IResource resource) throws JavaModelException {
        try {
            IMarker[] problemMarkers = resource.findMarkers(IJavaModelMarker.JAVA_MODEL_PROBLEM_MARKER, true, IResource.DEPTH_INFINITE);
            for (int i = 0; i < problemMarkers.length; i++) {
                if (problemMarkers[i].getAttribute(IMarker.SEVERITY, -1) == IMarker.SEVERITY_ERROR) return true;
            }
            return false;
        } catch (JavaModelException e) {
            throw e;
        } catch (CoreException e) {
            throw new JavaModelException(e);
        }
    }

    private IFile getBehaviorSourceFile(ITestSuite testSuite) {
        String implementorPathName = testSuite.getImplementor().getLocation();
        if (implementorPathName != null) {
            IPath implementorPath = new Path(implementorPathName);
            IContainer container = null;
            //- retrive the behavior file container
            if (implementorPath.segmentCount() == 1) {
                //- implementor has been generated directly under the project (no source folder) 
                container = ResourcesPlugin.getWorkspace().getRoot().getProject(implementorPath.toString());
            } else {
                //- in this case there is a source folder
                container = ResourcesPlugin.getWorkspace().getRoot().getFolder(implementorPath);
            }
            if (container != null) {
                return container.getFile(new Path(testSuite.getImplementor().getResource().replace('.', '/') + ".java")); //$NON-NLS-1$
            }
        }
        return null;
    }

    private static Diagnostic createOKDiagnostic() {
        return new Diagnostic() {
            public String getMessage() {
                return ""; //$NON-NLS-1$
            }

            public int getSeverity() {
                return OK;
            }
        };
    }

    private static Diagnostic createErrorDiagnostic(final String message) {
        return new Diagnostic() {
            public int getSeverity() {
                return ERROR;
            }

            public String getMessage() {
                return message;
            }
        };
    }

    private static Diagnostic createWarningDiagnostic(final String message) {
        return new Diagnostic() {
            public int getSeverity() {
                return WARNING;
            }

            public String getMessage() {
                return message;
            }
        };
    }

    /**
     * Bugzilla #95789: validation of the ROOTDIR value
     */
    Diagnostic validateDeployment(ILaunchConfiguration launchConfiguration, ResourceSet resourceSet) {
        Diagnostic ret = createOKDiagnostic();
        //- Now validate the deployment
        CFGLocation location = null;
        try {
            location = retrieveLocation(launchConfiguration, resourceSet);
            if (location != null) {
                Diagnostic diagnosticCached = (Diagnostic) diagnosticMap.get(location);
                if (diagnosticCached != null) {
                    return diagnosticCached; //- Diagnostic computed previously 
                }
                ret = validateLocation(location);
            }
        } catch (Exception e) {
            ret = createWarningDiagnostic(CommonPluginMessages._WARNING_LAUNCH_VALIDATOR);
            CorePlugin.logError(e);
        }
        if (ret != null && location != null) {
            diagnosticMap.put(location, ret);
        }
        return ret;
    }

    /**
     * Retrieves the location that will be used to deploy. This algorithm should be the same as 
     * implememented in AbstractLaunchConfigurationDelegate2.invokeTestExecutionHarness() to ensure 
     * that the validated location is the one on which the test will be executed.
     * 
     * @throws CoreException
     */
    private CFGLocation retrieveLocation(ILaunchConfiguration launchConfiguration, ResourceSet resourceSet) throws CoreException {
        TPFTest test = TestLaunchConfigurationFacade.getTest(launchConfiguration, resourceSet);
        TPFTestSuite testSuite = null;
        TPFTest subTest = null;
        if (test instanceof TPFTestCase) {
            subTest = test;
            testSuite = ((TPFTestCase) subTest).getTestSuite();
        } else if (test instanceof TPFTestSuite) {
            testSuite = (TPFTestSuite) test;
            subTest = test;
        }
        IImplementor theImplementor = null;
        if (subTest != null) {
            theImplementor = subTest.getImplementor();
        } else {
            theImplementor = testSuite.getImplementor();
        }
        CFGClass rootResource = null;
        ITest owningTest = theImplementor.getOwner();
        if (owningTest != null) {
            if (owningTest instanceof ITestSuite)
                rootResource = (CFGClass) owningTest;
            else {
                rootResource = (CFGClass) ((ITestCase) owningTest).getOwner();
            }
        } else {
            TPFTestComponent component = ((TPFBehavior) theImplementor).getTestComponent();
            if (component.eContainer() == null) {
                rootResource = component;
            } else {
                rootResource = (CFGClass) component.eContainer();
            }
        }
        TPFDeployment deployment = DeploymentLaunchConfigurationFacade.getDeployment(launchConfiguration, resourceSet);
        if (deployment != null && !deployment.getArtifactLocations().isEmpty()) {
            CFGArtifactLocationPair pair = ConfigurationUtil.searchPairWithTestAsset(rootResource, deployment);
            if (pair != null) {
                return pair.getLocation();
            }
        }
        return null;
    }

    /**
     * Validate a location using the file system service provided by the IFileManagerExtended interface.
     * 
     * @param location the location to be analysed
     * @return An error diagnostic if the ROOTDIR does not exist on the location.
     * @throws Exception
     * 
     * @author pnedelec
     * @since 4.2
     */
    public static Diagnostic validateLocation(CFGLocation location) throws Exception {
        //- Search the hostname
        String hostname = ConfigurationUtil.getHostName(location);
        if (hostname == null) {
            return createErrorDiagnostic(CommonPluginMessages.VALIDATION_NO_HOST_NAME_ERROR);
        }
        //- Search the value of the ROOTDIR property
        String userRootDir = null;
        CFGPropertyGroup propGroup = ConfigurationUtil.searchPropertyGroupById(location.getPropertyGroups(), ConfigurationUtil.ATTRS_PROP_GROUP_ID_LOCATION);
        if (propGroup == null) {
            return createOKDiagnostic();
        }
        BVRProperty[] properties = ConfigurationUtil.searchPropertiesByName(propGroup.getProperties(), LocationUtil.PROPERTY_NAME_ROOTDIR, false);
        if (properties == null || properties.length < 1) {
            return createOKDiagnostic();
        } else {
            userRootDir = ((CFGComparableProperty) properties[0]).getValue();
        }
        int port = Integer.parseInt(TestCorePlugin.getDefault().getPluginPreferences().getString(TestCorePreferences.DEFAULT_PORT));
        Node node = NodeFactory.createNode(hostname);
        IFileManagerExtended fileManager = FileManagerFactory.getInstance().create(node.connect(port));
        if (fileManager != null && userRootDir != null) {
            boolean[] res = fileManager.isDirectoryExistent((FileIdentifierList.create(userRootDir)));
            if (res.length > 0 && !res[0]) {
                return createErrorDiagnostic(NLS.bind(CommonPluginMessages.VALIDATION_BAD_ROOTDIR_ERROR_, new Object[] { userRootDir, hostname}));
            }
        }
        return createOKDiagnostic();
    }

    /** 
     * @see org.eclipse.hyades.test.core.internal.launch.extensions.LaunchDisposeListener#launchDisposed()
     */
    public void launchDisposed() {
        diagnosticMap.clear();
    }
}
