/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JUnitGenerator.java,v 1.19 2007/04/26 20:06:25 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.tools.core.internal.java.codegen;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.test.core.internal.changes.CompilationUnitChange;
import org.eclipse.hyades.test.core.internal.changes.CreateFileChange;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.ASTHelper;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.DelegateProjectDependencyUpdater;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.Helper;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.IProjectDependencyUpdater;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.ImportManager;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.JavaGenerator;
import org.eclipse.hyades.test.tools.core.internal.java.JavaMessages;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTParser;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.Javadoc;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.Name;
import org.eclipse.jdt.core.dom.SimpleType;
import org.eclipse.jdt.core.dom.SingleVariableDeclaration;
import org.eclipse.jdt.core.dom.TagElement;
import org.eclipse.jdt.core.dom.Type;
import org.eclipse.jdt.core.dom.TypeDeclaration;
import org.eclipse.jface.text.Document;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;
import org.eclipse.ltk.core.refactoring.NullChange;
import org.eclipse.text.edits.MalformedTreeException;
import org.eclipse.text.edits.TextEdit;

/**
 * JUnit code generator and code updater. This generator synchronizes the model behavior with
 * the static suite() method, and matches each test case with a test method.
 * This concrete implementation handles code generation and code update for JUnit and
 * JUnit Plugin types.
 * <p>If you do not need code synchronization, consider implementing {@link JavaGenerator}.</p>
 * @author marcelop, jcanches
 * @since 1.0.2
 */
public class JUnitGenerator extends JavaGenerator {
	
	private String superclassName;
	private boolean destructiveChangeFound;

	public JUnitGenerator(ITestSuite testSuite, IProjectDependencyUpdater updater) {
		super(testSuite, new JUnitProjectDependencyUpdater(updater, !testSuite.getImplementor().isExternalImplementor()));
	}
	
	public JUnitGenerator(ITestSuite testSuite, IProjectDependencyUpdater updater, String superclassName) {
		this(testSuite, updater);
		this.superclassName = superclassName;
	}
	
	protected Change createGenerateCodeChange(IFile file, IProgressMonitor monitor) throws CoreException {
		Helper helper = new Helper();
		try {
			String code = generateCode(helper, monitor);
			Change fileCreateChange = new CreateFileChange(file, code, CHARSET_UTF8);
			Change modelChange = helper.getMethodNamesChange();
			if (modelChange != null) {
				CompositeChange cchange = new CompositeChange("Composite source refactoring"); //$NON-NLS-1$
				cchange.add(fileCreateChange);
				cchange.add(modelChange);
				cchange.markAsSynthetic();
				return cchange;
			}
			return fileCreateChange;
		} finally {
			helper.dispose();
		}
	}

	/**
	 * @see org.eclipse.hyades.test.common.internal.codegen.Generator#generateFile(org.eclipse.hyades.models.common.facades.behavioral.ITestSuite, org.eclipse.core.resources.IFile, org.eclipse.core.runtime.IProgressMonitor)
	 */
	protected String generateCode(Helper helper, IProgressMonitor monitor) throws CoreException {
		if (this.superclassName != null) {
			helper.setSuperclassName(this.superclassName);
		}
		ITestSuite testSuite = getTestSuite();
		computeTestMethodNames(testSuite, testSuite.getImplementor().isExternalImplementor(), helper);
		GenTestSuite generator = new GenTestSuite();
		return Helper.formatContent(generator.generate(testSuite, helper));
	}
	
	protected Change createSourceUpdateChange(IFile file, SubProgressMonitor monitor) throws CoreException {
		monitor.beginTask("", 4); //$NON-NLS-1$
		try {
			ASTParser parser = ASTParser.newParser(AST.JLS3); // TODO Compute the argument
			ITestSuite testSuite = getTestSuite();
			ICompilationUnit compilationUnit = JavaCore.createCompilationUnitFrom(file);
			
			Helper helper = new Helper();
			String packageName = helper.getPackageName(testSuite);
			helper.setImportManager(new ImportManager(packageName));
			try {
				// Parse the source code
				destructiveChangeFound = false;
				Document doc = new Document(compilationUnit.getBuffer().getContents());
				parser.setSource(doc.get().toCharArray());
				CompilationUnit cu = (CompilationUnit) parser.createAST(new SubProgressMonitor(monitor, 1));
				cu.recordModifications();
				TypeDeclaration mainType = ASTHelper.getMainType(cu);
				if (mainType == null) {
					// TODO Error
					return new NullChange();
				}
				
				// Modify the AST tree
				updateTestMethods(testSuite, mainType, helper, new SubProgressMonitor(monitor, 1));
				// The generation of suite() must be performed after the test methods update
				// because the methods names might change during the update.
				if (!testSuite.getImplementor().isExternalImplementor()) {
					updateSuiteMethod(testSuite, mainType, helper);
					checkConstructor(mainType, helper);
					checkSuperType(mainType, helper);
				}
				monitor.worked(1);
				// Emit to AST the imports that are newly required by the updated code
				helper.emitSortedImports(cu);
				
				// Compute the resulting document and save it
				TextEdit edit = cu.rewrite(doc, /*options*/null);
				if (edit.getChildrenSize() != 0) {
					CompilationUnitChange change = new CompilationUnitChange(JavaMessages.UPDATE_JUNIT_CODE, compilationUnit, destructiveChangeFound);
					change.setEdit(edit);
					Change modelChange = helper.getMethodNamesChange();
					if (modelChange != null) {
						CompositeChange cchange = new CompositeChange("Composite source refactoring"); //$NON-NLS-1$
						cchange.add(change);
						cchange.add(modelChange);
						cchange.markAsSynthetic();
						return cchange;
					}
					return change;
				}
			} catch (JavaModelException e) {
				CorePlugin.logError(e);
			} catch (MalformedTreeException e) {
				CorePlugin.logError(e);
			} finally {
				helper.dispose();
			}
		} finally {
			monitor.done();
		}
		return new NullChange();
	}

	protected void checkSuperType(TypeDeclaration mainType, Helper helper) {
		boolean doIt = false;
		Type superclassType = mainType.getSuperclassType();
		if (superclassType == null) {
			doIt = true;
		}
		
		if (superclassType.isSimpleType()) {
			SimpleType st = (SimpleType)superclassType;
			Name supertypeName = ASTHelper.resolveName((CompilationUnit)mainType.getParent(), st.getName());
			if (supertypeName != null && !Helper.HYADES_TEST_CASE_CLASS_NAME.equals(supertypeName.getFullyQualifiedName())) {
				doIt = true;
			}
		}
		if (doIt) {
			destructiveChangeFound = true;
			helper.addImport(Helper.HYADES_TEST_CASE_CLASS_NAME);
			Name newName = mainType.getAST().newName(helper.getImportedName(Helper.HYADES_TEST_CASE_CLASS_NAME));
			if (superclassType == null) {
				superclassType = mainType.getAST().newSimpleType(newName);
				mainType.setSuperclassType(superclassType);
			} else {
				((SimpleType)superclassType).setName(newName);
			}
		}
	}
	
	/**
	 * Update the content of the suite() method for the specified class,
	 * with the behavior of the specified test suite.
	 * This implementation replaces the content of the suite() method, or creates
	 * the method if it does not exist.
	 * @throws JavaModelException 
	 */
	protected void updateSuiteMethod(ITestSuite testSuite, TypeDeclaration mainType, Helper helper) throws JavaModelException {
		// Compute the suite() method body
		GenSuiteMethod generator = new GenSuiteMethod();
		String content = generator.generate(testSuite, helper);
		MethodDeclaration newSuiteMethod = ASTHelper.parseMethod(mainType.getAST(), content);
		
		// Compute the existing suite() method index and removes it if it exists
		MethodDeclaration suiteMethod = ASTHelper.findMethodWithNoParameter(mainType, "suite"); //$NON-NLS-1$
		int index;
		if (suiteMethod == null) {
			index = ASTHelper.getFirstMethodIndex(mainType);
		} else {
			ITestSuite previousVersion = getRepositoryTestSuite();
			if (previousVersion != null && previousVersion.getImplementor().isExternalImplementor()) {
				destructiveChangeFound = true;
			}
			index = mainType.bodyDeclarations().indexOf(suiteMethod);
			mainType.bodyDeclarations().remove(index);
		}
		
		// Inserts the new MethodDeclaration at the right position
		if (index == -1) {
			mainType.bodyDeclarations().add(newSuiteMethod);
		} else {
			mainType.bodyDeclarations().add(index, newSuiteMethod);
		}
	}
	
	/**
	 * Ensures that there is a constructor with a String parameter in the specified
	 * class. If found, this method does nothing. Otherwise, it creates one.
	 * @param mainClass
	 * @param helper
	 * @throws JavaModelException 
	 */
	protected void checkConstructor(TypeDeclaration mainType, Helper helper) throws JavaModelException {
		MethodDeclaration pattern = mainType.getAST().newMethodDeclaration();
		pattern.setName(mainType.getAST().newSimpleName(mainType.getName().getIdentifier()));
		SingleVariableDeclaration svd = mainType.getAST().newSingleVariableDeclaration();
		svd.setType(mainType.getAST().newSimpleType(mainType.getAST().newSimpleName("String"))); //$NON-NLS-1$
		pattern.parameters().add(svd);
		
		MethodDeclaration constructor = ASTHelper.findMethod(mainType, pattern);
		if (constructor == null) {
			GenTestSuiteConstructor generator = new GenTestSuiteConstructor();
			String contents = generator.generate(mainType.getName().getIdentifier(), helper);
			constructor = ASTHelper.parseMethod(mainType.getAST(), contents);
			int index = ASTHelper.getFirstMethodIndex(mainType);
			if (index == -1) {
				mainType.bodyDeclarations().add(constructor);
			} else {
				mainType.bodyDeclarations().add(index, constructor);
			}
		}
	}
	
	/**
	 * Returns whether a method is a test method.
	 * @param method
	 * @return
	 */
	public static boolean isTestMethod(MethodDeclaration method) {
		String name = method.getName().getIdentifier();
		return name.startsWith("test") && method.parameters().size() == 0; //$NON-NLS-1$
	}
	
	/**
	 * Returns whether a method is a test method.
	 * @param method
	 * @return
	 */
	public static boolean isTestMethod(IMethod method) {
		String name = method.getElementName();
		String[] types = method.getParameterTypes();
		return name.startsWith("test") && types.length == 0; //$NON-NLS-1$
	}

	/**
	 * Update the list of test methods in the specified class, so that the list of
	 * test methods matches the list of test cases of the specified test suite.
	 * @param testSuite
	 * @param mainClass
	 * @param helper
	 * @throws JavaModelException
	 */
	protected void updateTestMethods(ITestSuite testSuite, TypeDeclaration mainType, Helper helper, IProgressMonitor monitor) throws JavaModelException {
		MethodDeclaration[] methods = mainType.getMethods();
		monitor.beginTask("", methods.length + testSuite.getITestCases().size()); //$NON-NLS-1$
		try {
			List usedTestMethods = new ArrayList(methods.length);
			// 1) Add missing test methods
			Iterator it = testSuite.getITestCases().iterator();
			while (it.hasNext()) {
				ITestCase testCase = (ITestCase) it.next();
				String actualMethodName = helper.getTestMethodName(testCase, false);
				String computedMethodName = helper.computeTestMethodName(testCase, testSuite.getImplementor().isExternalImplementor());
				if (actualMethodName == null) {
					// The test case has been added: use the computed method name
					actualMethodName = computedMethodName;
					helper.setTestMethodName(testCase, actualMethodName, /*immediatePerform*/false);
				}
				MethodDeclaration method = ASTHelper.findMethodWithNoParameter(mainType, actualMethodName);
				if (method == null) {
					method = createTestMethod(mainType, testCase, helper);
				} else {
					updateTestMethod(testCase, computedMethodName, method, helper);
				}
				usedTestMethods.add(method);
				monitor.worked(1);
			}
			// 2) Remove unused test methods
			for (int i = 0; i < methods.length; i++) {
				if (isTestMethod(methods[i]) && !usedTestMethods.contains(methods[i])) {
					destructiveChangeFound = true;
					mainType.bodyDeclarations().remove(methods[i]);
				}
				monitor.worked(1);
			}
		} finally {
			monitor.done();
		}
	}
	
	protected void updateTestMethod(ITestCase testCase, String newName, MethodDeclaration method, Helper helper) {
		// Update method name if necessary
		if (!newName.equals(method.getName().getIdentifier())) {
			destructiveChangeFound = true;
			method.setName(method.getAST().newSimpleName(newName));
			helper.setTestMethodName(testCase, newName, /*immediatePerform*/false);
		}
		// Update comments if necessary
		Javadoc javadoc = method.getJavadoc();
		if (javadoc == null) {
			javadoc = method.getAST().newJavadoc();
			method.setJavadoc(javadoc);
		}
		updateDescriptionTag(javadoc, testCase);
	}
	
	protected void updateDescriptionTag(Javadoc javadoc, ITestCase testCase) {
		TagElement oldtag = null;
		if (!javadoc.tags().isEmpty()) {
			TagElement tag = (TagElement) javadoc.tags().get(0);
			if (tag.getTagName() == null) {
				oldtag = tag;
			}
		}
		
		if (oldtag != null) {
			String sourceDescription = ASTHelper.extractDescription(oldtag);
			String modelDescription = makeJavadocDescription(testCase);
			if (!Helper.compareJavaComments(sourceDescription, modelDescription)) {
				destructiveChangeFound = true;
				javadoc.tags().remove(oldtag);
				ASTHelper.addDescriptionToJavadoc(javadoc, modelDescription);
			}
		} else {
			ASTHelper.addDescriptionToJavadoc(javadoc, makeJavadocDescription(testCase));
		}
	}
	
	public static String makeJavadocDescription(ITestCase testCase) {
		StringBuffer buf = new StringBuffer();
		buf.append(testCase.getName());
		if (testCase.getDescription() != null) {
			String descr = testCase.getDescription().trim();
			if (descr.length() > 0) {
				buf.append("\r\n"); //$NON-NLS-1$
				buf.append(descr);
			}
		}
		return buf.toString();
	}

	/**
	 * Creates a test method for the specified test case.
	 */
	protected MethodDeclaration createTestMethod(TypeDeclaration mainType, ITestCase testCase, Helper helper) throws JavaModelException {
		GenTestMethod generator = new GenTestMethod();
		String content = generator.generate(testCase, helper);
		MethodDeclaration method = ASTHelper.parseMethod(mainType.getAST(), content);
		mainType.bodyDeclarations().add(method);
		return method;
	}
	
	public static class JUnitProjectDependencyUpdater extends DelegateProjectDependencyUpdater {

		public JUnitProjectDependencyUpdater(IProjectDependencyUpdater delegate, boolean isModelBehavior) {
			super(delegate);
			addRequiredPlugin("org.junit", "junit.jar"); //$NON-NLS-1$ //$NON-NLS-2$
			if (isModelBehavior) {
				addRequiredPlugin(CorePlugin.getID(), "common.runner.jar"); //$NON-NLS-1$
				addRequiredPlugin(CorePlugin.getID(), "java.runner.jar"); //$NON-NLS-1$
			}
		}

	}
	
}
