/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JUnitExecutionDeploymentAdapter.java,v 1.6 2007/05/03 01:48:23 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.core.internal.java.harness;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.hyades.execution.harness.JavaExecutionDeploymentAdapter;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.hyades.test.tools.core.internal.util.PluginProjectUtil;
import org.eclipse.hyades.test.tools.core.java.JUnitTestSuiteFacade;

public class JUnitExecutionDeploymentAdapter extends JavaExecutionDeploymentAdapter {

	/**
	 * Returns the collection of files in a plugin project that need to
	 * be copied to the remote directory. The build.properties is used to
	 * determine the list. An empty collection is returned if the project is
	 * not a PDE project, or does not have a build.properties files.
	 * @param project A plugin project
	 * @return A collection of IFile that belong to project.
	 * @throws CoreException 
	 */
	protected static Collection getPluginFiles(IProject project) throws CoreException {
		String[] filePaths = PluginProjectUtil.getDeployables(project);
		List ret = new ArrayList();
		for (int i = 0; i < filePaths.length; i++) {
			String path = filePaths[i];
			if (".".equals(path)) { //$NON-NLS-1$
				// Special entry defining the plugin itself. We ignore it, since
				// we are deploying the plugin unjarred.
				continue;
			}
			if (path.charAt(path.length()-1) == '/') {
				// This is directory
				addIfExist(project.getFolder(path), ret);
			} else {
				addIfExist(project.getFile(path), ret);
			}
		}
		return ret;
	}

	protected static void addIfExist(IResource resource, Collection col) throws CoreException {
		if (resource.exists()) {
			if (resource.getType() == IResource.FILE) {
				col.add(resource);
			} else if (resource.getType() == IResource.FOLDER) {
				// Add recursively each folder member
				IFolder folder = (IFolder)resource;
				IResource[] members = folder.members();
				for(int i = 0; i < members.length; i++) {
					addIfExist(members[i], col);
				}
			}
		}
	}

	/**
	 * Utility method to add a file from the workspace, if it exists, to the deployable files.
	 */
	protected void addPluginFile(boolean isStandalone, IFile file, HashMap deployableFiles, String deployRootDir) {
		if (file.exists()) {
			String srcFile = file.getLocation().toString();
			String destFile = getDeployFilePath(srcFile, deployRootDir, isStandalone);
			deployableFiles.put(srcFile, destFile);
		}
	}
	
	/**
	 * This implementation collects all files referenced in the junit source file project,
	 * if this project is a plug-in project. Otherwise, it does nothing.
	 * See bugzilla_140715 for details.
	 */
	protected void collectTestSuiteFiles(boolean isStandalone, HashMap deployableFiles, TPFTestSuite testSuite, CFGLocation location, String deployRootDir) {
		// Compute the project that contains the test class
		IFile sourceFile = JUnitTestSuiteFacade.getJUnitSourceFile(testSuite);
		if (sourceFile == null) return;
		IProject project = sourceFile.getProject();
		try {
			for (Iterator it = getPluginFiles(project).iterator(); it.hasNext();) {
				IFile file = (IFile) it.next();
				addPluginFile(isStandalone, file, deployableFiles, deployRootDir);
			}
		} catch (CoreException e) {
			// Don't stop the deployment, but log the problem
			CorePlugin.logError(e);
		}
	}
	
}
