/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JUnitExtensionsRegistry.java,v 1.3 2007/05/02 19:36:27 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.core.internal.java.modelsync;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.hyades.test.tools.core.java.IJUnitTestSuiteFactory;

/**
 * A singleton registry that manages registered extensions to TPTP-JUnit extension points.
 * @author jcanches
 * @since 4.3
 */
public class JUnitExtensionsRegistry {

	private static JUnitExtensionsRegistry instance;
	private final static String EXT_POINT_ID = CorePlugin.getID() + ".junitFactory"; //$NON-NLS-1$
	
	public static JUnitExtensionsRegistry getInstance() {
		if (instance == null) {
			instance = new JUnitExtensionsRegistry();
		}
		return instance;
	}
	
	private Map factoryDescriptors;
	
	private JUnitExtensionsRegistry() {
		registerFactories();
	}
	
	public class JUnitTestSuiteFactoryDescriptor {
		private IConfigurationElement element;
		private IJUnitTestSuiteFactory factory;
		public JUnitTestSuiteFactoryDescriptor(IConfigurationElement element) {
			this.element = element;
		}
		public String getType() {
			return element.getAttribute("type"); //$NON-NLS-1$
		}
		public boolean isUpdateCapable() {
			return Boolean.valueOf(element.getAttribute("updateSupport")).booleanValue(); //$NON-NLS-1$
		}
		public IJUnitTestSuiteFactory getFactory() {
			if (factory == null) {
				try {
					factory =(IJUnitTestSuiteFactory)element.createExecutableExtension("class"); //$NON-NLS-1$
				} catch (CoreException e) {
					CorePlugin.logError(e);
					invalidFactory(this);
				}
			}
			return factory;
		}
		public boolean equals(Object obj) {
			if (!(obj instanceof JUnitTestSuiteFactoryDescriptor)) return false;
			if (getType() == null) return false; // This is an error case
			JUnitTestSuiteFactoryDescriptor other = (JUnitTestSuiteFactoryDescriptor) obj;
			return getType().equals(other.getType());
		}
		public int hashCode() {
			String type = getType();
			if (type == null) return super.hashCode();
			else return type.hashCode();
		}
		
	}
	
	protected void invalidFactory(JUnitTestSuiteFactoryDescriptor descr) {
		factoryDescriptors.remove(descr.getType());
	}
	
	public IJUnitTestSuiteFactory getFactory(String type) {
		JUnitTestSuiteFactoryDescriptor descr = getDescriptor(type);
		if (descr != null) {
			return descr.getFactory();
		}
		return null;
	}
	
	public JUnitTestSuiteFactoryDescriptor getDescriptor(String type) {
		return (JUnitTestSuiteFactoryDescriptor) factoryDescriptors.get(type);
	}
	
	public JUnitTestSuiteFactoryDescriptor[] getAvailableFactories(boolean updateSupport) {
		Collection list;
		if (updateSupport) {
			list = getUpdateCapableFactories();
		} else {
			list = factoryDescriptors.values();
		}
		JUnitTestSuiteFactoryDescriptor[] ret = new JUnitTestSuiteFactoryDescriptor[list.size()];
		return (JUnitTestSuiteFactoryDescriptor[]) factoryDescriptors.values().toArray(ret);
	}
	
	private Collection getUpdateCapableFactories() {
		Collection values = factoryDescriptors.values();
		ArrayList ret = new ArrayList(values.size());
		Iterator it = values.iterator();
		while (it.hasNext()) {
			JUnitTestSuiteFactoryDescriptor descr = (JUnitTestSuiteFactoryDescriptor) it.next();
			if (descr.isUpdateCapable()) {
				ret.add(descr);
			}
		}
		return ret;
	}

	private void registerFactories() {
		factoryDescriptors = new HashMap();
		IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint(EXT_POINT_ID);
		IExtension[] extensions = point.getExtensions();
		for (int i = 0; i < extensions.length; i++) {
			IExtension extension = extensions[i];
			IConfigurationElement[] elements = extension.getConfigurationElements();
			for (int j = 0; j < elements.length; j++) {
				IConfigurationElement element = elements[j];
				JUnitTestSuiteFactoryDescriptor descr = new JUnitTestSuiteFactoryDescriptor(element);
				factoryDescriptors.put(descr.getType(), descr);
			}
		}
	}
}
