/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JUnitModelUpdater.java,v 1.9 2007/05/03 01:40:08 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.core.internal.java.modelsync;

import java.util.Iterator;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.resources.WorkspaceJob;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.models.common.configuration.CFGClass;
import org.eclipse.hyades.models.common.configuration.CFGInstance;
import org.eclipse.hyades.models.common.facades.behavioral.IBlock;
import org.eclipse.hyades.models.common.facades.behavioral.IImplementor;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.fragments.BVRInteraction;
import org.eclipse.hyades.models.common.interactions.BVRLifeline;
import org.eclipse.hyades.models.common.testprofile.TPFBehavior;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.event.JUnitTestSuiteCreatedEvent;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.event.JUnitTestSuiteDetachedEvent;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.event.JUnitTestSuiteFactoryEventManager;
import org.eclipse.hyades.test.tools.core.java.IJUnitTestSuiteFactory;
import org.eclipse.hyades.test.tools.core.java.IJUnitTestSuiteUpdateFactory;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;

/**
 * This class provides methods for creating/updating a JUnit TestSuite model
 * from a JUnit source code (this is the counterpart of
 * {@link org.eclipse.hyades.test.tools.core.internal.java.codegen.JUnitGenerator}).
 * @author jcanches
 */
public class JUnitModelUpdater {

	private final static String TPTP_JUNIT_TEST_MARKER = "org.eclipse.hyades.test.tools.core.junit"; //$NON-NLS-1$
	private final static String TPTP_JUNIT_DEFINITION_ATTRIBUTE = "definition"; //$NON-NLS-1$
	
	/**
	 * Update the Test Suite model in order to take the last changes from the code
	 * into account. This is done by invoking the registered update factory for the type
	 * of the test suite. If no update factory is registered, this method does nothing and return
	 * <code>false</code>.
	 * @param testSuite
	 * @param javaFile 
	 * @return <code>true</code> if any update was applied to the specified test suite,
	 * or <code>false</code> if the test suite was already up-to-date.
	 * @throws CoreException
	 */
	public static boolean updateTestSuiteFromCode(ITestSuite testSuite, IFile junitCode) throws CoreException {
		IJUnitTestSuiteFactory factory = JUnitExtensionsRegistry.getInstance().getFactory(testSuite.getType());
		if (factory instanceof IJUnitTestSuiteUpdateFactory) {
			IJUnitTestSuiteUpdateFactory ufactory = (IJUnitTestSuiteUpdateFactory) factory;
			ICompilationUnit cu = JavaCore.createCompilationUnitFrom(junitCode);
			return ufactory.update(testSuite, cu);
		}
		return false;
	}
	
	/**
	 * This method is temporarily public. Clients should not use it.
	 * @param implementor
	 * @param block
	 */
	public static void setImplementorBlock(IImplementor implementor, IBlock block) {
		// TODO Ask to add IImplementor.setBlock(IBlock) the behavioral facade
		if (implementor instanceof TPFBehavior &&
			 (block == null || block instanceof BVRInteraction)) {
			TPFBehavior testBehavior = (TPFBehavior)implementor;
			BVRInteraction interaction = (BVRInteraction)block;
			// 1) Remove any instance that is referenced by a lifeline
			// of the previous interaction
			TPFTest test = testBehavior.getTest();
			BVRInteraction previousInteraction = testBehavior.getInteraction();
			if (test instanceof CFGClass && previousInteraction != null) {
				CFGClass theClass = (CFGClass) test;
				Iterator it = theClass.getInstances().iterator();
				while (it.hasNext()) {
					CFGInstance instance = (CFGInstance) it.next();
					BVRLifeline lifeline = instance.getLifeline();
					if (lifeline != null && previousInteraction.equals(lifeline.getInteraction())) {
						it.remove();
						instance.setClassType(null);
					}
				}
			}
			// 2) Set the new interaction
			testBehavior.setInteraction(interaction);
			// 3) Reconnect any dangling reference to the test
			if (test instanceof CFGClass && interaction != null) {
				Iterator it = interaction.getLifelines().iterator();
				while (it.hasNext()) {
					BVRLifeline lifeline = (BVRLifeline) it.next();
					Iterator it2 = lifeline.getInstances().iterator();
					while (it2.hasNext()) {
						CFGInstance instance = (CFGInstance) it2.next();
						if (instance.getClassOwner() == null) {
							instance.setClassOwner((CFGClass)test);
							instance.setClassType((CFGClass)test);
						}
					}
				}
			}
		}
	}

	/**
	 * Finds the TestSuite that is using the specified JUnit source code as its
	 * behavior.
	 * @param javaFile
	 * @return
	 */
	public static TPFTestSuite findTestSuite(IType type, ResourceSet resourceSet) {
		try {
			IResource res = type.getUnderlyingResource();
			if (res == null || !(res instanceof IFile))
				return null;
			IFile testSuiteFile = findTestSuiteFile((IFile)res);
			if (testSuiteFile != null) {
				EObject[] eObjects = EMFUtil.getEObjects(resourceSet,
						testSuiteFile, /*loadOnDemand*/true);
				for (int i = 0; i < eObjects.length; i++) {
					if (eObjects[i] instanceof TPFTestSuite) {
						TPFTestSuite tSuite = (TPFTestSuite) eObjects[i];
						IImplementor implementor = tSuite.getImplementor();
						if (implementor != null && type.getFullyQualifiedName().equals(implementor.getResource())) {
							return tSuite;
						}
					}
				}
			}
			return null;
		} catch (Throwable e) {
			CorePlugin.logError(e);
			return null;
		}
	}
	
	private static class RemoveMarkerJob extends WorkspaceJob {
		private IFile file;
		public RemoveMarkerJob(IFile file) {
			super("Removing obsolete markers"); //$NON-NLS-1$ // because it is a system job
			this.file = file;
			setSystem(true);
			setRule(file);
		}
		public IStatus runInWorkspace(IProgressMonitor monitor) throws CoreException {
			monitor.beginTask("", 1); //$NON-NLS-1$
			try {
				removeMarkers(file);
			} finally {
				monitor.done();
			}
			return new Status(IStatus.OK, CorePlugin.PLUGIN_ID, 0, "", null); //$NON-NLS-1$
		}
	}
	
	protected static void removeMarkers(IFile file) throws CoreException {
		file.deleteMarkers(TPTP_JUNIT_TEST_MARKER, /*includeSubTypes*/false, /*depth*/0);
		JUnitTestSuiteDetachedEvent event = new JUnitTestSuiteDetachedEvent(file);
		JUnitTestSuiteFactoryEventManager.getInstance().fireJUnitTestSuiteGenerated(event);
	}
	
	/**
	 * Finds the existing file associated to the given java file as the test suite file.
	 * @param javaFile A java file.
	 * @return A file that contains the test suite model whose java file implements the
	 * behavior, or <code>null</code> if none. The returned file is guaranted to exist.
	 * @throws CoreException
	 */
	public static IFile findTestSuiteFile(final IFile javaFile) throws CoreException {
		IFile file = getRegisteredTestSuiteFile(javaFile);
		if (file != null) {
			if (file.exists()) {
				return file;
			}
			// As we discover that the java file refers to a test suite that no longer
			// exists, schedule an asynchronous task that removes the marker, and triggers
			// the appropriate event.
			if (ResourcesPlugin.getWorkspace().isTreeLocked()) {
				Job job = new RemoveMarkerJob(javaFile);
				job.schedule();
			} else {
				ResourcesPlugin.getWorkspace().run(new IWorkspaceRunnable() {
					public void run(IProgressMonitor monitor) throws CoreException {
						removeMarkers(javaFile);
					}
				}, file, IWorkspace.AVOID_UPDATE, new NullProgressMonitor());
			}
		}
		return null;
	}
	
	/**
	 * Returns the test suite file that is registered as associated to the given java file.
	 * @param javaFile A java file.
	 * @return The registered test suite file (even if it does not exist), or <code>null</code> if no test
	 * suite file is registered for this java file.
	 * @throws CoreException
	 */
	public static IFile getRegisteredTestSuiteFile(IFile javaFile) throws CoreException {
		IMarker[] markers = javaFile.findMarkers(TPTP_JUNIT_TEST_MARKER, /*includeSubTypes*/false, 0);
		if (markers.length > 0) {
			String testSuiteDefinition = markers[0].getAttribute(TPTP_JUNIT_DEFINITION_ATTRIBUTE, null);
			if (testSuiteDefinition != null) {
				return ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(testSuiteDefinition));
			}
		}
		return null;
	}
	
	/**
	 * Marks the specified junit source file as being associated to a TPTP JUnit test. This method
	 * does not perform any workspace operation if the association is already up-to-date.
	 * @param junitSourceFile A Java file.
	 * @param testSuiteFile A file that contains a JUnit TestSuite definition.
	 * @parma testSuite The test suite, within testSuiteFile, that is associated to the source code.
	 * @throws CoreException If one of the resources do not exist, or if they belong to a closed
	 * project.
	 */
	public static void associateTestSuiteToJUnitSourceFile(final IFile junitSourceFile, IFile testSuiteFile, ITestSuite testSuite) throws CoreException {
		final String definition = testSuiteFile.getFullPath().toString();
		IMarker[] markers = junitSourceFile.findMarkers(TPTP_JUNIT_TEST_MARKER, /*includeSubTypes*/false, /*depth*/0);
		boolean removeExistingMarkers = false;
		if (markers.length > 0) {
			String defAttr = markers[0].getAttribute(TPTP_JUNIT_DEFINITION_ATTRIBUTE, null);
			if (definition.equals(defAttr)) {
				// Already up-to-date
				return;
			} else {
				// Get rid of all existing markers
				removeExistingMarkers = true;
			}
		}
		final boolean doRemoveExistingMarkers = removeExistingMarkers;
		ResourcesPlugin.getWorkspace().run(new IWorkspaceRunnable() {		
			public void run(IProgressMonitor monitor) throws CoreException {
				if (doRemoveExistingMarkers) {
					junitSourceFile.deleteMarkers(TPTP_JUNIT_TEST_MARKER, /*includeSubTypes*/false, /*depth*/0);
				}
				IMarker marker = junitSourceFile.createMarker(TPTP_JUNIT_TEST_MARKER);
				marker.setAttribute(TPTP_JUNIT_DEFINITION_ATTRIBUTE, definition);
			}		
		}, /*rule*/junitSourceFile, /*flags*/0, null);
		JUnitTestSuiteCreatedEvent event = new JUnitTestSuiteCreatedEvent(testSuite, junitSourceFile);
		JUnitTestSuiteFactoryEventManager.getInstance().fireJUnitTestSuiteGenerated(event);
	}
	
}
