/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JUnitProjectBuilder.java,v 1.13 2007/05/03 01:48:40 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.core.internal.java.modelsync;

import java.util.Map;

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IResourceProxy;
import org.eclipse.core.resources.IResourceProxyVisitor;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.models.common.util.SaveManager;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.hyades.test.tools.core.internal.java.JavaMessages;
import org.eclipse.hyades.test.tools.core.java.JUnitTestSuiteFacade;
import org.eclipse.jdt.core.JavaModelException;

public class JUnitProjectBuilder extends IncrementalProjectBuilder {

	public final static String BUILDER_ID = CorePlugin.PLUGIN_ID + ".java.junit.builder"; //$NON-NLS-1$
	
	protected IProject[] build(int kind, Map args, IProgressMonitor monitor) throws CoreException {
		if (kind == IncrementalProjectBuilder.FULL_BUILD) {
			fullBuild(monitor);
		} else {
			IResourceDelta delta = getDelta(getProject());
			if (delta == null) {
				fullBuild(monitor);
			} else {
				incrementalBuild(delta, monitor);
			}
		}
		return null;
	}

	private void fullBuild(IProgressMonitor monitor) {
		monitor.beginTask(JavaMessages.BUILDER_LINK_TASK, 1);
		try {
			setupTestSuiteBacklinks(getProject());
			monitor.worked(1);
		} finally {
			monitor.done();
		}
	}

	private void incrementalBuild(final IResourceDelta delta, IProgressMonitor monitor) {
		try {
			ResourcesPlugin.getWorkspace().run(new IWorkspaceRunnable() {
				public void run(IProgressMonitor monitor) throws CoreException {
					monitor.beginTask("", IProgressMonitor.UNKNOWN); //$NON-NLS-1$
					try {
						ResourceDeltaVisitor visitor = new ResourceDeltaVisitor();
						delta.accept(visitor);
						if (visitor.needsToPurgeJavaMarkers()) {
							purgeJavaMarkers(getProject());
							monitor.worked(1);
						}
					} finally {
						monitor.done();
					}
				}
			}, this.getProject(), 0, monitor);
		} catch (CoreException e) {
			CorePlugin.logError(e);
		}
	}
	
	private class ResourceDeltaVisitor implements IResourceDeltaVisitor {

		private boolean purgeJavaMarkers = false;
		
		public boolean visit(IResourceDelta delta) throws CoreException {
			switch (delta.getKind()) {
				case IResourceDelta.CHANGED: {
					IResource res = delta.getResource();
					if (res.getType() == IResource.FILE) {
						IFile file = (IFile)res;
						if (isJavaFile(file)) {
							javaFileChanged(file);
						} else if (ICommonConstants.TEST_SUITE_FILE_EXTENSION.equals(file.getFileExtension())) {
							setupTestSuiteBacklink(file);
						}
					}
					break;
				}
				case IResourceDelta.ADDED: {
					IResource res = delta.getResource();
					if (res.getType() == IResource.FILE
						&& ICommonConstants.TEST_SUITE_FILE_EXTENSION.equals(res.getFileExtension())) {
						setupTestSuiteBacklink((IFile)res);
					}
					break;
				}
				case IResourceDelta.REMOVED: {
					IResource res = delta.getResource();
					if (res.getType() == IResource.FILE
						&& ICommonConstants.TEST_SUITE_FILE_EXTENSION.equals(res.getFileExtension())) {
						purgeJavaMarkers = true;
					}
					break;
				}
			}
			return true;
		}
		
		public boolean needsToPurgeJavaMarkers() {
			return purgeJavaMarkers;
		}
		
	}
	
	private static boolean isJavaFile(IFile resource) {
		String ext = resource.getFileExtension();
		if (ext != null) {
			return ext.toLowerCase().equals("java"); //$NON-NLS-1$
		}
		return false;
	}
	
	private void javaFileChanged(IFile javaFile) {
		try {
			TPFTestSuite testSuite = (TPFTestSuite)JUnitTestSuiteFacade.findTestSuite(javaFile, new ResourceSetImpl());
			if (testSuite != null) {
				if (JUnitModelUpdater.updateTestSuiteFromCode(testSuite, javaFile)) {
					SaveManager.saveResource(testSuite);
					IFile file = EMFUtil.getWorkspaceFile(testSuite);
					file.refreshLocal(0, null);
				}
			}
		} catch (JavaModelException e) {
			// Keep silent. Parsing the java file failed, so we ignore this file. It will
			// synchronize the next time it is saved without compile errors.
		} catch (Throwable t) {
			// Something went wrong -- the test suite associated to the java file
			// (if any) won't be updated.
			CorePlugin.logError(t);
		}
	}
	
	private void purgeJavaMarkers(IContainer container) {
		try {
			container.accept(new IResourceProxyVisitor() {
				public boolean visit(IResourceProxy proxy) /*throws CoreException*/ {
					if (proxy.getType() == IResource.FILE) {
						if (proxy.getName().toLowerCase().endsWith(".java")) { //$NON-NLS-1$
							IFile javaFile = (IFile)proxy.requestResource();
							// The simple call below will trigger the marker purge if
							// necessary.
							try {
								JUnitModelUpdater.findTestSuiteFile(javaFile);
							} catch (CoreException e) {
								CorePlugin.logError(e);
							}
						}
						return false;
					}
					return true;
				}
			}, IResource.NONE);
		} catch (CoreException e) {
			// Should not happen but anyway...
			CorePlugin.logError(e);
		}
	}
	
	private void setupTestSuiteBacklinks(IContainer container) {
		final String FILE_NAME_END = ICommonConstants.TEST_SUITE_FILE_EXTENSION;
		try {
			container.accept(new IResourceProxyVisitor() {
				public boolean visit(IResourceProxy proxy) throws CoreException {
					if (proxy.getType() == IResource.FILE) {
						if (proxy.getName().endsWith(FILE_NAME_END)) {
							setupTestSuiteBacklink((IFile)proxy.requestResource());
						}
						return false;
					}
					return true;
				}
			}, IResource.NONE);
		} catch (CoreException e) {
			// Should not happen but anyway...
			CorePlugin.logError(e);
		}
	}

	private void setupTestSuiteBacklink(IFile file) {
		try {
			ResourceSet resourceSet = new ResourceSetImpl();
			EObject[] objects = EMFUtil.getEObjects(resourceSet, file, /*loadOnDemand*/true);
			for (int i = 0; i < objects.length; i++) {
				if (objects[i] instanceof ITestSuite) {
					ITestSuite testSuite = (ITestSuite)objects[i];
					IFile sourceFile = JUnitTestSuiteFacade.getJUnitSourceFile(testSuite);
					if (sourceFile != null) {
						JUnitModelUpdater.associateTestSuiteToJUnitSourceFile(sourceFile, file, testSuite);
					}
					// We expect at most one test suite per file and we found it.
					break;
				}
			}
		} catch (Throwable t) {
			// In a builder, a failure should not be intrusive.
			CorePlugin.logError(t);
		}
	}
	
	public static void installBuilder(IProject project) throws CoreException {
		IProjectDescription d = project.getDescription();
		ICommand[] buildSpec = d.getBuildSpec();
		for (int i = 0; i < buildSpec.length; i++) {
			if (buildSpec[i].getBuilderName().equals(BUILDER_ID)) {
				return;
			}
		}
		buildSpec = new ICommand[d.getBuildSpec().length + 1];
		System.arraycopy(d.getBuildSpec(), 0, buildSpec, 0, d.getBuildSpec().length);
		ICommand command = d.newCommand();
		command.setBuilderName(BUILDER_ID);
		buildSpec[buildSpec.length-1] = command;
		d.setBuildSpec(buildSpec); 
		project.setDescription(d, null);
	}
	
	public static void uninstallBuilder(IProject project) throws CoreException {
		IProjectDescription d = project.getDescription();
		ICommand[] buildSpec = d.getBuildSpec();
		int found = 0;
		for (int i = 0; i < buildSpec.length; i++) {
			if (buildSpec[i].getBuilderName().equals(BUILDER_ID)) {
				buildSpec[i] = null;
				found++;
			}
		}
		ICommand[] newCommands = new ICommand[buildSpec.length - found];
		int j = 0;
		for (int i = 0; i < buildSpec.length; i++) {
			if (buildSpec[i] != null) {
				newCommands[j++] = buildSpec[i];
			}
		}
		d.setBuildSpec(newCommands); 
		project.setDescription(d, null);
	}
	
	public static boolean isBuilderInstalled(IProject project) throws CoreException {
		IProjectDescription d = project.getDescription();
		ICommand[] buildSpec = d.getBuildSpec();
		for (int i = 0; i < buildSpec.length; i++) {
			if (buildSpec[i].getBuilderName().equals(BUILDER_ID)) {
				return true;
			}
		}
		return false;
	}

}
