/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JUnitTypeFactory.java,v 1.5 2007/07/30 21:30:46 jptoomey Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.core.internal.java.modelsync;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceImpl;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.facades.behavioral.impl.HyadesFactory;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.hyades.test.tools.core.common.TestCommon;
import org.eclipse.hyades.test.tools.core.common.util.TestCommonUtil;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.ASTHelper;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.Helper;
import org.eclipse.hyades.test.tools.core.internal.java.codegen.JUnitGenerator;
import org.eclipse.hyades.test.tools.core.java.IJUnitTestSuiteUpdateFactory;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.ITypeHierarchy;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.dom.TagElement;

/**
 * This test suite factory handles the generation and update of test suite models
 * of type "TPTP JUnit".
 * @author jcanches
 * @since 4.3
 */
public class JUnitTypeFactory implements IJUnitTestSuiteUpdateFactory {

	/**
	 * Sub-classes should redefine this method to support an additional type.
	 * @return The test suite type that this factory can generate.
	 */
	protected String getTestSuiteType() {
		return TestCommon.JUNIT_TEST_SUITE_TYPE;
	}

    /**
     * This implementation accepts generation for the given compilation unit provided that
     * its main type implements a JUnit test interface.
     */
	public boolean isSupported(ICompilationUnit cu) {
		try {
			IType type = cu.findPrimaryType();
			if (type != null) {
				ITypeHierarchy typeHier= type.newSupertypeHierarchy(null);
				IType[] superInterfaces= typeHier.getAllInterfaces();
				for (int i= 0; i < superInterfaces.length; i++) {
					if (superInterfaces[i].getFullyQualifiedName().equals(Helper.JUNIT_TEST_CLASS_NAME))
						return true;
				}
			}
			return false;
		} catch (JavaModelException e) {
			return false;
		}
	}

	/**
	 * Generates a basic model, with no model behavior, and one test case per test method.
	 */
	public ITestSuite generate(ICompilationUnit cu) throws JavaModelException {
		Resource resource = new ResourceImpl();
		ITestSuite testSuite = HyadesFactory.INSTANCE.createTestSuite(resource);
		testSuite.setType(getTestSuiteType());

		HyadesFactory.INSTANCE.createImplementor(testSuite, /*isExternalImplementor*/true);
		// This TestSuite has a code-implemented behavior:
		JUnitModelUpdater.setImplementorBlock(testSuite.getImplementor(), null);
		
		IType mainType = cu.findPrimaryType();
		if (mainType != null) {
			testSuite.setName(mainType.getElementName());
			populateTestCases(testSuite, mainType);
		}
		return testSuite;
	}

	/**
	 * Updates the Test Suite model in order to take the last changes from the code
	 * into account. Invoking this method updates the list of TestCases belonging to
	 * the specified TestSuite.<br>
	 * This method does the following:
	 * <ul>
	 *   <li>for each test method (ie "void testXxxx()") in the code, ensures that there is a TestCase
	 *   associated to it. If not, creates a TestCase with the same name as the test method.</li>
	 *   <li>for each TestCase, ensures that the method it is associated with still exists.
	 *   If not, deletes the TestCase.</li>
	 * </ul>
	 * @param testSuite
	 * @param cu
	 * @return <code>true</code> if any update was applied to the specified test suite,
	 * or <code>false</code> if the test suite was already up-to-date.
	 * @throws CoreException
	 */
	public boolean update(ITestSuite testSuite, ICompilationUnit cu) throws CoreException {
		boolean changed = false;
		if (cu == null) {
			throw new CoreException(makeErrorStatus("JUnit Compilation Unit has no primary type", null)); //$NON-NLS-1$
		}
		if (!cu.isStructureKnown()) {
			throw new CoreException(makeErrorStatus("JUnit Compilation Unit structure is unknown", null)); //$NON-NLS-1$
		}
		IType primaryType = cu.findPrimaryType();
		boolean isExternalBehavior = testSuite.getImplementor().isExternalImplementor();
		// Search for new test methods (=do not have a matching test case)
		IMethod[] methods = primaryType.getMethods();
		List usedTestCases = new ArrayList(methods.length);
		for (int i = 0; i < methods.length; i++) {
			if (JUnitGenerator.isTestMethod(methods[i])) {
				ITestCase testCase = findTestCase(testSuite, methods[i]);
				if (testCase == null) {
					String methodName = methods[i].getElementName();
					testCase = TestCommonUtil.createTestCase(testSuite, TestCommon.JUNIT_TEST_CASE_TYPE, isExternalBehavior, methodName);
					testCase.getImplementor().setResource(methodName);
					changed = true;
				}
				usedTestCases.add(testCase);
				if (updateTestCaseDescription(methods[i], testCase)) {
					changed = true;
				}
			}
		}
		// Search for Test Cases that do not have a matching test method
		List unusedTestCases = new ArrayList();
		Iterator it = testSuite.getITestCases().iterator();
		while (it.hasNext()) {
			ITestCase testCase = (ITestCase) it.next();
			if (!usedTestCases.contains(testCase)) {
				String methodName = Helper.getTestMethodName(testCase);
				if (methodName != null) {
					IMethod method = primaryType.getMethod(methodName, new String[0]);
					if (!method.exists()) {
						unusedTestCases.add(testCase);
					}
				}
			}
		}
		// Remove test cases that are not associated to a test method any more
		it = unusedTestCases.iterator();
		while (it.hasNext()) {
			EObject testCase = (EObject)it.next();
			EMFUtil.remove(testCase);
			changed = true;
		}
		return changed;
	}
	
	private static boolean updateTestCaseDescription(IMethod method, ITestCase testCase) throws JavaModelException {
		TagElement sourceTag = ASTHelper.getJavadocDescription(method);
		if (sourceTag == null) {
			if (testCase.getDescription() != null && testCase.getDescription().length() > 0) {
				testCase.setDescription(null);
				return true;
			}
			return false;
		}
		
		String sourceDescription = ASTHelper.extractDescription(sourceTag);
		if (sourceDescription.startsWith(testCase.getName())) {
			sourceDescription = sourceDescription.substring(testCase.getName().length());
			if (sourceDescription.startsWith("\n")) sourceDescription = sourceDescription.substring("\n".length());  //$NON-NLS-1$//$NON-NLS-2$
		}
		
		String tcDescription = testCase.getDescription();
		if (tcDescription == null) tcDescription = ""; //$NON-NLS-1$
		if (!Helper.compareJavaComments(sourceDescription, tcDescription)) {
			testCase.setDescription(sourceDescription.trim());
			return true;
		}
		return false;
	}

	/**
	 * Populates an empty JUnit ITestSuite with one ITestCase for each test method found
	 * in compilation unit cu.
	 * For the more generic case where the ITestSuite already owns ITestCase children which
	 * need to updated according to the new state of the source code, see 
	 * ModelUpdater.updateTestSuiteFromCode(ITestSuite).
	 * @param testSuite
	 * @param type
	 * @throws JavaModelException
	 * @see org.eclipse.hyades.test.tools.core.internal.java.modelsync.JUnitModelUpdater#updateTestSuiteFromCode(ITestSuite)
	 */
    private static void populateTestCases(ITestSuite testSuite, IType type) throws JavaModelException {
    	if (type != null) {
    		IMethod[] methods = type.getMethods();
    		//List testCases = testSuite.getITestCases();
    		for (int i = 0; i < methods.length; i++) {
    			if (JUnitGenerator.isTestMethod(methods[i])) {
					String methodName = methods[i].getElementName();
					ITestCase tc = TestCommonUtil.createTestCase(testSuite, TestCommon.JUNIT_TEST_CASE_TYPE, /*externalBehavior*/true, methodName);
					Helper.setTestMethodName(tc, methodName);
    			}
    		}
    	}
    }

    /**
     * Returns the test case matching a given test method, or <code>null</code> if there
     * is none.
     * @param testSuite
     * @param testMethod
     * @return
     */
	private static ITestCase findTestCase(ITestSuite testSuite, IMethod testMethod) {
		Iterator it = testSuite.getITestCases().iterator();
		Helper helper = new Helper();
		try {
			String testMethodName = testMethod.getElementName();
			while (it.hasNext()) {
				ITestCase testCase = (ITestCase) it.next();
				String methodName = Helper.getTestMethodName(testCase);
				if (testMethodName.equals(methodName)) {
					return testCase;
				}
			}
		} finally {
			helper.dispose();
		}
		return null;
	}
	
	private static IStatus makeErrorStatus(String message, Throwable exception) {
		return new Status(IStatus.ERROR,
				CorePlugin.PLUGIN_ID,
				1,
				message,
				exception);
	}

	public boolean isSupported(IJavaProject project) {
		return true;
	}
	
}
