/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ManifestUtil.java,v 1.4 2010/05/13 17:39:14 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.core.internal.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Properties;
import java.util.jar.Attributes;
import java.util.jar.JarFile;
import java.util.jar.Manifest;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.osgi.service.pluginconversion.PluginConversionException;
import org.eclipse.osgi.service.pluginconversion.PluginConverter;
import org.eclipse.osgi.util.ManifestElement;
import org.osgi.framework.BundleException;
import org.osgi.util.tracker.ServiceTracker;

/**
 * <p>Manifest file utilities.</p>
 * 
 * 
 * @author  Julien Canches
 * @author  Paul E. Slauenwhite
 * @version May 13, 2010
 * @since   April 26, 2007
 */
public class ManifestUtil {
	
	/**
	 * The name of the manifest file (e.g. /META-INF/MANIFEST.MF).
	 */
	public final static String MANIFEST_NAME = JarFile.MANIFEST_NAME;

	/**
	 * The symbolic name (ID) property of the manifest file (e.g. /META-INF/MANIFEST.MF).
	 */
	public final static String ID_PROPERTY = "Bundle-SymbolicName"; //$NON-NLS-1$		
	
	/**
	 * The version property of the manifest file (e.g. /META-INF/MANIFEST.MF).
	 */
	public final static String VERSION_PROPERTY = "Bundle-Version"; //$NON-NLS-1$		

	/**
	 * The fragment host plug-in property of the manifest file (e.g. /META-INF/MANIFEST.MF).
	 */
	public final static String HOST_PLUGIN_PROPERTY = "Fragment-Host"; //$NON-NLS-1$		

	/**
	 * The requires (dependencies) property of the manifest file (e.g. /META-INF/MANIFEST.MF).
	 */
	public final static String REQUIRES_PROPERTY = "Require-Bundle"; //$NON-NLS-1$		

	/**
	 * Resolves the manifest file (e.g. /META-INF/MANIFEST.MF) for the parameter 
	 * plug-in project. 
	 * <p>
	 * If the parameter project is not a manifest-based plug-in project, 
	 * <code>null</code> is returned.
	 * <p>
	 * 
	 * @param pluginProject Plug-in project to be evaluated.
	 * @return The manifest file (e.g. /META-INF/MANIFEST.MF) for the parameter plug-in project, otherwise <code>null</code>. 
	 */
	public static IFile getManifestFile(IProject pluginProject) {
		return (pluginProject.getFile(MANIFEST_NAME));
	}
	
	/**
	 * Resolves the string value of a named property in the parameter manifest 
	 * file (e.g. /META-INF/MANIFEST.MF). 
	 * <p>
	 * If the parameter file is not a manifest file or the property name is invalid, 
	 * <code>null</code> is returned.
	 * <p>
	 * Property names are defined as static constants in this class.
	 * <p>
	 * If the property contains multiple values, a comma-delimited string is returned.
	 * <p>
	 * 
	 * @param manifestFile Manifest file to be evaluated.
	 * @param propertyName Name of the property in the manifest file.
	 * @return The string value of a named property in the parameter manifest file, otherwise <code>null</code>. 
	 */
	public static String getPropertyValue(IFile manifestFile, String propertyName) {
		
		try {
			
			String[] propertyValues = getPropertyValues(manifestFile, propertyName);
			
			if(propertyValues != null){				

				if(propertyValues.length > 1){
					
					StringBuffer propertyValue = new StringBuffer();
					
					for (int counter = 0; counter < propertyValues.length; counter++) {
						
						if(propertyValue.length() > 0){
							propertyValue.append(", "); //$NON-NLS-1$
						}
						
						propertyValue.append(propertyValues[counter]);
					}
					
					return (propertyValue.toString());
				}
				
				return (propertyValues[0]);
			}
		} 
		catch (Exception e) {
			//Ignore and return null.
		}

		return null;
	}	

	/**
	 * Resolves an array containing the string values of a named property in the parameter manifest 
	 * file (e.g. /META-INF/MANIFEST.MF). 
	 * <p>
	 * If the parameter file is not a manifest file or the property name is invalid, 
	 * <code>null</code> is returned.
	 * <p>
	 * Property names are defined as static constants in this class.
	 * <p>
	 * If the property contains only one value, an array of size one is returned.
	 * <p>
	 * 
	 * @param manifestFile Manifest file to be evaluated.
	 * @param propertyName Name of the property in the manifest file.
	 * @return The array containing the string values of a named property in the parameter manifest file, otherwise <code>null</code>. 
	 */
	public static String[] getPropertyValues(IFile manifestFile, String propertyName) {
		
		InputStream manifestFileInputStream = null;
		
		try {
			
			manifestFileInputStream = manifestFile.getContents(true);
			
			String propertyValue = new Manifest(manifestFileInputStream).getMainAttributes().getValue(propertyName);
			
			if(propertyValue != null){
				
				propertyValue = propertyValue.trim();
				
				String[] propertyValues = propertyValue.split("\\s*,\\s*"); //$NON-NLS-1$		
				int colonIndex = -1;
					
				for (int counter = 0; counter < propertyValues.length; counter++) {
						
					colonIndex = propertyValues[counter].indexOf(';');
						
					if(colonIndex != -1){
						propertyValues[counter] = propertyValues[counter].substring(0, colonIndex).trim();
					}

					if((propertyName.equals(VERSION_PROPERTY)) && (propertyValues[counter].endsWith(".qualifier"))){ //$NON-NLS-1$		
						propertyValues[counter] = propertyValues[counter].substring(0, (propertyValues[counter].length() - 10));
					}
				}

				return propertyValues;
			}
		} 
		catch (Exception e) {
			//Ignore and return null.
		}
		finally{
			
			if(manifestFileInputStream != null){
				
				try {
					manifestFileInputStream.close();
				} 
				catch (IOException e) {
					//Ignore since the steam cannot be closed.
				}
			}
		}
		
		return null;
	}
	
	/**
	 * Copies a manifest file and add new attributes to one of its properties.
	 * @param inputFile The original manifest file.
	 * @param outputFile The generated manifest file with modifications included. This argument
	 * may be the same as inputFile (in that case inputFile is modified).
	 * @param property The manifest property to modify.
	 * @param attributes The list of attributes to add the property.
	 */
	public static void addAttributesToManifestProperty(File inputFile, File outputFile, String property, Collection attributes) {
		// The following code was conceived after
		// org.eclipse.pde.internal.core.PDEPluginConvert.modifyManifest()
		Collection attributesToAdd = new ArrayList(attributes);
		InputStream manifestStream = null;
		try {
			manifestStream = new FileInputStream(inputFile);
			Manifest manifest = new Manifest(manifestStream);
			Properties prop = manifestToProperties(manifest.getMainAttributes());
			String requires = prop.getProperty(property);
			StringBuffer buffer = new StringBuffer();
			if (requires != null) {
				ManifestElement[] elements = ManifestElement.parseHeader(property, requires);
				for (int i = 0; i < elements.length; i++) {
					buffer.append(elements[i].getValue());
					buffer.append(","); //$NON-NLS-1$
					buffer.append(System.getProperty("line.separator")); //$NON-NLS-1$
					buffer.append(" "); //$NON-NLS-1$
					attributesToAdd.remove(elements[i].getValue());
				}
			}
			for (Iterator it = attributesToAdd.iterator(); it.hasNext();) {
				String pluginId = (String) it.next();
				buffer.append(pluginId);
				if (it.hasNext()) {
					buffer.append(","); //$NON-NLS-1$
					buffer.append(System.getProperty("line.separator")); //$NON-NLS-1$
					buffer.append(" "); //$NON-NLS-1$
				}
			}
			prop.put(property, buffer.toString());
			
			if (!attributesToAdd.isEmpty() || !inputFile.equals(outputFile)) {
				ServiceTracker tracker = new ServiceTracker(CorePlugin.getInstance().getContext(),
						PluginConverter.class.getName(), null);
				tracker.open();
				PluginConverter converter = (PluginConverter) tracker.getService();
				converter.writeManifest(outputFile, prop, false);
				tracker.close();
			}
		} catch (FileNotFoundException e) {
		} catch (IOException e) {
		} catch (BundleException e) {
		} catch (PluginConversionException e) {
		} finally {
			try {
				if (manifestStream != null)
					manifestStream.close();
			} catch (IOException e) {
			}
		}
	}

	private static Properties manifestToProperties(Attributes d) {
		Iterator iter = d.keySet().iterator();
		Properties result = new Properties();
		while (iter.hasNext()) {
			Attributes.Name key = (Attributes.Name) iter.next();
			result.put(key.toString(), d.get(key));
		}
		return result;
	}	
}
