/*******************************************************************************
 * Copyright (c) 2007, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: PluginProjectUtil.java,v 1.7 2010/05/13 17:39:14 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.core.internal.util;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.osgi.framework.Bundle;

/**
 * <p>Plug-in project utilities.</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @version May 13, 2010
 * @since   April 26, 2007
 */
public class PluginProjectUtil {

	/**
	 * Evaluates the parameter project to determine if it is a 
	 * strict plug-in project. 
	 * <p>
	 * Fragment projects are not considered strict plug-in projects. 
	 * <p>
	 * 
	 * @param project Project to be evaluated.
	 * @return True if the parameter plug-in project is a strict plug-in project, otherwise false.
	 */
	public static boolean isStrictPluginProject(IProject project) {

		if(isPluginXMLBasedPluginProject(project)){
			return true; 
		}
		else if ((!isFragmentPluginProject(project)) && (isManifestBasedPluginProject(project))){

			//Inspect the MANIFEST.MF file:
			return (ManifestUtil.getPropertyValue(ManifestUtil.getManifestFile(project), ManifestUtil.HOST_PLUGIN_PROPERTY) == null); 
		}
		
		return false;
	}
	
	/**
	 * Evaluates the parameter project to determine if it is a 
	 * plug-in fragment project. 
	 * <p>
	 * A plug-in fragment project contains an XML fragment file (e.g. /fragment.xml).
	 * <p>
	 * 
	 * @param project Project to be evaluated.
	 * @return True if the parameter plug-in project is a plug-in XML-based plug-in project, otherwise false.
	 */
	public static boolean isFragmentPluginProject(IProject project) {
		
		IFile fragmentXmlFile = project.getFile("fragment.xml"); //$NON-NLS-1$
	
		return ((fragmentXmlFile != null) && (fragmentXmlFile.exists()));
	}
	
	/**
	 * Evaluates the parameter project to determine if it is a 
	 * plug-in XML-based plug-in project. 
	 * <p>
	 * A plug-in XML-based plug-in project contains an XML plug-in file (e.g. /plugin.xml).
	 * <p>
	 * 
	 * @param project Project to be evaluated.
	 * @return True if the parameter plug-in project is a plug-in XML-based plug-in project, otherwise false.
	 */
	public static boolean isPluginXMLBasedPluginProject(IProject project) {
		
		IFile pluginXmlFile = PluginXmlUtil.getPluginXmlFile(project);
	
		return ((pluginXmlFile != null) && (pluginXmlFile.exists()));
	}
	
	/**
	 * Evaluates the parameter project to determine if it is a 
	 * manifest-based plug-in project. 
	 * <p>
	 * A manifest-based plug-in project contains a manifest file (e.g. /META-INF/MANIFEST.MF).
	 * <p>
	 * 
	 * @param project Project to be evaluated.
	 * @return True if the parameter plug-in project is a manifest-based plug-in project, otherwise false.
	 */
	public static boolean isManifestBasedPluginProject(IProject project) {
		
		IFile manifestFile = ManifestUtil.getManifestFile(project);
		
		return ((manifestFile != null) && (manifestFile.exists()));
	}
	
	
	/**
	 * Determines if the parameter plug-in project requires the parameter plug-in ID. 
	 * <p>
	 * If the parameter project is not a valid plug-in project, <code>false</code> is returned.
	 * <p>
	 * 
	 * @param pluginProject Plug-in project to be evaluated.
	 * @param pluginId The ID of the dependent plug-in to be compared to the parameter plug-in project.
	 * @return True if the parameter plug-in project requires the parameter plug-in ID, otherwise false. 
	 */
	public static boolean containsPluginRequires(IProject pluginProject, String pluginId) {
		
		String propertyValue = null;
		
		if(isManifestBasedPluginProject(pluginProject)){			
			propertyValue = ManifestUtil.getPropertyValue(ManifestUtil.getManifestFile(pluginProject), ManifestUtil.REQUIRES_PROPERTY);			
		}
		else if(isPluginXMLBasedPluginProject(pluginProject)){
			propertyValue = PluginXmlUtil.getPropertyValue(PluginXmlUtil.getPluginXmlFile(pluginProject), PluginXmlUtil.REQUIRES_PROPERTY);
		}

		if(propertyValue != null){
			return (propertyValue.indexOf(pluginId.trim()) != -1);
		}

		return false;
	}	
	
	/**
	 * Resolves a non-null list ({@link IProject}) of dependent project for the parameter plug-in 
	 * project. 
	 * <p>
	 * 
	 * @param pluginProject Plug-in project to be evaluated.
	 * @return Non-null list ({@link IProject}) of dependent project for the parameter plug-in project.
	 */
	public static List getDependentProjects(IProject pluginProject) {
		
		try {
			
			IProject[] projects = pluginProject.getDescription().getDynamicReferences();
			List result = new ArrayList(projects.length);
			
			for (int counter = 0; counter < projects.length; counter++) {
				
				if (isStrictPluginProject(projects[counter])) {
					result.add(projects[counter]);
				}
			}
			
			return result;
		} 
		catch (CoreException e) {
			CorePlugin.logError(e);			
		}
		
		return Collections.EMPTY_LIST;
	}
	
	/**
	 * Resolves the symbolic name (ID) of the bundle for the parameter plug-in 
	 * project. 
	 * <p>
	 * If the parameter project is not a valid plug-in project, <code>null</code> is returned.
	 * <p>
	 * 
	 * @param pluginProject Plug-in project to be evaluated.
	 * @return The symbolic name (ID) of the bundle for the parameter plug-in project, otherwise <code>null</code>. 
	 */
	public static String getBundleSymbolicName(IProject pluginProject) {
		
		if(isManifestBasedPluginProject(pluginProject)){
			return (ManifestUtil.getPropertyValue(ManifestUtil.getManifestFile(pluginProject), ManifestUtil.ID_PROPERTY)); 			
		}
		else if(isPluginXMLBasedPluginProject(pluginProject)){
			return (PluginXmlUtil.getPropertyValue(PluginXmlUtil.getPluginXmlFile(pluginProject), PluginXmlUtil.ID_PROPERTY)); 			
		}

		return null;
	}	
	
	/**
	 * Resolves the version of the bundle for the parameter plug-in 
	 * project. 
	 * <p>
	 * If the parameter project is not a valid plug-in project, <code>null</code> is returned.
	 * <p>
	 * 
	 * @param pluginProject Plug-in project to be evaluated.
	 * @return The version of the bundle for the parameter plug-in project, otherwise <code>null</code>. 
	 */
	public static String getBundleVersion(IProject pluginProject) {
	
		if(isManifestBasedPluginProject(pluginProject)){
			return (ManifestUtil.getPropertyValue(ManifestUtil.getManifestFile(pluginProject), ManifestUtil.VERSION_PROPERTY)); 	
		}
		else if(isPluginXMLBasedPluginProject(pluginProject)){
			return (PluginXmlUtil.getPropertyValue(PluginXmlUtil.getPluginXmlFile(pluginProject), PluginXmlUtil.VERSION_PROPERTY));			
		}
		
		return null;
	}	
	
	/**
	 * Resolves the version of the bundle of the parameter symbolic name. 
	 * <p>
	 * If the parameter symbolic name is not a valid bundle, <code>null</code> is returned.
	 * <p>
	 * 
	 * @param symbolicName The symbolic name of the bundle to be evaluated.
	 * @return The version of the bundle of the parameter symbolic name, otherwise <code>null</code>. 
	 */
	public static String getBundleVersion(String symbolicName) {
		
		String version = null;
		
		Bundle bundle = Platform.getBundle(symbolicName);

		if (bundle != null) {
			
			version = ((String) (bundle.getHeaders().get(ManifestUtil.VERSION_PROPERTY))); 

			if(version != null){
				
				version = version.trim();
				
				if(version.endsWith(".qualifier")){ //$NON-NLS-1$		
					version = (version.substring(0, (version.length() - 10)));
				}
			}
		}		
		
		return version;
	}	

	/**
	 * Resolves a non-null array ({@link String}) of project-relative deployable directories and 
	 * files for the parameter plug-in project. 
	 * <p>
	 * Deployable directories and files are identified in the build properties file (e.g. build.properties) for the parameter 
	 * plug-in project.  That is, deployable directories and files would be exported to an external plug-in
	 * by the Plug-in Export Wizard.
	 * <p>
	 * Directories are denoted by an appended backslash.
	 * <p>
	 * 
	 * @param pluginProject Plug-in project to be evaluated.
	 * @return Non-null array ({@link String}) of project-relative deployable directories and files for the parameter plug-in project.
	 */
	public static String[] getDeployables(IProject pluginProject) {
		
		IFile buildFile = pluginProject.getFile("build.properties"); //$NON-NLS-1$
		
		if (buildFile.exists()) {
			
			InputStream buildFileInputStream = null;
			
			try {
				
				buildFileInputStream = buildFile.getContents(true);
				
				Properties properties = new Properties();
				properties.load(buildFileInputStream);
				
				String binIncludes = properties.getProperty("bin.includes"); //$NON-NLS-1$
				
				if(binIncludes != null){
					return (binIncludes.split("\\s*,\\s*")); 
				}
			} 
			catch (Exception e) {
				CorePlugin.logError(e);
			} 
			finally{
				
				if(buildFileInputStream != null){
					
					try {
						buildFileInputStream.close();
					} 
					catch (IOException e) {
						//Ignore since the steam cannot be closed.
					}
				}
			}
		}
		
		return (new String[0]);
	}
}
