/*******************************************************************************
 * Copyright (c) 2007, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: PluginXmlUtil.java,v 1.4 2010/05/13 17:39:14 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.core.internal.util;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.xsd.util.XSDParser;
import org.eclipse.xsd.util.XSDResourceImpl;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * <p>Plug-in XML file utilities.</p>
 * 
 * 
 * @author  Julien Canches
 * @author  Paul E. Slauenwhite
 * @version May 13, 2010
 * @since   April 26, 2007
 */
public class PluginXmlUtil {
	
	/**
	 * The name of the plug-in XML file (e.g. plugin.xml).
	 */
	public final static String PLUGIN_XML_NAME = "plugin.xml"; //$NON-NLS-1$

	/**
	 * The symbolic name (ID) property of the plug-in XML file (e.g. plugin.xml).
	 */
	public final static String ID_PROPERTY = "id"; //$NON-NLS-1$		
	
	/**
	 * The version property of the plug-in XML file (e.g. plugin.xml).
	 */
	public final static String VERSION_PROPERTY = "version"; //$NON-NLS-1$		

	/**
	 * The fragment host plug-in property of the plug-in XML file (e.g. plugin.xml).
	 */
	public final static String HOST_PLUGIN_PROPERTY = "plugin-id"; //$NON-NLS-1$	
	
	/**
	 * The requires (dependencies) property of the plug-in XML file (e.g. plugin.xml).
	 */
	public final static String REQUIRES_PROPERTY = "requires"; //$NON-NLS-1$		
	
	/**
	 * Resolves the plug-in XML file (e.g. plugin.xml) for the parameter 
	 * plug-in project. 
	 * <p>
	 * If the parameter project is not a plug-in XML-based plug-in project, <code>null</code> is returned.
	 * <p>
	 * 
	 * @param pluginProject Plug-in project to be evaluated.
	 * @return The plug-in XML file (e.g. plugin.xml) for the parameter plug-in project, 
	 * otherwise <code>null</code>. 
	 */
	public static IFile getPluginXmlFile(IProject pluginProject) {
		return pluginProject.getFile(PLUGIN_XML_NAME);
	}

	/**
	 * Resolves the string value of a named property in the parameter plug-in 
	 * XML file (e.g. plugin.xml). 
	 * <p>
	 * If the parameter file is not a plug-in XML file or the property name is invalid, 
	 * <code>null</code> is returned.
	 * <p>
	 * Property names are defined as static constants in this class.
	 * <p>
	 * If the property contains multiple values, a comma-delimited string is returned.
	 * <p>
	 * 
	 * @param pluginXMLFile Plug-in XML file to be evaluated.
	 * @param propertyName Name of the property  in the plug-in XML file.
	 * @return The string value of a named property in the parameter plug-in XML file, otherwise <code>null</code>. 
	 */
	public static String getPropertyValue(IFile pluginXMLFile, String propertyName){
		
		try {
			
			Document document = getDocument(pluginXMLFile);
			
			if(document != null){				
					
				
				if(propertyName.equals(REQUIRES_PROPERTY)){
					
					Element requiresElement = ((Element)(document.getDocumentElement().getElementsByTagName(propertyName).item(0))); 
					
					if(requiresElement != null){
						
						NodeList importElements = requiresElement.getElementsByTagName("import"); //$NON-NLS-1$
						
						if(importElements.getLength() > 0){
						
							StringBuffer propertyValues = new StringBuffer();
							
							for(int counter = 0; counter < importElements.getLength(); counter++) {
								
								Node importElement = importElements.item(counter);
								
								if(importElement instanceof Element) {
									
									if(propertyValues.length() > 0){
										propertyValues.append(", "); //$NON-NLS-1$
									}
									
									propertyValues.append(((Element)(importElement)).getAttribute("plugin")); //$NON-NLS-1$
								}
							}
							
							return (propertyValues.toString());
						}
					}
				}
				else{
				
					String propertyValue = document.getDocumentElement().getAttribute(propertyName);
						
					if(propertyValue != null){
							
						propertyValue = propertyValue.trim();
							
						if((propertyName.equals(VERSION_PROPERTY)) && (propertyValue.endsWith(".qualifier"))){ //$NON-NLS-1$		
							propertyValue = (propertyValue.substring(0, (propertyValue.length() - 10)));
						}
							
						return propertyValue;
					}
				}
			}
		} 
		catch (Exception e) {
			//Ignore and return null.
		}

		return null;
	}	
	
	/**
	 * Resolves an array containing the string values of a named property in the parameter plug-in 
	 * XML file (e.g. plugin.xml). 
	 * <p>
	 * If the parameter file is not a plug-in XML file or the property name is invalid, 
	 * <code>null</code> is returned.
	 * <p>
	 * Property names are defined as static constants in this class.
	 * <p>
	 * If the property contains only one value, an array of size one is returned.
	 * <p>
	 * 
	 * @param pluginXMLFile Plug-in XML file to be evaluated.
	 * @param propertyName Name of the property in the plug-in XML file.
	 * @return The array containing the string values of a named property in the parameter plug-in XML file, otherwise <code>null</code>. 
	 */
	public static String[] getPropertyValues(IFile pluginXMLFile, String propertyName) {
		
		String propertyValue = getPropertyValue(pluginXMLFile, propertyName);
		
		if(propertyValue != null){
			return (propertyValue.split("\\s*,\\s*"));
		}
		
		return null;
	}
	
	/**
	 * Adds the given list of libraries to a copy of the plugin.xml of the given project.
	 * This temporary file is returned.
	 * @param project
	 * @param dirs a list of libraries that need to be added to the plugin.xml.
	 * @return the new tmp file created
	 */
	public static File addOutputDirectories(IProject project, Collection dirs) {
		IFile pluginFile = getPluginXmlFile(project);
		Document doc = getDocument(pluginFile);
		if (doc != null) {
			Element runtimeElt = getRuntimeElement(doc);
			for (Iterator it = dirs.iterator(); it.hasNext();) {
				Element libElt = doc.createElement("library"); //$NON-NLS-1$
				libElt.setAttribute("name", (String) it.next()); //$NON-NLS-1$
				Element exportElt = doc.createElement("export"); //$NON-NLS-1$
				exportElt.setAttribute("name", "*"); //$NON-NLS-1$//$NON-NLS-2$
				libElt.appendChild(exportElt);
				runtimeElt.appendChild(libElt);
			}
			return makeTmpPluginXmlFile(doc);
		}
		return null;
	}

	/**
	 * Updates the plugin.xml file of the given project by adding plugins that are 
	 * not yet in the requires section from the given plugins list
	 * @param project
	 * @param requiredPlugins the plugin ids list that the project should depend to.
	 */
	public static void addMissingRequiredPlugins(IProject project, Collection requiredPlugins) {
		Collection currentRequireds = new LinkedList();
		IFile pluginFile = getPluginXmlFile(project);
		boolean hasChanged = false;
		Document doc = getDocument(pluginFile);
		if(doc != null) {
			Element requires = getRequiresElement(doc);
			NodeList imports = requires.getElementsByTagName("import"); //$NON-NLS-1$
			for(int i=0; i < imports.getLength(); i++) {
				Node node = imports.item(i);
				if(node instanceof Element) {
					currentRequireds.add(((Element)node).getAttribute("plugin")); //$NON-NLS-1$
				}
			}
			for (Iterator it = requiredPlugins.iterator(); it.hasNext();) {
				String req = (String)it.next();
				if(!currentRequireds.contains(req)) {
					Element importElt = doc.createElement("import"); //$NON-NLS-1$
					importElt.setAttribute("plugin", (String) req); //$NON-NLS-1$
					requires.appendChild(importElt);
					hasChanged = true;
				}
			}
			if (hasChanged) {
				//- save the modified plugin.xml
				ByteArrayOutputStream bos = new ByteArrayOutputStream(4096);
				XSDResourceImpl.serialize(bos, doc, "UTF-8"); //$NON-NLS-1$
				ByteArrayInputStream bis = new ByteArrayInputStream(bos.toByteArray());
				try {
					pluginFile.setContents(bis, true, false, new NullProgressMonitor());
					bos.close();
					bis.close();
				} catch (Exception e) {
					CorePlugin.logError(e);
				}
			}
		}
	}

	private static Document getDocument(IFile pluginFile) {
		
		InputStream pluginFileInputStream = null;
		
		try {
			
			pluginFileInputStream = pluginFile.getContents(true); //force synchronise with real xml file

			XSDParser parser = new XSDParser(null);
			parser.parse(pluginFileInputStream);
			
			return (parser.getDocument());
		} 
		catch (CoreException e) {
			CorePlugin.logError(e);
		}
		finally{

			if(pluginFileInputStream != null){

				try {
					pluginFileInputStream.close();
				} 
				catch (IOException e) {
					//Ignore since the steam cannot be closed.
				}
			}
		}
		
		return null;
	}

	private static File makeTmpPluginXmlFile(Document doc) {
		try {
			File tmpFile = File.createTempFile("tptp.", ".plugin.xml"); //$NON-NLS-1$ //$NON-NLS-2$

			//Request for the file to be deleted when the JVM exits:
			tmpFile.deleteOnExit();
			
			BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(tmpFile));
			XSDResourceImpl.serialize(bos, doc, "UTF-8"); //$NON-NLS-1$
			bos.close();
			return tmpFile;
		} catch (IOException e) {
			CorePlugin.logError(e);
			return null;
		} 
	}

	private static Element getRuntimeElement(Document doc) {
		Element pluginElt = doc.getDocumentElement();
		Element runtimeElt = (Element) pluginElt.getElementsByTagName("runtime").item(0); //$NON-NLS-1$
		if(runtimeElt == null) {
			//- not yet 'runtime' element in the plugin.xml file.
			runtimeElt = doc.createElement("runtime"); //$NON-NLS-1$
			pluginElt.appendChild(runtimeElt);
		}
		return runtimeElt;
	}

	private static Element getRequiresElement(Document doc) {
		Element pluginElt = doc.getDocumentElement();
		Element requiresElt = (Element) pluginElt.getElementsByTagName("requires").item(0); //$NON-NLS-1$
		if(requiresElt == null) {
			//- not yet 'requires' element in the plugin.xml file.
			requiresElt = doc.createElement("requires"); //$NON-NLS-1$
			pluginElt.appendChild(requiresElt);
		}
		return requiresElt;
	}
}
