/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JUnitTestSuiteFacade.java,v 1.22 2010/04/12 12:38:53 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.tools.core.java;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceImpl;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.hyades.test.tools.core.common.TestCommon;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.JUnitExtensionsRegistry;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.JUnitModelUpdater;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.JUnitProjectBuilder;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.JUnitResourceFactory;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.osgi.util.NLS;

/**
 * <p>Provides high-level methods for navigating between JUnit source files (as JDT model
 * elements) and TPTP concepts (as TPTP Test Model elements).</p>
 * 
 * 
 * @author      Julien Canches
 * @author      Paul E. Slauenwhite
 * @author		Jerome Bozier
 * @version     November 17, 2009
 * @since       April 12, 2010
 * @provisional As of TPTP V4.1.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
 */
public class JUnitTestSuiteFacade {

	/**
	 * <p>Creates a Test Suite resource of type Junit that is using the specified compilation
	 * unit as its implementing behavior.</p>
	 * This is a convenience method. Invoking this method is fully equivalent to
	 * <code>createTestSuite(junitTestCompilationUnit,TestCommon.JUNIT_TEST_SUITE_TYPE,force)</code>.
	 * @param junitTestCompilationUnit A compilation unit that contains a Junit Test class
	 * as a primary type.
	 * @param force Whether any existing resource should be overwritten.
	 * @throws CoreException If a resource problem occurs, or if the resource already exists
	 * and <code>force</code> is <code>false</code>.
	 * @return The new TestSuite. 
	 */
    public static ITestSuite createTestSuite(ICompilationUnit junitTestCompilationUnit, boolean force) throws CoreException {
    	return createTestSuite(junitTestCompilationUnit, TestCommon.JUNIT_TEST_SUITE_TYPE, force);
    }
    
	/**
	 * Creates a Test Suite resource of the specified type and associates the specified compilation
	 * unit as its implementing behavior.
	 * @param junitTestCompilationUnit A compilation unit that contains a Junit Test class
	 * as a primary type.
	 * @param type The type of the returned Test Suite.
	 * @param force Whether any existing resource should be overwritten.
	 * @throws CoreException If a resource problem occurs, or if the resource already exists
	 * and <code>force</code> is <code>false</code>.
	 * @return The new TestSuite. 
	 */
    public static ITestSuite createTestSuite(ICompilationUnit junitTestCompilationUnit, String type, boolean force) throws CoreException {
   		String name = junitTestCompilationUnit.getElementName();
		name = name.substring(0, name.indexOf('.'));
		IFile javaFile = (IFile)junitTestCompilationUnit.getCorrespondingResource();
		IFile testSuiteFile = javaFile
				.getParent()
				.getFile(new Path(name + "." + ICommonConstants.TEST_SUITE_FILE_EXTENSION)); //$NON-NLS-1$

		ITestSuite testSuite = createTestSuite(junitTestCompilationUnit, type);
		testSuite.setName(name);
		saveTestSuite(testSuite, testSuiteFile, force);
		return testSuite;
    }
    
    /**
     * Creates a test suite of a given type for a given compilation unit. The resulting test suite
     * is volatile and should be saved if necessary to disk by invoking
     * {@link #saveTestSuite(ITestSuite, IFile, boolean)}.
     * @param cu A compilation unit that contains a JUnit class.
     * @param type The type of the generated test suite.
     * @return A test suite. This test suite holds a reference to the source file.
     * @throws CoreException If no test suite factory is registered for the given type, or if
     * a the registered test suite factory does not support generation for the specified compilation
     * unit, or if the factory failed to generate the model.
     * @since 4.3
     */
    public static ITestSuite createTestSuite(ICompilationUnit cu, String type) throws CoreException {
    	IJUnitTestSuiteFactory factory = JUnitExtensionsRegistry.getInstance().getFactory(type);
    	if (factory == null) {
    		throw new CoreException(makeErrorStatus(NLS.bind("No JUnit test suite factory registered for the type {0}", type), null)); //$NON-NLS-1$
    	}
		ITestSuite testSuite = factory.generate(cu);
		fixupResource(testSuite, cu);
		// Store in the Test Suite the reference to the JUnit code:
		testSuite.getImplementor().setLocation(getPackageFragmentRoot(cu).getCorrespondingResource().getFullPath().toString());
		testSuite.getImplementor().setResource(cu.findPrimaryType().getFullyQualifiedName());
		return testSuite;
    }
    
    /**
     * Ensures that a test suite is contained in a resource with a valid URI pointing
     * to a given resource. Ensuring this allows to have in-memory test suites (ie not
     * persisted) compatible with the expectations that are made at various points in TPTP
     * for ages.
     * @param testSuite
     * @param resource
     */
    private static void fixupResource(ITestSuite testSuite, ICompilationUnit cu) {
		if (!(testSuite instanceof EObject)) return;
		EObject eTestSuite = (EObject)testSuite;
		Resource res = eTestSuite.eResource();
		if (res == null) {
			res = new ResourceImpl();
			res.getContents().add(eTestSuite);
		}
		URI uri = JUnitResourceFactory.getURI(cu, testSuite.getType());
		res.setURI(uri);
	}

	/**
     * Saves a test suite generated with {@link #createTestSuite(ICompilationUnit, String)}.
     * This save operation also registers the association between the model and the source,
     * so they can be kept in synchronization.
     * @param testSuite A test suite model that has a reference to its associated source file.
     * @param file The file that the test model will be persisted to.
     * @param force Whether <code>file</code> should be overwritten, if it already exists.
     * @throws CoreException
     * @since 4.3
     */
    public static void saveTestSuite(final ITestSuite testSuite, final IFile file, boolean force) throws CoreException {
		if (file.exists()) {
			if (force) {
				file.delete(/*force*/true, /*keepHistory*/true, null);
			} else {
				throw new CoreException(makeErrorStatus(NLS.bind("Resource {0} already exists.", file.getFullPath()), null)); //$NON-NLS-1$
			}
		}
		ResourceSet resourceSet = new ResourceSetImpl();
		URI uri = URI.createPlatformResourceURI(file.getFullPath().toString(), false);
		final Resource resource = resourceSet.createResource(uri);
		resource.getContents().add((EObject) testSuite);

		ResourcesPlugin.getWorkspace().run(new IWorkspaceRunnable() {
			public void run(IProgressMonitor monitor) throws CoreException {
				try {
					EMFUtil.save(resource);
				} catch (Exception e) {
					throw new CoreException(makeErrorStatus("Problem in saving resource " + file.getFullPath(), e)); //$NON-NLS-1$
				}
				JUnitProjectBuilder.installBuilder(file.getProject());
				IFile javaFile = getJUnitSourceFile(testSuite);
				JUnitProjectBuilder.installBuilder(javaFile.getProject());
				JUnitModelUpdater.associateTestSuiteToJUnitSourceFile(javaFile, file, testSuite);
			}
		}, new NullProgressMonitor());
    }

	/**
	 * Returns the JUnit source file associated to a JUnit TestSuite.
	 * @param testSuite
	 * @return The source file associated with the Test Suite, or <code>null</code> if this file
	 * is not specified or is not accessible.
	 */
	public static IFile getJUnitSourceFile(ITestSuite testSuite) {
		if (testSuite.getImplementor() == null) return null;
        String location = testSuite.getImplementor().getLocation();
        String qualifiedName = testSuite.getImplementor().getResource();
        if (location == null || qualifiedName == null) return null;
        IResource sourceFolder = ResourcesPlugin.getWorkspace().getRoot().findMember(new Path(location));
		if (sourceFolder != null && sourceFolder instanceof IContainer) {
			IPath compilationUnitPath = new Path(qualifiedName.replace('.', '/')+".java"); //$NON-NLS-1$
			IResource compilationUnitResource = ((IContainer)sourceFolder).findMember(compilationUnitPath);
			if (compilationUnitResource != null && compilationUnitResource.getType() == IResource.FILE) {
				return (IFile)compilationUnitResource;
			}
		}
		return null;
	}
	
	private static URI getResourceURI(ITestSuite testSuite) {
		if (testSuite instanceof EObject) {
			Resource res = ((EObject)testSuite).eResource();
			if (res != null) {
				return res.getURI();
			}
		}
		return null;
	}
	
	/**
	 * Returns the model resource that contains the test suite, if the test suite has been
	 * added to a model resource.
	 * @param testSuite A test suite created by this facade.
	 * @return the model resource that contains the test suite, or <code>null</code>if the test
	 * suite has been never saved or manually added to a resource. 
	 */
	public static IFile getResourceFile(ITestSuite testSuite) {
		URI uri = getResourceURI(testSuite);
		if (uri != null && !JUnitResourceFactory.PROTOCOL.equals(uri.scheme())) {
			return EMFUtil.getWorkspaceFile(uri);
		}
		return null;
	}
	
	/**
	 * Returns the underlying file from which the test suite was loaded. This is either a
	 * model resource (.testsuite) if the test suite was loaded from a model resource,
	 * or a java file, if the test suite was directly generated from a java file.
	 * @param testSuite
	 * @return The underlying resource, or <code>null</code> if not available.
	 */
	public static IFile getUnderlyingFile(ITestSuite testSuite) {
		URI uri = getResourceURI(testSuite);
		if (uri != null && !JUnitResourceFactory.PROTOCOL.equals(uri.scheme())) {
			return EMFUtil.getWorkspaceFile(uri);
		} else {
			IPath path = new Path(uri.path());
			return ResourcesPlugin.getWorkspace().getRoot().getFile(path);
		}
	}
	
    private static IPackageFragmentRoot getPackageFragmentRoot(IJavaElement javaElement) {
    	if (javaElement instanceof IPackageFragmentRoot) {
    		return (IPackageFragmentRoot)javaElement;
    	} else {
    		IJavaElement parent = javaElement.getParent();
    		if (parent != null) {
    			return getPackageFragmentRoot(parent);
    		} else {
    			return null;
    		}
    	}
    }
    
	/**
	 * Finds the TPTP JUnit TestSuite model associated to the JUnit source file.
	 * @param junitSourceFile A Java file defining a JUnit test.
	 * @param resourceSet The ResourceSet where the returned value will be loaded in.
	 * @return A valid JUnit TestSuite, or <code>null</code> if no TPTP Test is associated
	 * to the JUnit source code.
	 * @throws JavaModelException If the specified junitSourceFile could not be read
	 * or parsed.
	 */
	public static ITestSuite findTestSuite(IFile junitSourceFile, ResourceSet resourceSet) throws JavaModelException {
		ICompilationUnit cu = JavaCore.createCompilationUnitFrom(junitSourceFile);
		if (cu.isStructureKnown()) {
			IType type = cu.findPrimaryType();
			if (type != null) {
				return JUnitModelUpdater.findTestSuite(type, resourceSet);
			}
		}
		return null;
	}

	private static IStatus makeErrorStatus(String message, Throwable exception) {
		return new Status(IStatus.ERROR,
				CorePlugin.PLUGIN_ID,
				1,
				message,
				exception);
	}

}
