/*******************************************************************************
 * Copyright (c) 2008, 2009 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Stanislav Polevic, Intel - Initial API and Implementation 
 * 	  
 * $Id: BFReader.java,v 1.9 2009/08/28 14:01:00 jwest Exp $ 
 *******************************************************************************/
package org.eclipse.hyades.loaders.internal.binary;

import java.io.UnsupportedEncodingException;

public class BFReader implements NumberReader, StringReader {


	public static final StringReader ASCII_READER = new StringReader() {
			private static final char EOS = '\0';
			
			private static final String DEFAULT_ENCODING = "UTF-8";

			private String encoding = DEFAULT_ENCODING;
			
			public void setEncoding(String encoding) {
				this.encoding = encoding;
			}

			public String readString(byte[] b, Offset offset) {
				int start = offset.getOffset();
				int end = -1;
				for(int i = start; i < b.length; ++i) {
					char c = (char) b[i];
					if (c == EOS) {
						end = i;
						break;
					}
				}
				
				if (end >= 0) {
					int stringOffset = (end - start) + 1;
					if (offset.increaseOffset(stringOffset) < 0) {
						return null;
					}
					
					try {
						return new String(b, start, stringOffset - 1, encoding);
					} catch (UnsupportedEncodingException e) {
						setEncoding(DEFAULT_ENCODING);
						return  new String(b, start, stringOffset - 1);
					}
				}
				return null;
			}

			public String[] readStringArray(byte[] b, int length, Offset offset) {
				String[] res = new String[length];
				for (int i = 0; i < length; ++i) {
					res[i] = readString(b, offset);
				}
				return res;
			}

		};
	
		public static final NumberReader BE_64_READER = new NumberReader() { // Big-endian, 64

			private long cpuFrequency;
			
			public short readShort(byte[] b, Offset offset) {
				int start = offset.getOffset();
		        if (offset.increaseOffset(2) < 0) {
		        	return 0;
		        }
		        int ch1 = b[start];
		        int ch2 = b[start + 1];
		        
		        return (short)((ch2 << 0) + (ch1 << 8));			
			} 
			
			public int readUnsignedShort(byte[] b, Offset offset) {
				int start = offset.getOffset();
		        if (offset.increaseOffset(2) < 0) {
		        	return 0;
		        }
		        int ch1 = b[start] & 0xFF;
		        int ch2 = b[start + 1] & 0xFF;
		        
		        return ((ch2 << 0) + (ch1 << 8));			
			}

			public int readInt(byte[] b, Offset offset) {
				int start = offset.getOffset();
				if (offset.increaseOffset(4) < 0) {
					return 0;
				}
		        int ch1 = b[start];
		        int ch2 = b[start + 1];
		        int ch3 = b[start + 2];
		        int ch4 = b[start + 3];
		        
		        return ((ch4 << 0) + (ch3 << 8) + (ch2 << 16) + (ch1 << 24));
			}
		
			public long readUnsignedInt(byte[] b, Offset offset) {
				int start = offset.getOffset();
				if (offset.increaseOffset(4) < 0) {
					return 0;
				}
		        int ch1 = b[start] & 0xFF;
		        int ch2 = b[start + 1] & 0xFF;
		        int ch3 = b[start + 2] & 0xFF;
		        int ch4 = b[start + 3] & 0xFF;
		        
		        return ((ch4 << 0) + (ch3 << 8) + (ch2 << 16) + (ch1 << 24));
			}

			public long readLong(byte[] b, Offset offset) {
				int start = offset.getOffset();
				if (offset.increaseOffset(8) < 0) {
					return 0;
				}
		        int ch1 = b[start];  // Most Significant Byte (MSB)
		        int ch2 = b[start + 1];
		        int ch3 = b[start + 2];
		        int ch4 = b[start + 3];
		        int ch5 = b[start + 4];
		        int ch6 = b[start + 5];
		        int ch7 = b[start + 6];
		        int ch8 = b[start + 7]; // Least Significant Byte (LSB)
		        

		        return  (((long)(ch8 & 255) << 0) + ((long)(ch7 & 255) << 8) + ((long)(ch6 & 255) << 16) + ((long)(ch5 & 255) << 24) +
		                ((long)(ch4 & 255) << 32) + ((long)(ch3 & 255) << 40) + ((long)(ch2 & 255) <<  48) + ((long)(ch1) <<  56));

	        }

			public Long[] readLongArray(byte[] b, int length, Offset offset) {
				Long[] res = new Long[length];
				for (int i = 0; i < length; ++i) {
					res[i] = new Long(readLong(b, offset));
				}
				return res;
			}

			public void setCpuFrequency(long frequency) {
				this.cpuFrequency = frequency;
			}

			public double readTimestamp(byte[] b, Offset offset) {
				long value = readLong(b, offset);
				
				if (cpuFrequency > 0) {
					value /= cpuFrequency;
				}
				
		    	double high = (value / 1000000000);
		    	double low = (value % 1000000000) * 0.000000001;
		    	double result = high + low;

		    	return result;
			}

			public String readTimestampAsString(byte[] b, Offset offset) {
				long value = readLong(b, offset);
				
		    	String integer = Long.toString(value / 1000000000);
		    	String remainder = Long.toString(Math.abs(value % 1000000000));
		    	// remainder with leading zeros 
		    	remainder = (new StringBuffer()).append("000000000").replace(9 - remainder.length(), 10, remainder).toString();
		    	return integer + "." + remainder;
			}
			
	};
		
	public static final NumberReader LE_64_READER = new NumberReader() { // Little-endian, IA64

			private long cpuFrequency;
			
			public short readShort(byte[] b, Offset offset) {
				int start = offset.getOffset();
		        if (offset.increaseOffset(2) < 0) {
		        	return 0;
		        }
		        int ch1 = b[start];
		        int ch2 = b[start + 1];
		        
		        return (short)((ch2 << 8) + (ch1 << 0));			
			} 
			
			public int readUnsignedShort(byte[] b, Offset offset) {
				int start = offset.getOffset();
		        if (offset.increaseOffset(2) < 0) {
		        	return 0;
		        }
		        int ch1 = b[start] & 0xFF;
		        int ch2 = b[start + 1] & 0xFF;
		        
		        return ((ch2 << 8) + (ch1 << 0));			
			}

			public int readInt(byte[] b, Offset offset) {
				int start = offset.getOffset();
				if (offset.increaseOffset(4) < 0) {
					return 0;
				}
		        int ch1 = b[start];
		        int ch2 = b[start + 1];
		        int ch3 = b[start + 2];
		        int ch4 = b[start + 3];
		        
		        return ((ch4 << 24) + (ch3 << 16) + (ch2 << 8) + (ch1 << 0));
			}
		
			public long readUnsignedInt(byte[] b, Offset offset) {
				int start = offset.getOffset();
				if (offset.increaseOffset(4) < 0) {
					return 0;
				}
		        int ch1 = b[start] & 0xFF;
		        int ch2 = b[start + 1] & 0xFF;
		        int ch3 = b[start + 2] & 0xFF;
		        int ch4 = b[start + 3] & 0xFF;
		        
		        return ((ch4 << 24) + (ch3 << 16) + (ch2 << 8) + (ch1 << 0));
			}

			public long readLong(byte[] b, Offset offset) {
				int start = offset.getOffset();
				if (offset.increaseOffset(8) < 0) {
					return 0;
				}
		        int ch1 = b[start]; // Least Significant Byte (LSB) 
		        int ch2 = b[start + 1];
		        int ch3 = b[start + 2];
		        int ch4 = b[start + 3];
		        int ch5 = b[start + 4];
		        int ch6 = b[start + 5];
		        int ch7 = b[start + 6];
		        int ch8 = b[start + 7]; // Most Significant Byte (MSB)
		        
		        return (((long)ch8 << 56) + ((long)(ch7 & 255) << 48) + ((long)(ch6 & 255) << 40) + ((long)(ch5 & 255) << 32) +
		                ((long)(ch4 & 255) << 24) + ((ch3 & 255) << 16) + ((ch2 & 255) <<  8) + ((ch1 & 255) <<  0));
	        }

			public Long[] readLongArray(byte[] b, int length, Offset offset) {
				Long[] res = new Long[length];
				for (int i = 0; i < length; ++i) {
					res[i] = new Long(readLong(b, offset));
				}
				return res;
			}

			public void setCpuFrequency(long frequency) {
				this.cpuFrequency = frequency;
			}

			public double readTimestamp(byte[] b, Offset offset) {
				long value = readLong(b, offset);
				
				if (cpuFrequency > 0) {
					value /= cpuFrequency;
				}
				
		    	double high = (value / 1000000000);
		    	double low = (value % 1000000000) * 0.000000001;
		    	double result = high + low;

		    	return result;
			}

			public String readTimestampAsString(byte[] b, Offset offset) {
				long value = readLong(b, offset);
				
		    	String integer = Long.toString(value / 1000000000);
		    	String remainder = Long.toString(Math.abs(value % 1000000000));
		    	// remainder with leading zeros 
		    	remainder = (new StringBuffer()).append("000000000").replace(9 - remainder.length(), 10, remainder).toString();
		    	return integer + "." + remainder;
			}
			
		};
	
	private NumberReader numberReader;
	
	private StringReader stringReader;

	public BFReader(NumberReader numberReader, StringReader stringReader) {
		this.numberReader = numberReader;
		this.stringReader = stringReader;
	}
	
	public BFReader() {
		this(LE_64_READER, ASCII_READER);
	}

	public NumberReader getNumberReader() {
		return numberReader;
	}

	public void setNumberReader(NumberReader numberReader) {
		this.numberReader = numberReader;
	}

	public StringReader getStringReader() {
		return stringReader;
	}

	public void setStringReader(StringReader stringReader) {
		this.stringReader = stringReader;
	}

	public String readString(byte[] b, Offset offset) {
		return stringReader.readString(b, offset);
	}
	
	public String[] readStringArray(byte[] b, int length, Offset offset) {
		return stringReader.readStringArray(b, length, offset);
	}

	public byte readByte(byte[] b, Offset offset) {
		int start = offset.getOffset();
		if (offset.increaseOffset(1) < 0) {
			return 0;
		}
		return b[start];
	}

	public short readShort(byte[] b, Offset offset) {
		return numberReader.readShort(b, offset);
	}

	public int readUnsignedShort(byte[] b, Offset offset) {
		return numberReader.readUnsignedShort(b, offset);
	}

	public int readInt(byte[] b, Offset offset) {
		return numberReader.readInt(b, offset);
	}

	public long readUnsignedInt(byte[] b, Offset offset) {
		return numberReader.readUnsignedInt(b, offset);
	}

	public long readLong(byte[] b, Offset offset) {
		return numberReader.readLong(b, offset);
	}

	public Long[] readLongArray(byte[] b, int length, Offset offset) {
		return numberReader.readLongArray(b, length, offset);
	}

	public double readTimestamp(byte[] b, Offset offset) {
		return numberReader.readTimestamp(b, offset);
	}

	public String readTimestampAsString(byte[] b, Offset offset) {
		return numberReader.readTimestampAsString(b, offset);
	}
	
	
	public void setCpuFrequency(long frequency) {
		numberReader.setCpuFrequency(frequency);
	}

	public void setEncoding(String encoding) {
		stringReader.setEncoding(encoding);
	}
}
