/*******************************************************************************
 * Copyright (c) 2008, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Stanislav Polevic, Intel - Initial API and Implementation 
 * 	  
 * $Id: ParserImpl1.java,v 1.7 2010/11/20 19:08:26 jwest Exp $
 *******************************************************************************/
package org.eclipse.hyades.loaders.internal.binary.v1;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.hyades.loaders.internal.binary.BFParser;
import org.eclipse.hyades.loaders.internal.binary.BFParserFactory;
import org.eclipse.hyades.loaders.internal.binary.BFReader;
import org.eclipse.hyades.loaders.internal.binary.BinaryFragment;
import org.eclipse.hyades.loaders.internal.binary.BinaryFragmentParser;
import org.eclipse.hyades.loaders.internal.binary.IgnoredBinaryFragmentParser;
import org.eclipse.hyades.loaders.internal.binary.Offset;
import org.eclipse.hyades.loaders.util.HierarchyContext;

public class ParserImpl1 implements BFParser {
	
	private static final Double version = new Double(1.0);
	private int eventsParsed = 0;
	
	// ID (Short) -> Class (class)
	private Map<Short, Class<?>> messageHandlers = new HashMap<Short, Class<?>>();
	
	private final HierarchyContext context;
	
	private final BFReader reader;
	
	public void reset() {
		eventsParsed = 0;
	}
	
	public ParserImpl1(HierarchyContext context, BFReader reader) {
		this.context = context;
		this.reader = reader;
		initialize();
	}

	private void initialize() {
		// System handlers 0-1000
		messageHandlers.put(BFEncodingParser.ID, BFEncodingParser.class);
		messageHandlers.put(BFFrequencyParser.ID, BFFrequencyParser.class);
		
		// Data handlers 1000-32767
		messageHandlers.put(BFNodeParser.ID, BFNodeParser.class);
		messageHandlers.put(BFProcessCreateParser.ID, BFProcessCreateParser.class);
		messageHandlers.put(BFAgentCreateParser.ID, BFAgentCreateParser.class);
		messageHandlers.put(BFAgentDestroyParser.ID, BFAgentDestroyParser.class);
		messageHandlers.put(BFTraceStartParser.ID, BFTraceStartParser.class);
		messageHandlers.put(BFTraceEndParser.ID, BFTraceEndParser.class);
		messageHandlers.put(BFFilterParser.ID, BFFilterParser.class);
		messageHandlers.put(BFOptionParser.ID, BFOptionParser.class);
		messageHandlers.put(BFThreadStartParser.ID, BFThreadStartParser.class);
		messageHandlers.put(BFThreadEndParser.ID, BFThreadEndParser.class);
		messageHandlers.put(BFClassDefParser.ID, BFClassDefParser.class);
		messageHandlers.put(BFMethodDefParser.ID, BFMethodDefParser.class);
		messageHandlers.put(BFObjAllocParser.ID, BFObjAllocParser.class);
		messageHandlers.put(BFMethodEntryParser.ID, BFMethodEntryParser.class);
		messageHandlers.put(BFMethodExitParser.ID, BFMethodExitParser.class);
		// Method call ignored
		// Method exit ignored
		messageHandlers.put(BFInvocationContextParser.ID, BFInvocationContextParser.class);
		messageHandlers.put(BFObjDefParser.ID, BFObjDefParser.class);
		messageHandlers.put(BFValueParser.ID, BFValueParser.class);
		messageHandlers.put(BFMethodCountParser.ID, BFMethodCountParser.class);
		messageHandlers.put(BFLineParser.ID, BFLineParser.class);
		messageHandlers.put(BFGCStartParser.ID, BFGCStartParser.class);
		messageHandlers.put(BFObjFreeParser.ID, BFObjFreeParser.class);
		messageHandlers.put(BFClassUnloadParser.ID, BFClassUnloadParser.class);
		messageHandlers.put(BFObjMoveParser.ID, BFObjMoveParser.class);
		messageHandlers.put(BFGCFinishParser.ID, BFGCFinishParser.class);
		messageHandlers.put(BFThrowParser.ID, BFThrowParser.class);
		messageHandlers.put(BFCatchParser.ID, BFCatchParser.class);
		messageHandlers.put(BFRuntimeInitDoneParser.ID, BFRuntimeInitDoneParser.class);
		messageHandlers.put(BFRuntimeShutdownParser.ID, BFRuntimeShutdownParser.class);
		messageHandlers.put(BFMonWaitParser.ID, BFMonWaitParser.class);
		messageHandlers.put(BFMonWaitedParser.ID, BFMonWaitedParser.class);
		messageHandlers.put(BFMonContendedEnterParser.ID, BFMonContendedEnterParser.class);
		messageHandlers.put(BFMonContendedEnteredParser.ID, BFMonContendedEnteredParser.class);
		messageHandlers.put(BFAGMethodEntryParser.ID, BFAGMethodEntryParser.class);
		messageHandlers.put(BFAGMethodExitParser.ID, BFAGMethodExitParser.class);
		messageHandlers.put(BFHeapDumpDefParser.ID, BFHeapDumpDefParser.class);
		messageHandlers.put(BFGCRootParser.ID, BFGCRootParser.class);
		messageHandlers.put(BFObjRefParser.ID, BFObjRefParser.class);
		// Insert Custom message parser here
		messageHandlers.put(BFMonNotifyCalledParser.ID, BFMonNotifyCalledParser.class);
		messageHandlers.put(BFThreadInterruptCalledParser.ID, BFThreadInterruptCalledParser.class);
		messageHandlers.put(BFThreadStartCalledParser.ID, BFThreadStartCalledParser.class);
		
		messageHandlers.put(BFThreadSleepStartParser.ID, BFThreadSleepStartParser.class);
		messageHandlers.put(BFThreadSleepEndParser.ID, BFThreadSleepEndParser.class);
	}
	
	public void parseEvents(byte[] buffer, Offset marker) {
		try {
			do {
				BinaryFragment binaryFragment = getBinaryFragment(buffer, marker);
				if (binaryFragment == null) {
					break;
				} else if (binaryFragment.getId() == null) {
					continue;
				}
				parseBinaryFragment(binaryFragment, marker);
			} while (true);
			
		} catch (Throwable e) {
			e.printStackTrace();
		}
	}

	public void parseEvents(InputStream inputStream, Offset marker) {
		try {
			do {
				BinaryFragment binaryFragment = getBinaryFragment(inputStream, marker);
				if (binaryFragment == null) {
					break;
				} else if (binaryFragment.getId() == null) {
					continue;
				}
				parseBinaryFragment(binaryFragment, marker);
			} while (true);
		} catch (Throwable e) {
			e.printStackTrace();
		}
	}

	private BinaryFragment getBinaryFragment(byte[] buffer, Offset marker) throws IOException {
		if (BinaryFragment.isEOF(buffer, marker.getOffset())) {
			return null;
		}
		BinaryFragment binaryFragment = new BinaryFragment(buffer, marker.getOffset(), reader);
		if (marker.increaseOffset(BinaryFragment.HEADER_LENGTH) < 0) {
			return null;
		}
		
		binaryFragment.setBody(buffer);
		binaryFragment.setBodyOffset(marker.getOffset());
		if (marker.increaseOffset(binaryFragment.getLength()) < 0) {
			return null;
		}
		
		if (binaryFragment.getId() == null) {
			//ModelDebugger.log("Unparseable event " + buffer);
		}

		return binaryFragment;
	}
	
	private BinaryFragment getBinaryFragment(InputStream inputStream, Offset marker) throws IOException {
		byte[] header = new byte[BinaryFragment.HEADER_LENGTH];
		if (inputStream.read(header) < 0) {
			return null;
		}
		if (BinaryFragment.isEOF(header, 0)) {
			return null;
		}
		BinaryFragment binaryFragment = new BinaryFragment(header, 0, reader);
		if (marker.increaseOffset(BinaryFragment.HEADER_LENGTH) < 0) {
			return null;
		}

		byte[] body = new byte[binaryFragment.getLength()];
		if(inputStream.read(body) < 0) {
			return null;
		}

		binaryFragment.setBody(body);
		if (marker.increaseOffset(binaryFragment.getLength()) < 0) {
			return null;
		}
		
		if (binaryFragment.getId() == null) {
			//ModelDebugger.log("Unparseable event " + header);
		}

		return binaryFragment;
	}
	
	private void parseBinaryFragment(BinaryFragment binaryFragment, Offset marker)  throws InstantiationException, IllegalAccessException {
		
		BinaryFragmentParser fragmentParser = getFragmentParser(binaryFragment.getId());
		if (fragmentParser.getClass().equals(IgnoredBinaryFragmentParser.class)) {
			//ModelDebugger.log("Got unknown message at " + marker.getOffset());
		}

		fragmentParser.initialize(context);
		
		Offset bodyOffset = new Offset(binaryFragment.getBodyOffset(), binaryFragment.getBodyOffset() + binaryFragment.getLength());
		
		// jgw: Init+parser are definitely reducing performance by about 12-20%, but they are not the 50% of addYinC() below
		
		if (fragmentParser.parse(binaryFragment, bodyOffset, reader)) {
			fragmentParser.addYourselfInContext();
			eventsParsed++;
			
			if ((context.getAgent() != null) && (context.getAgent().eResource() != null) && !context.getAgent().eResource().isModified()) {
				context.getAgent().eResource().setModified(true);
			}
		}
	}

	private BinaryFragmentParser getFragmentParser(Short id) throws InstantiationException, IllegalAccessException {
		Class clazz = (Class) messageHandlers.get(id);

		if (clazz == null) {
			return new IgnoredBinaryFragmentParser();
		}
		BinaryFragmentParser fragmentParser = (BinaryFragmentParser) clazz.newInstance();
		if (fragmentParser == null) {
			return new IgnoredBinaryFragmentParser();
		}
		
		return fragmentParser; 
	}

	public static void register(BFParserFactory parserFactory) {
		parserFactory.put(version, ParserImpl1.class);
	}
	
	public int getNumEventsParsed() {
		return eventsParsed;
	}

}
