/*******************************************************************************
 * Copyright (c) 2008, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Stanislav Polevic, Intel - Initial API and Implementation 
 * 	  
 * $Id: BinaryLoader.java,v 1.4 2010/11/20 19:08:26 jwest Exp $
 *******************************************************************************/
package org.eclipse.hyades.loaders.util;

import java.io.IOException;
import java.io.InputStream;

import org.eclipse.hyades.loaders.internal.binary.BFHeader;
import org.eclipse.hyades.loaders.internal.binary.BFParser;
import org.eclipse.hyades.loaders.internal.binary.BFParserFactory;
import org.eclipse.hyades.loaders.internal.binary.BFReader;
import org.eclipse.hyades.loaders.internal.binary.Offset;
import org.eclipse.hyades.loaders.internal.binary.v1.ParserImpl1;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.util.internal.InvalidEventException;
import org.eclipse.hyades.models.util.ModelDebugger;

public class BinaryLoader extends XMLLoader {

	private final BFParserFactory parserFactory = new BFParserFactory();
	
	private final BFHeader formatHeader = new BFHeader();
	
	private final BFReader reader = new BFReader();
	
	private BFParser internalParser;
	
	private boolean loadXml = true; 

	public BinaryLoader(TRCAgent agent) {
		super(agent);
		init();
	}

	public BinaryLoader(TRCAgentProxy agentProxy) {
		super(agentProxy);
		init();
	}

	public BinaryLoader(TRCMonitor monitor) {
		super(monitor);
		init();
	}
	
	private void init() {
		/** Context and reader can't/won't change, and at the moment
		 * there is only one parser (ParserImpl1), so use of the BFParserFactory is
		 * not required. */
		internalParser = new ParserImpl1(context, reader);
	}

	public boolean isLoadXml() {
		return loadXml;
	}

	public void setLoadXml(boolean loadXml) {
		this.loadXml = loadXml;
	}

	public void loadEvent(byte[] buffer, int offset, int length, boolean loadToModel) throws InvalidXMLException {
		
		if (isLoadXml()) {
			super.loadEvent(buffer, offset, length, loadToModel);
			return;
		}
		
		setLoadToModel(loadToModel);
		if (ModelDebugger.INSTANCE.debugEventsToFile) {
			writeToBinaryOutputFile(buffer, offset, length);
		} else {
			Offset marker = new Offset(offset, length);
			if (formatHeader.parse(buffer, offset, marker)) {
				formatHeader.visit(reader);
			}

			// Every time get is called, a new instance of the parser is allocated, so there
			// is no need to "reset" the num events parsed to 0

			
			// Context and reader will never change (context is only set in the constructor, and reader is a single class) 
			

			BFParser parser = internalParser;
			
			// Old method of obtaining a parser... using the factory:
			//parser = parserFactory.get(formatHeader, context, reader);

			// Reset the parsed events value (and any other state data in the parser)
			parser.reset();

//			if(getContext().getAgent()==null)
//			{
//				LoadersUtils.createAgent(getContext());
//			}
			
			parser.parseEvents(buffer, marker);
			fragmentsCount += parser.getNumEventsParsed();
		}
	}

	public void loadEvents(InputStream inputStream, long offset, long length) throws InvalidEventException {
		if (isLoadXml()) {
			super.loadEvents(inputStream, offset, length);
			return;
		}

		if (offset > 0) {
			try {
				inputStream.skip(offset);
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		Offset marker = new Offset((int)offset, length);

		try {
			if (!formatHeader.parse(inputStream, marker)) {
				return;
			}
			formatHeader.visit(reader);
			
			// Every time get is called, a new instance of the parser is allocated, so there
			// is no need to "reset" the num events parsed to 0 
			BFParser parser = parserFactory.get(formatHeader, context, reader);
			
//			if(getContext().getAgent()==null)
//			{
//				LoadersUtils.createAgent(getContext());
//			}
			parser.parseEvents(inputStream, marker);
			fragmentsCount += parser.getNumEventsParsed();
			
			if ((getContext().getAgent() != null) && (getContext().getAgent().eResource() != null) && !getContext().getAgent().eResource().isModified()) {
				getContext().getAgent().eResource().setModified(true);
			}
		} catch (Exception e) {
			e.printStackTrace();
			throw new InvalidXMLException(e);
		}
	}
}
