/**********************************************************************
 * Copyright (c) 2003, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: GenericLookupService.java,v 1.3 2008/01/24 02:52:33 apnan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.util;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.hyades.models.hierarchy.util.IntToObjectMap;
import org.eclipse.hyades.models.hierarchy.util.LongToObjectMap;
import org.eclipse.hyades.models.util.ModelDebugger;

/**
 * @author slavescu
 */
public abstract class GenericLookupService extends HashMap implements LookupService {
    /**
	 * 
	 */
	protected static final long serialVersionUID = 751323724557818365L;
	//~ Instance fields ----------------------------------------------------------------------------

    protected HashSet types;

    //~ Constructors -------------------------------------------------------------------------------

    public GenericLookupService() {
        super();
        types = new HashSet();
    }

    //~ Methods ------------------------------------------------------------------------------------

    public final Set getSupportedTypes() {
        return types;
    }

    public boolean deregister(HierarchyContext context, Class type, Object key) {
        Map map = getTypeMap(context, type, false);

        if (map == null) {
            return false;
        }

        return map.remove(key) != null;
    }

    public final boolean deregister(HierarchyContext context, Object object) {
        if (object == null) {
            return false;
        }

        return processObject(context, object, true, false);
    }

    public Object locate(HierarchyContext context, Class type, Object key) {
        Map map = getTypeMap(context, type, false);

        if (map == null) {
            return null;
        }

        return map.get(key);
    }

    public Object locate(HierarchyContext context, Object key) {
        Object o = null;

        try {
            Map contextMap = getContextMap(context, false);

            if (contextMap == null) {
                return null;
            }

            for (Iterator iter = contextMap.values().iterator(); iter.hasNext();) {
                Map typeMap = (Map) iter.next();

                o = typeMap.get(key);

                if (o != null) {
                    return o;
                }
            }
        } catch (Exception e) {
            ModelDebugger.log(e);
        }

        return o;
    }

    public final boolean register(HierarchyContext context, Object object) {
        if (object == null) {
            return false;
        }

        return processObject(context, object, false, true);
    }

    public boolean deregister(HierarchyContext context, Class type) {
        Map typeMap = getTypeMap(context, type, false);

        if (typeMap != null) {
            typeMap.clear();

            return true;
        }

        return false;
    }

    public boolean deregister(HierarchyContext context) {
        Map contextMap = getContextMap(context, false);

        if (contextMap != null) {
            contextMap.clear();

            return true;
        }

        return false;
    }

    protected Map getContextMap(HierarchyContext context, boolean createMap) {
        if (context == null) {
            return this;
        }

        Map contextMap = null;

        try {
            contextMap = (Map) get(context.getContextURI());

            if ((contextMap == null) && createMap) {
                contextMap = new HashMap();
                put(context.getContextURI(), contextMap);
            }
        } catch (Exception e) {
        	// do nothing
        }

        return contextMap;
    }

    protected abstract boolean doProcessObject(Map typeMap, Class theClass, Object object, boolean remove);

    protected final void addOrRemove(Map map, Object key, Object object, boolean remove) {
        if (remove) {
            map.remove(key);
        } else {
            map.put(key, object);
        }
    }
    protected final void addOrRemove(IntToObjectMap map, int key, Object object, boolean remove) {
        if (remove) {
            map.remove(key);
        } else {
            map.put(key, object);
        }
    }
    protected final void addOrRemove(LongToObjectMap map, long key, Object object, boolean remove) {
        if (remove) {
            map.remove(key);
        } else {
            map.put(key, object);
        }
    }
    protected Map getTypeMap(HierarchyContext context, Class type, boolean create) {
    	Class equivalentType=null;
        if (!types.contains(type)) {
        	equivalentType = getEquivalentClass(type);
			if(equivalentType == type)
			{
				return null;
			}
			create = true;
			types.add(type);
        }

        Map typeMap = null;

        try {
            Map contextMap = getContextMap(context, create);

            if (contextMap != null) {
				typeMap = (Map) contextMap.get(type);

                if (typeMap == null && create)
                {
					if(equivalentType != null)
					{
						typeMap = (Map) contextMap.get(equivalentType);
						if(typeMap==null)
						{
							typeMap = createMap(equivalentType);
							contextMap.put(equivalentType, typeMap);
						}
						contextMap.put(type, typeMap);
					}
					else
					{
			        	equivalentType = getEquivalentClass(type);
						if(equivalentType != type)
						{
							typeMap = (Map) contextMap.get(equivalentType);
							if(typeMap==null)
							{
								typeMap = createMap(equivalentType);
								contextMap.put(equivalentType, typeMap);
							}
						}
						else
							typeMap = createMap(type);
						contextMap.put(type, typeMap);
					}
                }
            }
        } catch (Exception e) {
            ModelDebugger.log(e);
        }

        return typeMap;
    }

    protected Map createMap(Class equivalentType) {
		return new HashMap();
	}

	/**
	 * @return true if we should use isAssignableFrom for this type (then I will search through all supported types to see if this type can be assignable from one of them)
	 */
	protected Class getEquivalentClass(Class type) {
		return type;
	}

	protected boolean processObject(HierarchyContext context, Object object, boolean remove, boolean createMap) {
		Map typeMap;
		boolean ret = false;
		Class theClass = object.getClass();
    	if(theClass == GenericLookUpEntry.class)
    	{
    		GenericLookUpEntry entry = (GenericLookUpEntry)object;
			typeMap = getTypeMap(context, entry.getValue().getClass(), createMap);
			if (typeMap == null) {
				return false;
			}
			if(typeMap instanceof IntToObjectMap)
				addOrRemove((IntToObjectMap)typeMap, ((Integer)entry.getKey()).intValue(),entry.getValue(), remove);
			else if(typeMap instanceof LongToObjectMap)
				addOrRemove((LongToObjectMap)typeMap, ((Long)entry.getKey()).longValue(),entry.getValue(), remove);
			else
				addOrRemove(typeMap, entry.getKey(),entry.getValue(), remove);
			return true;
    	}
    	else
    	{
			typeMap = getTypeMap(context, theClass, createMap);
			if (typeMap == null) {
				return false;
			}

			try {
				ret = doProcessObject(typeMap,theClass,object, remove);
			} catch (Exception e) {
				ModelDebugger.log(e);
			}
    	}

        return ret;
    }
}
