/**********************************************************************
 * Copyright (c) 2003, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: RegistryReader.java,v 1.4 2008/02/28 03:12:01 jkubasta Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.util;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FilenameFilter;
import java.io.IOException;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.models.hierarchy.util.XMLUtil;
import org.eclipse.hyades.models.util.ModelDebugger;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * @author slavescu
 */
public abstract class RegistryReader {
	// ~ Static fields/initializers
	// -----------------------------------------------------------------

	/**
	 * When this property is set to "false" the workspaceMode will false, even if ResourcePlugin is available
	 */
	public static final String ECLIPSE_TPTP_APPLICATION_WORKSPACE_MODE = "eclipse.tptp.application.workspace.mode";

	public static StandaloneConfiguration standaloneConfiguration;

	protected static int platformMode = 0;

	protected static int workspaceMode = 0;

	protected static int rcpMode = 0;

	protected static final String TAG_DESCRIPTION = "description";

	protected static final String LOADER_PPID = "fragment_loader";

	protected static final String HANDLER_PPID = "fragment_handler";

	protected static final String LOADERS_FACTORY_PPID = "fragment_loaders_factory";

	protected static final String RESOURCE_LOADER_PPID = "resource_loader";

	protected static final String LOOKUP_SERVICE_PPID = "lookup_service";

	protected static final String HYADES_RESOURCE_EXTENSION_PPID = "hyades_resource_extension";

	protected static final String DATABASE_EXTENSIONS_PPID = "database_extensions";

	protected static final String GENERIC_HANDLER_PPID = "handler";
	
	protected String thisPluginId = "org.eclipse.tptp.platform.models";

	protected Map priorities;

	protected static final String ATT_PRIORITY = "priority";

	protected static final String ATT_CLASS = "class";

	// ~ Instance fields
	// ----------------------------------------------------------------------------

	protected List alreadyProcessedEclipseFolders;

	protected String extensionPointID;
	
	protected Properties pluginCustomization; // properties from plugin_customizations.ini file


	// ~ Constructors
	// -------------------------------------------------------------------------------

	public RegistryReader(String extensionPointID) {
		this.extensionPointID = extensionPointID;
	}

	public RegistryReader(String pluginId, String extensionPointID) {
		this.extensionPointID = extensionPointID;
		this.thisPluginId = pluginId;
	}

	public Map getPriorities() {
		if (priorities == null)
			priorities = new HashMap();
		return priorities;
	}

	// ~ Methods
	// ------------------------------------------------------------------------------------

	/**
	 * @return
	 */
	public static synchronized boolean isPlatformMode() {
		if (platformMode == 0) {
			setPlatformMode();
			if (platformMode == 1) {
				setWorkspaceMode();
			} else
				workspaceMode = 2;
		}

		return platformMode == 1;
	}
	
	/**
	 * Overrides the platfrom mode
	 * @param mode
	 */
	public static synchronized void setPlatformMode(int mode){
		platformMode = mode;
	}
	

	/**
	 * @return true if the workspace should be used, can be forced to "false" by using {@link #ECLIPSE_TPTP_APPLICATION_WORKSPACE_MODE} system property
	 * @since 4.2
	 */
	public static synchronized boolean isWorkspaceMode() {
		if (workspaceMode == 0) {
			setPlatformMode();
			if (platformMode == 1) {
				setWorkspaceMode();
			} else
				workspaceMode = 2;
		}
		return workspaceMode == 1;
	}

	/**
	 * 
	 * @return
	 * @deprecated This is not safe to use, for Resource creation use {@link #isWorkspaceMode()}
	 */
	public static synchronized boolean isRCPMode() {
		if (rcpMode == 0) {
			if (isPlatformMode()) {
					// If we're in platform mode, and if the app that we're
					// running
					// is not the eclipse
					// IDE app, then (we assert) we're running an RCP app.
					// Warning: InternalPlatform is the class that populates the
					// PROP_APPLICATION property,
					// so there's no guarantee that the property will continue
					// to be
					// populated (because
					// it's an internal class.) However, there does not appear
					// to be
					// another way to determine
					// if we're running in RCP mode (because, in fact, the IDE
					// is
					// simply an RCP app itself)
					String app = System.getProperty("eclipse.application");
					if (app != null && !app.equalsIgnoreCase("org.eclipse.ui.ide.workbench")) //$NON-NLS-1$
					{
						rcpMode = 1;
					} else {
						// althogh an RCP application could use the
						// ResourcesPlugin, for now
						// if all the other checks failed and in workspace mode
						// we say that is not RCP
						rcpMode = workspaceMode != 1 ? 1 : 2;
					}
			}
			else
				rcpMode = 2;
		}
		return rcpMode == 1;
	}

	protected static void setWorkspaceMode() {
		try {
			String applicationMode = System.getProperty(ECLIPSE_TPTP_APPLICATION_WORKSPACE_MODE);
			if ("false".equalsIgnoreCase(applicationMode)) {
				workspaceMode = 2; // enable file versus workspace resource creation, even if the ResourcesPlugin is available
				return;
			} 

			Class resourcePlugin = Class.forName("org.eclipse.core.resources.ResourcesPlugin");
			Method getWorkspace = resourcePlugin.getMethod("getWorkspace", null);
			Object workspace = getWorkspace.invoke(null, null);
			Method getRoot = workspace.getClass().getMethod("getRoot", null);
			Object root = getRoot.invoke(workspace, null);

			workspaceMode = root != null ? 1 : 2;
		} catch (ClassNotFoundException e) {
			workspaceMode = 2;
		} catch (SecurityException e) {
			workspaceMode = 2;
		} catch (NoSuchMethodException e) {
			workspaceMode = 2;
		} catch (IllegalArgumentException e) {
			workspaceMode = 2;
		} catch (IllegalAccessException e) {
			workspaceMode = 2;
		} catch (InvocationTargetException e) {
			workspaceMode = 2;
		} catch (NullPointerException e) {
			workspaceMode = 2;
		}
	}

	public IHyadesPluginClassDescriptor createPluginClassDescriptor(HyadesConfigurationElement element, String attribute) {
		if (isPlatformMode()) {
			return new PlatformPluginClassDescriptor((IConfigurationElement) element.getElement(), attribute);
		} else {
			return new StandalonePluginClassDescriptor((Node) element.getElement(), attribute);
		}
	}

	/**
	 * Reads from the plugin registry and parses it.
	 */
	public void readRegistry() {
		if (isPlatformMode()) {
			platformReadRegistry();
		} else {
			standaloneReadRegistry();
		}
	}

	protected void logError(HyadesConfigurationElement element, String text) {
		ModelDebugger.log("Plugin " + element.getDeclaringExtensionName() + ", extension " + element.getAttribute("class") + "\n" + text);
	}

	protected void logMissingAttribute(HyadesConfigurationElement element, String attributeName) {
		logError(element, "The required attribute '" + attributeName + "' not defined");
	}

	/**
	 * Implement this method to read element attributes. If this element has
	 * subelements, the reader will recursively cycle through them and will call
	 * this method, so don't do it here.
	 */
	protected abstract boolean readElement(HyadesConfigurationElement element);

	/**
	 * 
	 */
	protected static void setPlatformMode() {
		try {
			// Class resourcePlugin =
			// Class.forName("org.eclipse.core.resources.ResourcesPlugin");
			// Method getWorkspace = resourcePlugin.getMethod("getWorkspace",
			// null);
			// Object workspace = getWorkspace.invoke(null, null);
			// Method getRoot = workspace.getClass().getMethod("getRoot", null);
			// Object root = getRoot.invoke(workspace, null);
			Class platform = Class.forName("org.eclipse.core.runtime.Platform");
			Method isRunning = platform.getMethod("isRunning", null);
			Boolean value = (Boolean) isRunning.invoke(null, null);

			platformMode = value.booleanValue() ? 1 : 2;
		} catch (ClassNotFoundException e) {
			platformMode = 2;
		} catch (NoClassDefFoundError e) {
			platformMode = 2;
		} catch (SecurityException e) {
			platformMode = 2;
		} catch (NoSuchMethodException e) {
			platformMode = 2;
		} catch (IllegalArgumentException e) {
			platformMode = 2;
		} catch (IllegalAccessException e) {
			platformMode = 2;
		} catch (InvocationTargetException e) {
			platformMode = 2;
		} catch (NullPointerException e) {
			platformMode = 2;
		}

		// ResourcesPlugin.getWorkspace().getRoot()
	}

	/**
	 * 
	 */
	protected void addPackageExtensionPoints() {
		for (Iterator iter = standaloneConfiguration.getExtensionsList().iterator(); iter.hasNext();) {
			HyadesConfigurationElement element = (HyadesConfigurationElement) iter.next();

			if (element.getDeclaringExtensionName().equals("org.eclipse.emf.ecore.generated_package")) {
				try {
					String emfPackageClassName = element.getAttribute("class");

					if (standaloneConfiguration.getEmfPackageClassesList().contains(emfPackageClassName)) {
						continue;
					}

					standaloneConfiguration.getEmfPackageClassesList().add(emfPackageClassName);
				} catch (RuntimeException e) {
					ModelDebugger.log(e);
				}
			}
		}
	}

	protected void internalReadElement(HyadesConfigurationElement element) {
		boolean recognized = this.readElement(element);

		if (recognized) {
			HyadesConfigurationElement[] children = element.getChildren();

			for (int i = 0; i < children.length; ++i) {
				internalReadElement(children[i]);
			}
		} else {
			logError(element, "Error processing extension: " + element.getDeclaringExtensionName());
		}
	}

	protected void platformReadRegistry() {
		IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint(thisPluginId, extensionPointID);

		if (point != null) {
			IConfigurationElement[] elements = point.getConfigurationElements();

			for (int i = 0; i < elements.length; i++) {
				internalReadElement(new HyadesConfigurationElement(elements[i]));
			}
		}
	}

	protected void processEMFpackages() {
		addPackageExtensionPoints();

		for (Iterator iter = standaloneConfiguration.getEmfPackageClassesList().iterator(); iter.hasNext();) {
			String element = (String) iter.next();

			try {
				Class clazz = Class.forName(element);

				try {
					Field field = clazz.getField("eINSTANCE");

					field.get(null);
				} catch (NoSuchFieldException e1) {
					Method method = clazz.getMethod("init", null);

					method.invoke(null, null);
				}
			} catch (ClassNotFoundException e) {
				ModelDebugger.log(e);
			} catch (SecurityException e) {
				ModelDebugger.log(e);
			} catch (NoSuchMethodException e) {
				ModelDebugger.log(e);
			} catch (IllegalArgumentException e) {
				ModelDebugger.log(e);
			} catch (IllegalAccessException e) {
				ModelDebugger.log(e);
			} catch (InvocationTargetException e) {
				ModelDebugger.log(e);
			}
		}
	}

	protected void processEclipseFolder() {
		String eclipseFolder = standaloneConfiguration.getEclipseFolder();

		processEclipseFolder(eclipseFolder);

		File linksFolder = new File(eclipseFolder + "/links");

		if (linksFolder.exists()) {
			processLinksFolder(linksFolder);
		}
	}

	/**
	 * 
	 */
	protected void processEclipseFolder(String eclipseFolder) {
		if (eclipseFolder != null) {
			if (alreadyProcessedEclipseFolders.contains(eclipseFolder)) {
				return;
			}

			alreadyProcessedEclipseFolders.add(eclipseFolder);

			File folder = new File(eclipseFolder + "/plugins");

			if (folder.exists()) {
				File[] folders = folder.listFiles();

				for (int i = folders.length - 1; i >= 0; i--) {
					File pluginFile = new File(folders[i].getAbsolutePath() + "/plugin.xml");

					if (pluginFile.exists()) {
						String pluginXMLFileName = pluginFile.getAbsolutePath();

						if (standaloneConfiguration.getPluginsXMLFiles().contains(pluginXMLFileName)) {
							continue;
						}

						standaloneConfiguration.getPluginsXMLFiles().add(pluginXMLFileName);
					}
				}
			}
		}
	}

	protected void processExtensions() {
		for (Iterator iter = standaloneConfiguration.getExtensionsList().iterator(); iter.hasNext();) {
			HyadesConfigurationElement element = (HyadesConfigurationElement) iter.next();

			if (element.getDeclaringExtensionName().endsWith("." + extensionPointID)) {
				internalReadElement(element);
			}
		}
	}

	/**
	 * @param linksFolder
	 */
	protected void processLinksFolder(File linksFolder) {
		File[] files = linksFolder.listFiles(new FilenameFilter() { /*
																	 * (non-Javadoc)
																	 * 
																	 * @see java.io.FilenameFilter#accept(java.io.File,
																	 *      java.lang.String)
																	 */
			public boolean accept(File arg0, String arg1) {
				if (arg1.endsWith(".link")) {
					return true;
				}

				return false;
			}
		});

		for (int i = files.length - 1; i >= 0; i--) {
			try {
				BufferedReader bufferedReader = new BufferedReader(new FileReader(files[i]));
				String line;

				while ((line = bufferedReader.readLine()) != null) {
					if (line.startsWith("path=")) {
						line = line.substring("path=".length()).trim();
						processEclipseFolder(line + "/eclipse");
					}
				}
				bufferedReader.close();
			} catch (FileNotFoundException e) {
				ModelDebugger.log(e);
			} catch (IOException e) {
				ModelDebugger.log(e);
			}
		}
	}

	/**
	 * 
	 */
	protected void processPluginXMLFiles() {
		List alredyProcessed = new ArrayList();

		for (Iterator iter = standaloneConfiguration.getPluginsXMLFiles().iterator(); iter.hasNext();) {
			try {
				String element = (String) iter.next();

				if (alredyProcessed.contains(element)) {
					continue;
				}

				alredyProcessed.add(element);

				Document document = XMLUtil.getXmlDom(element);
				NodeList nodeList = document.getElementsByTagName("extension");

				for (int i = nodeList.getLength() - 1; i >= 0; i--) {
					Node extension = nodeList.item(i);

					standaloneConfiguration.getExtensionsList().add(new HyadesConfigurationElement(extension));
				}
			} catch (RuntimeException e) {
				ModelDebugger.log(e);
			}
		}
	}

	protected synchronized void standaloneReadRegistry() {
		if (standaloneConfiguration != null) {
			if (!standaloneConfiguration.isAlreadyProcessed()) {
				alreadyProcessedEclipseFolders = new ArrayList();
				processEclipseFolder();
				processPluginXMLFiles();
				processEMFpackages(); // initialize EMF packages
				processPluginCustomization();
				standaloneConfiguration.alreadyProcessed = true;
			}

			processExtensions(); // now process the extensions
		}
	}

	protected void processPluginCustomization() {
		if(standaloneConfiguration.getPluginCustomizationLocation()!=null)
		{
			try {
				pluginCustomization = new Properties();
				pluginCustomization.load(new FileInputStream(standaloneConfiguration.getPluginCustomizationLocation()));
			} catch (Exception e) {
				ModelDebugger.log(e, "Error parsing plugin_cusomization.ini file: "+standaloneConfiguration.getPluginCustomizationLocation());
			}
			
		}
		
	}

	// ~ Inner Classes
	// ------------------------------------------------------------------------------

	public static final class StandaloneConfiguration {
		protected List emfPackageClassesList = new ArrayList(); // list of class

		// names for EMF
		// package
		// initialization

		protected List extensionsList = new ArrayList(); // list of

		// HyadesConfigurationElement
		// instances for
		// each extension
		// point (using XML
		// DOM and by
		// following the
		// extension point
		// schema)

		protected List pluginsXMLFiles = new ArrayList(); // list of paths to

		// plugin.xml files

		protected String eclipseFolder; // path to an Eclipse runtime folder -

		// will support links folder also

		protected boolean alreadyProcessed = false;

		protected String pluginCustomizationLocation; // plugin_customizations.Ii file location
		
		public StandaloneConfiguration() {
			// empty method
		}

		/**
		 * @return
		 */
		public boolean isAlreadyProcessed() {
			return alreadyProcessed;
		}

		/**
		 * @param string
		 */
		public void setEclipseFolder(String string) {
			eclipseFolder = string;
		}

		/**
		 * @return
		 */
		public String getEclipseFolder() {
			return eclipseFolder;
		}

		/**
		 * @return
		 */
		public List getEmfPackageClassesList() {
			return emfPackageClassesList;
		}

		/**
		 * @return
		 */
		public List getExtensionsList() {
			return extensionsList;
		}

		/**
		 * @return
		 */
		public List getPluginsXMLFiles() {
			return pluginsXMLFiles;
		}

		public String getPluginCustomizationLocation() {
			return pluginCustomizationLocation;
		}

		public void setPluginCustomizationLocation(String pluginCustomizationLocation) {
			this.pluginCustomizationLocation = pluginCustomizationLocation;
		}
	}

	public String getPluginCustomizationString(String bundleSymbolicId,String key) {
		if(pluginCustomization!=null)
			return pluginCustomization.getProperty(bundleSymbolicId+"/"+key);
		else
			return Platform.getPreferencesService().getString(bundleSymbolicId, key, null,null);
	}

	public String getThisPluginId() {
		return thisPluginId;
	}

	public void setThisPluginId(String thisPluginId) {
		this.thisPluginId = thisPluginId;
	}

	public String getExtensionPointID() {
		return extensionPointID;
	}

	public void setExtensionPointID(String extensionPointID) {
		this.extensionPointID = extensionPointID;
	}
}

final class PlatformPluginClassDescriptor implements IHyadesPluginClassDescriptor {
	// ~ Instance fields
	// ----------------------------------------------------------------------------

	protected IConfigurationElement element;

	protected String attributeName;

	// ~ Constructors
	// -------------------------------------------------------------------------------

	public PlatformPluginClassDescriptor(IConfigurationElement element, String attributeName) {
		this.element = element;
		this.attributeName = attributeName;
	}

	// ~ Methods
	// ------------------------------------------------------------------------------------

	public Object createInstance() {
		try {
			return element.createExecutableExtension(attributeName);
		} catch (Throwable e) {
			ModelDebugger.log(e, "Exception when processing IConfigurationElement name=" + element.getName() + ", attributeName=" + attributeName);
			throw new RuntimeException(e.getLocalizedMessage());
		}
	}
}

/**
 * @author slavescu
 */
class StandalonePluginClassDescriptor implements IHyadesPluginClassDescriptor {
	// ~ Instance fields
	// ----------------------------------------------------------------------------

	protected Node element;

	protected String attributeName;

	// ~ Constructors
	// -------------------------------------------------------------------------------

	public StandalonePluginClassDescriptor(Node element, String attributeName) {
		this.element = element;
		this.attributeName = attributeName;
	}

	// ~ Methods
	// ------------------------------------------------------------------------------------

	public Object createInstance() {
		Object ret = null;
		String className = getFirstRealChild().getAttributes().getNamedItem(attributeName).getNodeValue();

		try {
			Class clazz = Class.forName(className);

			ret = clazz.newInstance();
		} catch (ClassNotFoundException e) {
			ModelDebugger.log(e);
			throw new RuntimeException(e.getLocalizedMessage());
		} catch (InstantiationException e) {
			ModelDebugger.log(e);
			throw new RuntimeException(e.getLocalizedMessage());
		} catch (IllegalAccessException e) {
			ModelDebugger.log(e);
			throw new RuntimeException(e.getLocalizedMessage());
		}

		return ret;
	}

	protected Node getFirstRealChild() {
		Node node = element.getFirstChild();

		while ((node != null) && (node.getAttributes() == null)) {
			node = node.getNextSibling();
		}

		return node;
	}
}
