/**********************************************************************
 * Copyright (c) 2003, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SAXFragmentHandler.java,v 1.2 2008/01/24 02:28:17 apnan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.util;
import java.io.IOException;
import java.io.InputStream;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import org.eclipse.hyades.models.util.ModelDebugger;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;
/**
 * @author slavescu
 */
/**
 * This class wraps an XMLLoader with a SAX DefaultHandler.
 */
public class SAXFragmentHandler implements XMLFragmentHandler {
	protected static final char[] END_ELEMENT = "endElement=".toCharArray();
	protected static final char[] START_ELEMENT = "startElement=".toCharArray();
	protected static final char[] START_DOCUMENT = "startDocument=".toCharArray();
	protected static final char[] END_DOCUMENT = "endDocument=".toCharArray();
	protected static final char[] CHARACTERS = "characters=".toCharArray();
	protected InputSource inputSource = null;
	protected InputStream xmlStream = null;
	protected ParserPipedInputStream pipedXmlStream = null;
	protected SAXParser parser;
	protected Thread parseThread;
	protected IXMLLoader handler;
	protected int depth;
	protected boolean initRequired;
	protected static final int WAIT_TIME = 500;
	protected static final char[] WB_IN = "wB>".toCharArray();
	protected static final char[] WB_OUT = "wB<".toCharArray();
	public void setXMLLoader(IXMLLoader loader) {
		handler = loader;
		initRequired=true;
	}
	public void init() {
		try {
			depth = -1;
			parser = makeParser();
			pipedXmlStream = new ParserPipedInputStream(this);
			xmlStream = pipedXmlStream;
			parseThread = new Thread(Thread.currentThread().getThreadGroup(), "xmlParserThread") {
				public void run() {
					if (xmlStream == null) {
						return;
					}
					try {
						parse(xmlStream, new RegularHandler());
					} catch (Exception e) {
						// ignore error
					}
					xmlStream = null;
				}
			};
			parseThread.start();
		} catch (Exception e) {
			throw new InvalidXMLException(e);
		}
	}
	public void scanContent(byte[] fragment, int offset, int length) throws InvalidXMLException {
		if (pipedXmlStream == null) 
		{
			if(initRequired)
				init();
			else
				return;
		}
		pipedXmlStream.writeBuf(fragment, offset, length);
	}
	public void terminateParser() {
		depth = -1;
		if (xmlStream != null) {
			try {

				if(pipedXmlStream!=null)
				{
					if (ModelDebugger.INSTANCE.debug) {
						ModelDebugger.log("SAXFragmentHandler.terminateParser() - close ParserPipedInputStream");
					}					
					ParserPipedInputStream temp = pipedXmlStream;
					pipedXmlStream = null;
					temp.makeClosed();
				}
				xmlStream = null;
			} catch (Exception e) {
				if (ModelDebugger.INSTANCE.debug) {
					ModelDebugger.log(e, "SAXFragmentHandler.terminateParser()");
				}
				// ignore exception
			}
		}
		synchronized (this) {
			notifyAll();
			if(parseThread!=null)
				parseThread.interrupt();
			if (ModelDebugger.INSTANCE.debug) {
				ModelDebugger.log("SAXFragmentHandler.terminateParser() - done");
			}
		}
	}
	/**
	 * Make either a validating or non-validating parser; throw an if one could
	 * not be made.
	 */
	protected SAXParser makeParser() throws ParserConfigurationException, SAXException {
		return XMLLoader.makeParser();
	}
	protected synchronized void parse(InputStream inputStream, DefaultHandler theHandler) throws InvalidXMLException {
		try {
			inputSource = new InputSource(inputStream);
			parser.parse(inputSource, theHandler);
		} catch (Exception e) {
			InvalidXMLException i;
			if (e instanceof InvalidXMLException)
				i = (InvalidXMLException) e;
			else
				i = new InvalidXMLException(e);
			handler.error(i);
		}
	}
	
	public class RegularHandler extends DefaultHandler
	{

		public void setDocumentLocator(Locator locator) {
		}

		public void characters(char[] ch, int start, int length) throws SAXException {
			handler.characters(ch, start, length);
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(CHARACTERS);
				System.out.println(length + "=" + new String(ch, start, length));
			}
		}

		public void endDocument() throws SAXException {
			if (depth == 0 && xmlStream != null) {
				if(pipedXmlStream !=null)
				{
					pipedXmlStream.makeClosed();
					pipedXmlStream = null;
				}
				xmlStream = null;
			}
			depth = -1;
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(END_DOCUMENT);
				System.out.println();
			}
			handler.endDocument(null, 0);
		}

		public void endElement(String uri, String localName, String qName) throws SAXException {
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(END_ELEMENT);
				System.out.println(qName);
			}
			handler.endElement(qName, 0);
			depth--;
			if (depth == 0) {
				if (xmlStream != null) {
					if(pipedXmlStream !=null)
					{
						pipedXmlStream.makeClosed();
						pipedXmlStream = null;
					}
					xmlStream = null;
				}
			}
		}

		public void endPrefixMapping(String prefix) throws SAXException {
		}

		public void error(SAXParseException e) throws SAXException {
			throw e;
		}

		public void fatalError(SAXParseException e) throws SAXException {
			throw e;
		}

		public void ignorableWhitespace(char[] ch, int start, int length) throws SAXException {
		}

		public void notationDecl(String name, String publicId, String systemId) throws SAXException {
		}

		public void processingInstruction(String target, String data) throws SAXException {
		}

		public InputSource resolveEntity(String publicId, String systemId) throws SAXException {
			return null;
		}

		public void skippedEntity(String name) throws SAXException {
		}

		public void startDocument() throws SAXException {
			depth = 0;
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(START_DOCUMENT);
				System.out.println();
			}
			handler.startDocument();
		}

		public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
			depth++;
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(START_ELEMENT);
				System.out.println(qName);
			}
			handler.startElement(qName, false, attributes.getLength() == 0);
			for (int i = 0; i < attributes.getLength(); i++) {
				handler.attributeName(attributes.getQName(i));
				handler.attributeValueCharacters(attributes.getValue(i));
			}
		}

		public void startPrefixMapping(String prefix, String uri) throws SAXException {
		}

		public void unparsedEntityDecl(String name, String publicId, String systemId, String notationName) throws SAXException {
		}

		public void warning(SAXParseException e) throws SAXException {
		}
		
	}
	
	/**
	 * 
	 * @author slavescu
	 *
	 * Input stream that can be feed from a different thread, see writeBuf method.
	 *
	 */
	public class ParserPipedInputStream extends InputStream {
		protected byte[] inBuf;
		protected boolean closed = false;
		protected int inCount;
		protected long inTotalLength;
		protected int inPos;
//		private SAXFragmentHandler handler;
		protected StringBuffer sb = new StringBuffer();
		/**
		 * Constructor MyPipedInputStream.
		 * 
		 * @param buf
		 */
		public ParserPipedInputStream(SAXFragmentHandler handler) {
//			this.handler = handler;
			inCount = 0;
			inPos = 0;
		}
		/**
		 *  
		 */
		protected synchronized void makeOpened() {
			inBuf = null;
			inCount = 0;
			inPos = 0;
			closed = false;
			notifyAll();
		}
		/**
		 * @see java.io.InputStream#available()
		 */
		public synchronized int available() throws IOException {
			waitForNewData();
			return inCount;
		}
		public void close() throws IOException {
			super.close();
			closed = true;
		}
		public synchronized boolean hasEmptyBuffer() {
			return inCount == 0;
		}
		/**
		 * 
		 * @see java.io.InputStream#mark(int)
		 */
		public synchronized void mark(int arg0) {
			super.mark(arg0);
		}
		/**
		 * @see java.io.InputStream#markSupported()
		 */
		public boolean markSupported() {
			return false;
		}
		/**
		 * @see java.io.InputStream#read()
		 */
		public synchronized int read() throws IOException {
			if (available() == 0) {
				return -1;
			} else {
				inCount--;
				return inBuf[inPos++];
			}
		}
		/**
		 * @see java.io.InputStream#read(byte[], int, int)
		 */
		public synchronized int read(byte[] outBuf, int offset, int length) throws IOException {
			if (available() == 0) {
				return -1;
			}
			int readBytes = Math.min(inCount, length);
			System.arraycopy(inBuf, inPos, outBuf, offset, readBytes);
			inPos = inPos + readBytes;
			inCount = inCount - readBytes;
			return readBytes;
		}
		/**
		 * @see java.io.InputStream#read(byte[])
		 */
		public int read(byte[] arg0) throws IOException {
			return read(arg0, 0, arg0.length);
		}
		public synchronized void reset() throws IOException {
			super.reset();
		}
		/**
		 * @see java.io.InputStream#skip(long)
		 */
		public synchronized long skip(long arg0) throws IOException {
			arg0 = arg0 + inPos;
			if (arg0 < inTotalLength) {
				inPos = (int) arg0;
				return arg0;
			} else {
				return -1;
			}
		}
		/**
		 * Sets the inBuf.
		 * 
		 * @param inBuf
		 *            The inBuf to set
		 */
		public synchronized void writeBuf(byte[] buf, int offset, int length) {
			if ((buf == null) || (length == 0)) {
				return;
			}
			inBuf = buf;
			inPos = offset;
			inTotalLength = inTotalLength + length;
			inCount = length;
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.print(WB_IN);
				sb.setLength(0);
				sb.append(length);
				System.out.println(LoadersUtils.getChars(sb));
				System.out.write(buf, offset, length);
				System.out.write(buf, offset, length);
				System.out.println();
			}
			notifyAll();
			waitForEmptyBuffer();
			if (ModelDebugger.INSTANCE.debugEventFlow) {
				System.out.println(WB_OUT);
			}
		}
		protected synchronized void makeClosed() {
			inBuf = null;
			inCount = 0;
			inPos = 0;
			closed = true;
			notifyAll();
		}
		protected void waitForEmptyBuffer() {
			while (!hasEmptyBuffer() && !closed) {
				try {
					wait(WAIT_TIME);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					if (hasEmptyBuffer() || closed) {
						return;
					}
				}
			}
		}
		protected synchronized void waitForNewData() {
			boolean notify = true;
			while (hasEmptyBuffer()) {
				if (notify) {
					notifyAll();
					notify = false;
				}
				if (closed) {
					return;
				}
				try {
					wait(WAIT_TIME);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					if (closed) {
						return;
					}
				}
			}
		}
	}
	public void scanContent(InputStream inputStream, long offset, long length) {
		try {
			initRequired=false;
			depth = -1;
			parser = makeParser();
			xmlStream = inputStream;//new BufferedInputStream(inputStream,256*1024);
			parse(xmlStream,new RegularHandler());
		} catch (Exception e) {
			InvalidXMLException i;
			if (e instanceof InvalidXMLException)
				i = (InvalidXMLException) e;
			else
				i = new InvalidXMLException(e);
			handler.error(i);
		}
	}
}